package com.biz.crm.worksignrule.service.impl;

import com.baomidou.mybatisplus.core.mapper.BaseMapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.aop.CrmLog;
import com.biz.crm.common.PageResult;
import com.biz.crm.eunm.CrmDelFlagEnum;
import com.biz.crm.eunm.CrmEnableStatusEnum;
import com.biz.crm.nebular.sfa.worksignrule.req.SfaWorkSignSpecialReqVo;
import com.biz.crm.nebular.sfa.worksignrule.resp.SfaWorkSignSpecialRespVo;
import com.biz.crm.util.CrmBeanUtil;
import com.biz.crm.util.StringUtils;
import com.biz.crm.util.ValidateUtils;
import com.biz.crm.worksignrule.mapper.SfaWorkSignSpecialMapper;
import com.biz.crm.worksignrule.model.SfaWorkSignSpecialEntity;
import com.biz.crm.worksignrule.model.SfaWorkSignTimeEntity;
import com.biz.crm.worksignrule.service.ISfaWorkSignSpecialService;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang.Validate;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * 特殊日期 接口实现
 *
 * @author liuhongming
 * @date 2020-09-16 15:06:20
 */
@Slf4j
@Service
@Transactional
@ConditionalOnMissingBean(name="SfaWorkSignSpecialServiceExpandImpl")
public class SfaWorkSignSpecialServiceImpl<M extends BaseMapper<T>, T> extends ServiceImpl<SfaWorkSignSpecialMapper, SfaWorkSignSpecialEntity> implements ISfaWorkSignSpecialService {

    @Resource
    private SfaWorkSignSpecialMapper sfaWorkSignSpecialMapper;

    /**
     * 列表
     * @param reqVo
     * @return
     */
    @Override
    @CrmLog
    public PageResult<SfaWorkSignSpecialRespVo> findList(SfaWorkSignSpecialReqVo reqVo){
        Page<SfaWorkSignSpecialRespVo> page = new Page<>(reqVo.getPageNum(), reqVo.getPageSize());
        List<SfaWorkSignSpecialRespVo> list = sfaWorkSignSpecialMapper.findList(page, reqVo);
        return PageResult.<SfaWorkSignSpecialRespVo>builder()
                .data(list)
                .count(page.getTotal())
                .build();
    }
    /**
     * 根据规则编码查询 Mapping 列表
     * @param ruleCodes
     * @return
     */
    @Override
    public Map<String, List<SfaWorkSignSpecialEntity>> selectMappingByRuleCodes(Set<String> ruleCodes) {
        if(org.springframework.util.CollectionUtils.isEmpty(ruleCodes)){
            return Maps.newHashMap();
        }
        return sfaWorkSignSpecialMapper.selectList(Wrappers.lambdaQuery(SfaWorkSignSpecialEntity.class)
                .in(SfaWorkSignSpecialEntity :: getRuleCode, ruleCodes)).stream().collect(Collectors.groupingBy(SfaWorkSignSpecialEntity :: getRuleCode));
    }
    /**
     * 不分页列表
     * @param reqVo
     * @return
     */
    @Override
    @CrmLog
    public List<SfaWorkSignSpecialRespVo> findNonPageList(SfaWorkSignSpecialReqVo reqVo){
        List<SfaWorkSignSpecialRespVo> list = sfaWorkSignSpecialMapper.findList(reqVo);
        return list;
    }

    /**
     * 查询
     * @param reqVo
     * @return sfaWorkSignSpecialRespVo
     */
    @Override
    @CrmLog
    public SfaWorkSignSpecialRespVo query(SfaWorkSignSpecialReqVo reqVo){
        List<SfaWorkSignSpecialRespVo> list = this.findList(reqVo).getData();
        if(CollectionUtils.isEmpty(list)){
            return new SfaWorkSignSpecialRespVo();
        }
        SfaWorkSignSpecialRespVo sfaWorkSignSpecialRespVo = list.get(0);
        return sfaWorkSignSpecialRespVo;
    }

    /**
     * 新增
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    @CrmLog
    public void save(SfaWorkSignSpecialReqVo reqVo){
        SfaWorkSignSpecialEntity entity = CrmBeanUtil.copy(reqVo,SfaWorkSignSpecialEntity.class);
        this.save(entity);
    }

    /**
     * 更新
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    @CrmLog
    public void update(SfaWorkSignSpecialReqVo reqVo){
        SfaWorkSignSpecialEntity entity = this.getById(reqVo.getId());
        this.updateById(entity);
    }

    /**
     * 删除
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    @CrmLog
    public void deleteBatch(SfaWorkSignSpecialReqVo reqVo){
        List<SfaWorkSignSpecialEntity> sfaWorkSignSpecialEntities = sfaWorkSignSpecialMapper.selectBatchIds(reqVo.getIds());
        if(CollectionUtils.isNotEmpty(sfaWorkSignSpecialEntities)){
            sfaWorkSignSpecialEntities.forEach(o -> {
                 o.setDelFlag(CrmDelFlagEnum.DELETE.getCode());
            });
        }
        this.updateBatchById(sfaWorkSignSpecialEntities);
    }

    /**
     * 启用
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    @CrmLog
    public void enableBatch(SfaWorkSignSpecialReqVo reqVo){
        //设置状态为启用
        List<SfaWorkSignSpecialEntity> sfaWorkSignSpecialEntities = sfaWorkSignSpecialMapper.selectBatchIds(reqVo.getIds());
        if(CollectionUtils.isNotEmpty(sfaWorkSignSpecialEntities)){
            sfaWorkSignSpecialEntities.forEach(o -> {
                o.setEnableStatus(CrmEnableStatusEnum.ENABLE.getCode());
            });
        }
        this.updateBatchById(sfaWorkSignSpecialEntities);
    }

    /**
     * 禁用
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    @CrmLog
    public void disableBatch(SfaWorkSignSpecialReqVo reqVo){
        //设置状态为禁用
        List<SfaWorkSignSpecialEntity> sfaWorkSignSpecialEntities = sfaWorkSignSpecialMapper.selectBatchIds(reqVo.getIds());
        if(CollectionUtils.isNotEmpty(sfaWorkSignSpecialEntities)){
                sfaWorkSignSpecialEntities.forEach(o -> {
                o.setEnableStatus(CrmEnableStatusEnum.DISABLE.getCode());
            });
        }
        this.updateBatchById(sfaWorkSignSpecialEntities);
    }

    /**
     * 根据考勤规则编码和考勤日期查询特殊日期记录
     * @param ruleCode
     * @param wssDate
     * @return
     */
    @Override
    public SfaWorkSignSpecialEntity findByRuleCodeAndWssDate(String ruleCode, String wssDate) {
        if(StringUtils.isEmpty(ruleCode) || StringUtils.isEmpty(wssDate)) {
            return null;
        }
        // TODO ValidateUtils.isTrue(StringUtils.isNotEmpty(ruleCode) && StringUtils.isNotEmpty(wssDate), "根据考勤规则编码和考勤日期查询特殊日期记录时，参数不能为空");
        SfaWorkSignSpecialEntity entity = this.sfaWorkSignSpecialMapper.selectOne(Wrappers.<SfaWorkSignSpecialEntity>lambdaQuery()
        .eq(SfaWorkSignSpecialEntity::getRuleCode, ruleCode)
        .eq(SfaWorkSignSpecialEntity::getWssDate, wssDate));
        return entity;
    }
}
