package com.biz.crm.worksummary.service.impl;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.mapper.BaseMapper;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.base.ApiResultUtil;
import com.biz.crm.base.BusinessException;
import com.biz.crm.common.PageResult;
import com.biz.crm.eunm.CrmDelFlagEnum;
import com.biz.crm.eunm.CrmEnableStatusEnum;
import com.biz.crm.eunm.sfa.SfaWorkSummaryEnum;
import com.biz.crm.mdm.position.MdmPositionFeign;
import com.biz.crm.nebular.mdm.position.req.MdmPositionUserOrgReqVo;
import com.biz.crm.nebular.mdm.position.resp.MdmPositionUserOrgRespVo;
import com.biz.crm.nebular.sfa.worksummary.req.SfaWorkSummaryCommentReqVo;
import com.biz.crm.nebular.sfa.worksummary.resp.SfaWorkSummaryAtLogRespVo;
import com.biz.crm.nebular.sfa.worksummary.resp.SfaWorkSummaryCommentRespVo;
import com.biz.crm.nebular.sfa.worksummary.resp.SfaWorkSummaryLikeRespVo;
import com.biz.crm.util.*;
import com.biz.crm.worksummary.mapper.*;
import com.biz.crm.worksummary.model.*;
import com.biz.crm.worksummary.service.ISfaWorkSummaryCommentService;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;
import org.springframework.util.ObjectUtils;

import javax.annotation.Resource;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * 工作总结评论记录接口实现
 *
 * @author Gavin
 * @date 2021-03-06 11:00:30
 */
@Slf4j
@Service
@ConditionalOnMissingBean(name = "SfaWorkSummaryCommentServiceExpandImpl")
public class SfaWorkSummaryCommentServiceImpl<M extends BaseMapper<T>, T> extends ServiceImpl<SfaWorkSummaryCommentMapper, SfaWorkSummaryCommentEntity> implements ISfaWorkSummaryCommentService {

    @Resource
    private SfaWorkSummaryCommentMapper sfaWorkSummaryCommentMapper;

    @Autowired
    private SfaWorkSummaryAtLogMapper sfaWorkSummaryAtLogMapper;

    @Autowired
    private SfaWorkSummaryScopeMapper sfaWorkSummaryScopeMapper;

    @Autowired
    private SfaWorkSummaryLikeMapper sfaWorkSummaryLikeMapper;

    @Autowired
    private SfaWorkSummaryPrimaryMapper sfaWorkSummaryPrimaryMapper;

    @Autowired
    private MdmPositionFeign mdmPositionFeign;

    /**
     * 列表
     *
     * @param reqVo
     * @return
     */
    @Override
    public PageResult<SfaWorkSummaryCommentRespVo> findList(SfaWorkSummaryCommentReqVo reqVo) {
        //当前登录人
        UserRedis currUser = UserUtils.getUser();
        Page<SfaWorkSummaryCommentRespVo> page = PageUtil.buildPage(reqVo.getPageNum(), reqVo.getPageSize());
        List<SfaWorkSummaryCommentRespVo> list = sfaWorkSummaryCommentMapper.findList(page, reqVo);
        //数据处理
        if (!CollectionUtils.isEmpty(list)) {
            //评论人
            Set<String> createCodes = Sets.newHashSet();
            //数据ID
            List<String> ids = Lists.newArrayList();
            list.forEach(o -> {
                createCodes.add(o.getCreateCode());
                ids.add(o.getId());
            });
            //查询@人信息
            List<SfaWorkSummaryAtLogRespVo> atLogRespVos = sfaWorkSummaryAtLogMapper.findListByBusinessIds(ids);
            Map<String, List<SfaWorkSummaryAtLogRespVo>> atLogRespVoMap = atLogRespVos.stream()
                    .collect(Collectors.groupingBy(SfaWorkSummaryAtLogRespVo::getBusinessId));
            atLogRespVoMap.forEach((k, v) -> {
                v.forEach(o -> {
                    createCodes.add(o.getAtUserCode());
                });
            });
            //查询人员信息
            MdmPositionUserOrgReqVo userQueryVo = new MdmPositionUserOrgReqVo();
            userQueryVo.setPrimaryFlag("1");
            userQueryVo.setUserNameList(createCodes.stream().collect(Collectors.toList()));
            List<MdmPositionUserOrgRespVo> userOrgRespVos = ApiResultUtil.objResult(mdmPositionFeign.findPositionUserOrgList(userQueryVo));
            Map<String, MdmPositionUserOrgRespVo> userOrgRespVoMap = userOrgRespVos.stream()
                    .collect(Collectors.toMap(MdmPositionUserOrgRespVo::getUserName, Function.identity()));

            //查询点赞数
            List<SfaWorkSummaryLikeRespVo> likeRespVos = sfaWorkSummaryLikeMapper.findCountByBusinessIds(ids);
            Map<String, SfaWorkSummaryLikeRespVo> likeRespVoMap = likeRespVos.stream()
                    .collect(Collectors.toMap(SfaWorkSummaryLikeRespVo::getBusinessId, Function.identity()));
            //查询自己点赞记录
            LambdaQueryWrapper<SfaWorkSummaryLikeEntity> buildLikeQuery = new LambdaQueryWrapper<SfaWorkSummaryLikeEntity>()
                    .eq(SfaWorkSummaryLikeEntity::getCreateCode, currUser.getUsername())
                    .in(SfaWorkSummaryLikeEntity::getBusinessId, ids);
            List<SfaWorkSummaryLikeEntity> summaryLikeEntities = sfaWorkSummaryLikeMapper.selectList(buildLikeQuery);
            Map<String, SfaWorkSummaryLikeEntity> likedMap = summaryLikeEntities.stream()
                    .collect(Collectors.toMap(SfaWorkSummaryLikeEntity::getBusinessId, Function.identity()));

            //查询评论数
            List<SfaWorkSummaryCommentRespVo> commentRespVos = sfaWorkSummaryCommentMapper.findCountByBusinessIds(ids);
            Map<String, SfaWorkSummaryCommentRespVo> commentRespVoMap = commentRespVos.stream()
                    .collect(Collectors.toMap(SfaWorkSummaryCommentRespVo::getBusinessId, Function.identity()));
            list.forEach(o -> {
                o.setComment(o.getCommentDesc());
                String businessId = o.getId();
                String id = o.getId();
                //头像
                MdmPositionUserOrgRespVo userOrgRespVo = userOrgRespVoMap.get(o.getCreateCode());
                o.setPhoto(userOrgRespVo.getUserHeadUrl());

                //点赞数
                SfaWorkSummaryLikeRespVo likeRespVo = likeRespVoMap.get(businessId);
                if (!ObjectUtils.isEmpty(likeRespVo)) {
                    o.setLikeSumCount(likeRespVo.getSumCount());
                }
                //评论数
                SfaWorkSummaryCommentRespVo sfaWorkSummaryCommentRespVo = commentRespVoMap.get(businessId);
                if (!ObjectUtils.isEmpty(sfaWorkSummaryCommentRespVo)) {
                    o.setCommentSumCount(sfaWorkSummaryCommentRespVo.getCommentSumCount());
                }

                //@人员
                List<SfaWorkSummaryAtLogRespVo> atLogArray = atLogRespVoMap.get(o.getId());
                if (!CollectionUtils.isEmpty(atLogArray)) {
                    atLogArray.forEach(e -> {
                        MdmPositionUserOrgRespVo userRespVo = userOrgRespVoMap.get(e.getAtUserCode());
                        if (!ObjectUtils.isEmpty(userRespVo)) {
                            e.setAtUserName(userRespVo.getFullName());
                        }
                    });
                    o.setAtLogRespVos(atLogArray);
                }
                //是否已点赞
                SfaWorkSummaryLikeEntity summaryLikeEntity = likedMap.get(id);
                if (!ObjectUtils.isEmpty(summaryLikeEntity)) {
                    o.setLiked("Y");
                } else {
                    o.setLiked("N");
                }
            });
        }
        return PageResult.<SfaWorkSummaryCommentRespVo>builder()
                .data(list)
                .count(page.getTotal())
                .build();
    }

    /**
     * 查询
     *
     * @param reqVo
     * @return sfaWorkSummaryCommentRespVo
     */
    @Override
    public SfaWorkSummaryCommentRespVo query(SfaWorkSummaryCommentReqVo reqVo) {
        return null;
    }

    /**
     * 新增
     *
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void save(SfaWorkSummaryCommentReqVo reqVo) {
        ValidateUtils.validate(reqVo.getBusinessId(), "待评论数据ID不能为空");
        //ValidateUtils.validate(reqVo.getSummaryId(),"工作总结ID不能为空");
        UserRedis user = UserUtils.getUser();
        //查询总结
        SfaWorkSummaryPrimaryEntity summaryPrimaryEntity = sfaWorkSummaryPrimaryMapper.selectById(reqVo.getBusinessId());
        if (!ObjectUtils.isEmpty(summaryPrimaryEntity)) {
            reqVo.setSummaryId(summaryPrimaryEntity.getId());
        } else {
            //查询评论
            SfaWorkSummaryCommentEntity summaryCommentEntity = sfaWorkSummaryCommentMapper.selectById(reqVo.getBusinessId());
            if (!ObjectUtils.isEmpty(summaryCommentEntity)) {
                reqVo.setSummaryId(summaryCommentEntity.getSummaryId());
            } else {
                throw new BusinessException("businessId数据异常");
            }
        }

        SfaWorkSummaryCommentEntity entity = CrmBeanUtil.copy(reqVo, SfaWorkSummaryCommentEntity.class);
        entity.setCommentDesc(reqVo.getComment());
        this.save(entity);
        String summaryId = reqVo.getSummaryId();
        String businessId = reqVo.getBusinessId();
        //保存@对象
        if (!CollectionUtils.isEmpty(reqVo.getAtColleagueCode())) {
            LambdaQueryWrapper<SfaWorkSummaryScopeEntity> build = new LambdaQueryWrapper<SfaWorkSummaryScopeEntity>()
                    .eq(SfaWorkSummaryScopeEntity::getBusinessId, businessId)
                    .in(SfaWorkSummaryScopeEntity::getScopeCode, reqVo.getAtColleagueCode())
                    .eq(SfaWorkSummaryScopeEntity::getScopeType, SfaWorkSummaryEnum.ScopeType.COLLEAGUE.getVal());
            List<SfaWorkSummaryScopeEntity> scopeEntities = sfaWorkSummaryScopeMapper.selectList(build);
            Map<String, SfaWorkSummaryScopeEntity> scopeEntityMap = scopeEntities.stream()
                    .collect(Collectors.toMap(SfaWorkSummaryScopeEntity::getScopeCode, Function.identity()));
            reqVo.getAtColleagueCode().forEach(o -> {
                SfaWorkSummaryAtLogEntity atLogEntity = new SfaWorkSummaryAtLogEntity();
                atLogEntity.setAtUserCode(o);
                atLogEntity.setSummaryId(summaryId);
                atLogEntity.setBusinessId(businessId);
                sfaWorkSummaryAtLogMapper.insert(atLogEntity);
                //@人员也具有查看动态权限所以把@人员也保存进范围信息中
                SfaWorkSummaryScopeEntity scopeEntity = scopeEntityMap.get(o);
                if (!ObjectUtils.isEmpty(scopeEntity)) {
                    scopeEntity.setScopeCount(scopeEntity.getScopeCount() + 1);
                    sfaWorkSummaryScopeMapper.updateById(scopeEntity);
                } else {
                    scopeEntity = new SfaWorkSummaryScopeEntity();
                    scopeEntity.setBusinessId(businessId);
                    scopeEntity.setScopeCode(o);
                    scopeEntity.setScopeCount(1);
                    scopeEntity.setScopeType(SfaWorkSummaryEnum.ScopeType.COLLEAGUE.getVal());
                    sfaWorkSummaryScopeMapper.insert(scopeEntity);
                }
            });
        }
    }

    /**
     * 更新
     *
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void update(SfaWorkSummaryCommentReqVo reqVo) {
        SfaWorkSummaryCommentEntity entity = this.getById(reqVo.getId());
        this.updateById(entity);
    }

    /**
     * 删除
     *
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteBatch(SfaWorkSummaryCommentReqVo reqVo) {
        List<SfaWorkSummaryCommentEntity> sfaWorkSummaryCommentEntities = sfaWorkSummaryCommentMapper.selectBatchIds(reqVo.getIds());
        if (!CollectionUtils.isEmpty(sfaWorkSummaryCommentEntities)) {
            sfaWorkSummaryCommentEntities.forEach(o -> {
                o.setDelFlag(CrmDelFlagEnum.DELETE.getCode());
            });
        }
        this.updateBatchById(sfaWorkSummaryCommentEntities);
    }

    /**
     * 启用
     *
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void enableBatch(SfaWorkSummaryCommentReqVo reqVo) {
        //设置状态为启用
        List<SfaWorkSummaryCommentEntity> sfaWorkSummaryCommentEntities = sfaWorkSummaryCommentMapper.selectBatchIds(reqVo.getIds());
        if (!CollectionUtils.isEmpty(sfaWorkSummaryCommentEntities)) {
            sfaWorkSummaryCommentEntities.forEach(o -> {
                o.setEnableStatus(CrmEnableStatusEnum.ENABLE.getCode());
            });
        }
        this.updateBatchById(sfaWorkSummaryCommentEntities);
    }

    /**
     * 禁用
     *
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void disableBatch(SfaWorkSummaryCommentReqVo reqVo) {
        //设置状态为禁用
        List<SfaWorkSummaryCommentEntity> sfaWorkSummaryCommentEntities = sfaWorkSummaryCommentMapper.selectBatchIds(reqVo.getIds());
        if (!CollectionUtils.isEmpty(sfaWorkSummaryCommentEntities)) {
            sfaWorkSummaryCommentEntities.forEach(o -> {
                o.setEnableStatus(CrmEnableStatusEnum.DISABLE.getCode());
            });
        }
        this.updateBatchById(sfaWorkSummaryCommentEntities);
    }
}
