package com.biz.crm.account.service.impl;

import com.alibaba.fastjson.JSON;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.mapper.BaseMapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.account.mapper.TpmAccountDetailMapper;
import com.biz.crm.account.mapper.TpmAccountFileMapper;
import com.biz.crm.account.mapper.TpmAccountInvoiceMapper;
import com.biz.crm.account.mapper.TpmAccountProductMapper;
import com.biz.crm.account.model.TpmAccountDetailEntity;
import com.biz.crm.account.model.TpmAccountFileEntity;
import com.biz.crm.account.model.TpmAccountInvoiceEntity;
import com.biz.crm.account.model.TpmAccountProductEntity;
import com.biz.crm.account.service.ITpmAccountDetailService;
import com.biz.crm.account.service.ITpmAccountFileService;
import com.biz.crm.account.service.ITpmAccountInvoiceService;
import com.biz.crm.account.service.ITpmAccountProductService;
import com.biz.crm.annotation.Klock;
import com.biz.crm.audit.mapper.TpmAuditDetailMapper;
import com.biz.crm.audit.model.TpmAuditDetailCustomerEntity;
import com.biz.crm.audit.model.TpmAuditDetailEntity;
import com.biz.crm.audit.service.ITpmAuditDetailCustomerService;
import com.biz.crm.audit.service.ITpmAuditDetailService;
import com.biz.crm.audit.service.ITpmAuditService;
import com.biz.crm.base.ApiResultUtil;
import com.biz.crm.base.BusinessException;
import com.biz.crm.base.config.ThreadLocalUtil;
import com.biz.crm.common.DictItemVo;
import com.biz.crm.common.GlobalParam;
import com.biz.crm.common.PageResult;
import com.biz.crm.common.TpmGlobalDictConstants;
import com.biz.crm.costtypefine.mapper.TpmCostTypeFineMapper;
import com.biz.crm.costtypefine.model.TpmCostTypeFineEntity;
import com.biz.crm.crmlog.handle.util.CrmLogSendUtil;
import com.biz.crm.eunm.CrmEnableStatusEnum;
import com.biz.crm.eunm.GlobalWhetherEnum;
import com.biz.crm.eunm.fee.FeePoolOperationTypeEnum;
import com.biz.crm.eunm.fee.FeePoolTypeEnum;
import com.biz.crm.eunm.tpm.AccountStatusEnum;
import com.biz.crm.eunm.tpm.ActPayTypeEnum;
import com.biz.crm.eunm.tpm.ActPayTypeTypeEnum;
import com.biz.crm.exception.tpm.AccountException;
import com.biz.crm.fee.pool.FeePoolFeign;
import com.biz.crm.liqueraudit.mapper.TpmLiqueurAuditAttachMapper;
import com.biz.crm.liqueraudit.model.TpmLiqueurAuditAttachEntity;
import com.biz.crm.liqueraudit.service.ITpmLiqueurAuditAttachService;
import com.biz.crm.nebular.fee.pool.req.FeePoolAccountReqVo;
import com.biz.crm.nebular.tpm.account.TpmAccountFileVo;
import com.biz.crm.nebular.tpm.account.req.*;
import com.biz.crm.nebular.tpm.account.resp.TpmAccountDetailMainRespVo;
import com.biz.crm.nebular.tpm.account.resp.TpmAccountDetailRespVo;
import com.biz.crm.nebular.tpm.account.resp.TpmAccountInvoiceRespVo;
import com.biz.crm.nebular.tpm.account.resp.TpmAccountProductRespVo;
import com.biz.crm.util.*;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.collections4.MapUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.math.BigDecimal;
import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * 费用上账表;接口实现
 *
 * @author zeyi
 * @date 2020-10-12 17:26:54
 */
@Slf4j
@Service
@ConditionalOnMissingBean(name = "TpmAccountDetailServiceExpandImpl")
public class TpmAccountDetailServiceImpl<M extends BaseMapper<T>, T> extends ServiceImpl<TpmAccountDetailMapper, TpmAccountDetailEntity> implements ITpmAccountDetailService {

    @Resource
    private TpmAccountDetailMapper tpmAccountDetailMapper;

    @Autowired
    private AccountServiceHelper accountServiceHelper;

    @Resource
    private TpmAccountProductMapper tpmAccountProductMapper;

    @Resource
    private TpmAccountInvoiceMapper tpmAccountInvoiceMapper;

    @Resource
    private TpmCostTypeFineMapper tpmCostTypeFineMapper;

    @Autowired
    private ITpmAuditService iTpmAuditService;

    @Autowired
    private ITpmAccountProductService accountProductService;

    @Autowired
    private ITpmAccountInvoiceService accountInvoiceService;

    @Resource
    private TpmAuditDetailMapper auditDetailMapper;

    @Resource
    private ITpmAuditDetailCustomerService auditDetailCustomerService;

    @Autowired
    private ITpmAuditDetailService auditDetailService;

    @Resource
    private TpmLiqueurAuditAttachMapper liqueurAuditAttachMapper;

    @Autowired
    private ITpmLiqueurAuditAttachService liqueurAuditAttachService;

    @Autowired
    private CrmLogSendUtil crmLogSendUtil;

    @Autowired
    private FeePoolFeign feePoolFeign;

    @Resource
    private TpmAccountFileMapper accountFileMapper;

    @Autowired
    private ITpmAccountFileService fileService;

    /**
     * 列表
     *
     * @param reqVo
     * @return
     */
    @Override
    public PageResult<TpmAccountDetailRespVo> findList(TpmAccountDetailReqVo reqVo) {
        Page<TpmAccountDetailRespVo> page = PageUtil.buildPage(reqVo.getPageNum(), reqVo.getPageSize());
        reqVo.setSelectedCodeList(accountServiceHelper.dealSelectedCodeList(reqVo.getSelectedCode(),reqVo.getSelectedCodeList()));
        List<TpmAccountDetailRespVo> list = tpmAccountDetailMapper.findList(page, reqVo);
        accountServiceHelper.convertListDate(list);
        return PageResult.<TpmAccountDetailRespVo>builder()
                .data(list)
                .count(page.getTotal())
                .build();
    }

    /**
     * 查看
     *
     * @param id
     * @return tpmAccountDetailRespVo
     */
    @Override
    public TpmAccountDetailMainRespVo queryView(String id) {

        TpmAccountDetailMainRespVo resultVo = new TpmAccountDetailMainRespVo();
        AssertUtils.isNotEmpty(id, "ID不能为空");
        TpmAccountDetailEntity accountDetailEntity = tpmAccountDetailMapper.selectById(id);
        AssertUtils.isNotNull(accountDetailEntity, AccountException.DATA_NOT_EXIST);
        TpmAccountDetailReqVo reqVo = new TpmAccountDetailReqVo();
        resultVo.setGroupId(accountDetailEntity.getGroupId());
        reqVo.setGroupId(accountDetailEntity.getGroupId());
        reqVo.setPageSize(-1);
        //查询附件
        Map<String, List<TpmAccountFileVo>> fileMap = accountFileMapper.selectList(Wrappers.lambdaQuery(TpmAccountFileEntity.class).eq(TpmAccountFileEntity::getGroupId, accountDetailEntity.getGroupId())).stream().map(o->{
           return CrmBeanUtil.copy(o, TpmAccountFileVo.class);
        }).collect(Collectors.groupingBy(TpmAccountFileVo::getAccountCode));
        List<TpmAccountDetailRespVo> detailRespVos = this.findList(reqVo).getData();
        for (TpmAccountDetailRespVo vo : detailRespVos) {
            if(MapUtils.isNotEmpty(fileMap)){
                List<TpmAccountFileVo> tpmAccountFileVos = fileMap.get(vo.getAccountCode());
                vo.setActivityFileList(tpmAccountFileVos);
            }
        }
        //计算本次可上账金额
        this.setCurrentAllowAmount(detailRespVos);
        resultVo.setAccountDetailVos(detailRespVos);
        //设置产品数据
        this.setProductInfo(resultVo);
        //设置发票数据
        this.setInvoiceInfo(resultVo);
        return resultVo;
    }

    /**
     * 设置发票信息
     *
     * @param resultVo
     */
    private void setInvoiceInfo(TpmAccountDetailMainRespVo resultVo) {
        List<TpmAccountInvoiceEntity> tpmAccountInvoiceEntities = tpmAccountInvoiceMapper.selectList(Wrappers.<TpmAccountInvoiceEntity>lambdaQuery()
                .eq(TpmAccountInvoiceEntity::getGroupId, resultVo.getAccountDetailVos().get(0).getGroupId()));
        List<TpmAccountInvoiceRespVo> invoiceRespVos = Lists.newArrayList();
        if (CollectionUtils.isNotEmpty(tpmAccountInvoiceEntities)) {
            invoiceRespVos = tpmAccountInvoiceEntities.stream().map(o -> {
                TpmAccountInvoiceRespVo vo = CrmBeanUtil.copy(o, TpmAccountInvoiceRespVo.class);
                vo.setAccountCodeList(JSON.parseArray(o.getAccountCodes(), String.class));
                vo.setAccountCodes(StringUtils.join(vo.getAccountCodeList(), ","));
                return vo;
            }).collect(Collectors.toList());
        }
        resultVo.setAccountInvoiceVos(invoiceRespVos);
    }

    /**
     * 设置产品信息
     *
     * @param resultVo
     */
    private void setProductInfo(TpmAccountDetailMainRespVo resultVo) {
        //设置产品信息
        List<TpmAccountProductEntity> productEntities = tpmAccountProductMapper.selectList(Wrappers.<TpmAccountProductEntity>lambdaQuery()
                .eq(TpmAccountProductEntity::getGroupId, resultVo.getGroupId()));
        if (CollectionUtils.isNotEmpty(productEntities)) {
            Map<String, List<TpmAccountProductEntity>> accountProductMap = productEntities.stream()
                    .collect(Collectors.groupingBy(TpmAccountProductEntity::getAccountCode));
            for (TpmAccountDetailRespVo detailRespVo : resultVo.getAccountDetailVos()) {
                List<TpmAccountProductEntity> entities = accountProductMap.get(detailRespVo.getAccountCode());
                List<TpmAccountProductRespVo> tpmAccountProductRespVos = CrmBeanUtil.copyList(entities, TpmAccountProductRespVo.class);
                detailRespVo.setReplenishmentProductList(Optional.ofNullable(tpmAccountProductRespVos).orElse(Lists.newArrayList()));
                List<String> list = Optional.ofNullable(detailRespVo.getAccountProductRespVos()).orElse(Lists.newArrayList());
                tpmAccountProductRespVos.stream().forEach(o -> {
                    list.add(o.getProductCode());
                });
                detailRespVo.setAccountProductRespVos(list);
            }
        }
    }

    /**
     * 计算本次可上账金额
     *
     * @param detailRespVos
     */
    private void setCurrentAllowAmount(List<TpmAccountDetailRespVo> detailRespVos) {
        if (CollectionUtils.isNotEmpty(detailRespVos)) {
            //酒类
            if (StringUtils.isNotBlank(detailRespVos.get(0).getAuditDetailAttachCode())) {
                Set<String> auditDetailAttachCodes = detailRespVos.stream().map(TpmAccountDetailRespVo::getAuditDetailAttachCode).collect(Collectors.toSet());
                List<TpmLiqueurAuditAttachEntity> tpmLiqueurAuditAttachEntities = liqueurAuditAttachMapper.selectList(Wrappers.lambdaQuery(TpmLiqueurAuditAttachEntity.class).select(TpmLiqueurAuditAttachEntity::getAuditDetailAttachCode, TpmLiqueurAuditAttachEntity::getAuditAmount, TpmLiqueurAuditAttachEntity::getFeeUsed).in(TpmLiqueurAuditAttachEntity::getAuditDetailAttachCode, auditDetailAttachCodes));
                Map<String, TpmLiqueurAuditAttachEntity> auditAttachEntityMap = Optional.ofNullable(tpmLiqueurAuditAttachEntities).orElse(Lists.newArrayList()).stream().collect(Collectors.toMap(TpmLiqueurAuditAttachEntity::getAuditDetailAttachCode, Function.identity()));
                if (MapUtils.isNotEmpty(auditAttachEntityMap)) {
                    detailRespVos.stream().forEach(o -> {
                        TpmLiqueurAuditAttachEntity auditAttachEntity = auditAttachEntityMap.get(o.getAuditDetailAttachCode());
                        if (Objects.nonNull(auditAttachEntity)) {
                            o.setCurrentAllowAmount(auditAttachEntity.getAuditAmount().subtract(Optional.ofNullable(auditAttachEntity.getFeeUsed()).orElse(BigDecimal.ZERO)));
                        } else {
                            o.setCurrentAllowAmount(BigDecimal.ZERO);
                        }
                    });
                }
                //标品
            } else {
                Set<String> collect = detailRespVos.stream().map(TpmAccountDetailRespVo::getAuditDetailCode).collect(Collectors.toSet());
                List<TpmAuditDetailEntity> tpmAuditDetailEntities = auditDetailMapper.selectList(Wrappers.lambdaQuery(TpmAuditDetailEntity.class).select(TpmAuditDetailEntity::getAuditDetailCode, TpmAuditDetailEntity::getAuditApplyAmount, TpmAuditDetailEntity::getFeeUsed).in(TpmAuditDetailEntity::getAuditDetailCode, collect));
                Map<String, TpmAuditDetailEntity> auditDetailEntityMap = Optional.ofNullable(tpmAuditDetailEntities).orElse(Lists.newArrayList()).stream().collect(Collectors.toMap(TpmAuditDetailEntity::getAuditDetailCode, Function.identity()));
                if (MapUtils.isNotEmpty(auditDetailEntityMap)) {
                    detailRespVos.stream().forEach(o -> {
                        TpmAuditDetailEntity auditDetailEntity = auditDetailEntityMap.get(o.getAuditDetailCode());
                        if (Objects.nonNull(auditDetailEntity)) {
                            o.setCurrentAllowAmount(auditDetailEntity.getAuditApplyAmount().subtract(Optional.ofNullable(auditDetailEntity.getFeeUsed()).orElse(BigDecimal.ZERO)));
                        } else {
                            o.setCurrentAllowAmount(BigDecimal.ZERO);
                        }
                    });
                }
            }
        }
    }

    /**
     * 新增
     *
     * @param accountDetailMainReqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    @Klock(keys = {"accountDetail", "#accountDetailMainReqVo.lockUserName"}, waitTime = 0, leaseTime = 5)
    public void save(TpmAccountDetailMainReqVo accountDetailMainReqVo) {
        accountServiceHelper.saveCheck(accountDetailMainReqVo);
        List<TpmAccountDetailReqVo> accountDetailReqVos = accountDetailMainReqVo.getAccountDetailVos();
        //分组id(标识同一次提交的数据)
        String groupId = UUIDGenerator.generate();
        //设置货补产品名称
        List<TpmAccountFileVo> fileVos=Lists.newArrayList();
        accountDetailReqVos.forEach(tpmAccountDetailReqVo -> {
            List<TpmAccountProductReqVo> tpmProductInfoReqVos = tpmAccountDetailReqVo.getReplenishmentProductList();
            if (CollectionUtils.isNotEmpty(tpmProductInfoReqVos)) {
                String productNames = tpmProductInfoReqVos.stream().map(TpmAccountProductReqVo::getProductName).collect(Collectors.joining(","));
                tpmAccountDetailReqVo.setProductNames(productNames);
            }
            tpmAccountDetailReqVo.setGroupId(groupId);
            if(CollectionUtils.isNotEmpty(tpmAccountDetailReqVo.getActivityFileList())){
                tpmAccountDetailReqVo.getActivityFileList().forEach(r->{
                    r.setAccountCode(tpmAccountDetailReqVo.getAccountCode());
                    r.setGroupId(tpmAccountDetailReqVo.getGroupId());
                    fileVos.add(r);
                });
            }
        });
        List<TpmAccountInvoiceReqVo> accountInvoiceReqVos = accountDetailMainReqVo.getAccountInvoiceVos();
        //保存费用上账数据
        List<TpmAccountDetailEntity> accountDetailEntities = CrmBeanUtil.copyList(accountDetailReqVos, TpmAccountDetailEntity.class);
        this.saveBatch(accountDetailEntities);
        //保存产品数据
        this.saveProductInfo(accountDetailMainReqVo.getNeedSaveProductList(), groupId);
        //保存发票数据
        this.saveInvoice(accountInvoiceReqVos, groupId);
        //保存附件
        this.saveFiles(fileVos);
        //更新核销明细的已上账金额和是否完全上账字段
        this.updateAuditDetails(accountDetailMainReqVo.getSaveType(), accountDetailEntities);
        this.batchAccountToFeePool(accountDetailMainReqVo);
        //日志
        Object menuCodeObj = ThreadLocalUtil.getObj(GlobalParam.MENU_CODE);
        for (TpmAccountDetailEntity detailEntity : accountDetailEntities) {
            TpmAccountDetailReqVo newData = CrmBeanUtil.copy(detailEntity, TpmAccountDetailReqVo.class);
            crmLogSendUtil.sendForAdd(menuCodeObj.toString(), newData.getId(), newData.getAccountCode(), newData);
        }
    }

    private void saveFiles(List<TpmAccountFileVo> fileVos) {
        if(CollectionUtils.isNotEmpty(fileVos)){
            List<String> fileIds = fileVos.stream().filter(o -> StringUtils.isNotEmpty(o.getId())).map(TpmAccountFileVo::getId).collect(Collectors.toList());
            accountFileMapper.delete(Wrappers.lambdaQuery(TpmAccountFileEntity.class).eq(TpmAccountFileEntity::getGroupId,fileVos.get(0).getGroupId()).notIn(CollectionUtils.isNotEmpty(fileIds),TpmAccountFileEntity::getId,fileIds));
            List<TpmAccountFileEntity> tpmAccountFileEntities = CrmBeanUtil.copyList(fileVos, TpmAccountFileEntity.class);
            fileService.saveOrUpdateBatch(tpmAccountFileEntities);

        }
    }

    /**
     * 批量上账到费用池
     * @param accountDetailMainReqVo
     */
    private void batchAccountToFeePool(TpmAccountDetailMainReqVo accountDetailMainReqVo){
        if(StringUtils.equals(AccountStatusEnum.SUBMIT.getCode(),accountDetailMainReqVo.getSaveType())){
            List<TpmAccountDetailReqVo> accountDetailVos = accountDetailMainReqVo.getAccountDetailVos();
            List<FeePoolAccountReqVo> feePoolAccountReqVoList = accountDetailVos.stream()
                    .filter(o->StringUtils.isNotEmpty(o.getCustomerCode()))
                    .filter(o->StringUtils.equals(ActPayTypeEnum.DISCOUNT.getCode(),o.getPayType())||StringUtils.equals(ActPayTypeEnum.REPLENISHMENT.getCode(),o.getPayType()))
                    .map(o -> {
                FeePoolAccountReqVo feePoolVo = new FeePoolAccountReqVo();
                CrmBeanUtil.copyProperties(o, feePoolVo);
                feePoolVo.setAmount(o.getAccountAmount());
                feePoolVo.setFromDesc("TPM费用上账");
                feePoolVo.setFromCode(o.getAccountCode());
                feePoolVo.setOperationType(FeePoolOperationTypeEnum.ACT_ACCOUNT.getValue());
                if (StringUtils.equals(ActPayTypeEnum.REPLENISHMENT.getCode(),o.getPayType())) {
                    feePoolVo.setPoolType(FeePoolTypeEnum.GOODS.getValue());
                    List<String> codes = o.getReplenishmentProductList().stream().map(TpmAccountProductReqVo::getProductCode).collect(Collectors.toList());
                    feePoolVo.setProductCodeList(Optional.ofNullable(codes).orElse(null));
                    feePoolVo.setProductLevelCode(StringUtils.isNotEmpty(o.getProductLevelCode()) ? o.getProductLevelCode() : null);
                } else {
                    feePoolVo.setPoolType(FeePoolTypeEnum.DISCOUNT.getValue());
                }
                accountServiceHelper.setPublicParamsNull(feePoolVo);
                return feePoolVo;
            }).collect(Collectors.toList());
            if(CollectionUtils.isNotEmpty(feePoolAccountReqVoList)){
                ApiResultUtil.objResult(feePoolFeign.accountBatch(feePoolAccountReqVoList),true);
            }
        }
    }
    /**
     * 更新核销明细的已上账金额和是否完全上账字段
     *
     * @param saveType
     * @param accountDetailEntities
     */
    public void updateAuditDetails(String saveType, List<TpmAccountDetailEntity> accountDetailEntities) {
        if (StringUtils.isNotEmpty(accountDetailEntities.get(0).getAttachCode())) {
            this.updateLiqueurAuditDetails(saveType, accountDetailEntities);
        } else if (StringUtils.equals(AccountStatusEnum.SUBMIT.getCode(), saveType)) {
            Map<String, List<TpmAccountDetailEntity>> map = Maps.newHashMap();
            List<String> auditDetailCodes = Lists.newArrayList();
            accountDetailEntities.forEach(o -> {
                List<TpmAccountDetailEntity> groupAccountDetailEntities= map.computeIfAbsent(o.getAuditDetailCode(),  k -> new ArrayList<>());
                groupAccountDetailEntities.add(o);
                auditDetailCodes.add(o.getAuditDetailCode());
            });
            //查询核销明细
            List<TpmAuditDetailEntity> tpmAuditDetailEntities = auditDetailMapper.selectList(Wrappers.lambdaQuery(TpmAuditDetailEntity.class).in(TpmAuditDetailEntity::getAuditDetailCode, auditDetailCodes.stream().distinct().collect(Collectors.toList())));
            //使用客户编码加核销编码找到核销明细对应的客户信息
            List<String> customerCodes = accountDetailEntities.stream().filter(o -> StringUtils.isNotBlank(o.getCustomerCode())).map(TpmAccountDetailEntity::getCustomerCode).collect(Collectors.toList());
            List<TpmAuditDetailCustomerEntity> allCustomerEntities = auditDetailCustomerService
                    .list(Wrappers.lambdaQuery(TpmAuditDetailCustomerEntity.class)
                            .in(TpmAuditDetailCustomerEntity::getAuditDetailCode, accountDetailEntities.stream().map(TpmAccountDetailEntity::getAuditDetailCode).distinct().collect(Collectors.toList()))
                            .in(CollectionUtils.isNotEmpty(customerCodes), TpmAuditDetailCustomerEntity::getCustomerCode, customerCodes));
            Map<String, TpmAuditDetailCustomerEntity> auditDetailCustomerEntityMap = allCustomerEntities.stream().collect(Collectors.toMap(o1 -> o1.getAuditDetailCode() + "-" + o1.getCustomerCode(), o2 -> o2));
            tpmAuditDetailEntities.forEach(o -> {
                List<TpmAccountDetailEntity> tpmAccountDetailEntities = map.get(o.getAuditDetailCode());
                BigDecimal all = Optional.ofNullable(o.getFeeUsed()).orElse(BigDecimal.ZERO);
                for (TpmAccountDetailEntity tpmAccountDetailEntity : tpmAccountDetailEntities) {
                    all = all.add(tpmAccountDetailEntity.getAccountAmount());
                    //更新核销明细客户信息,修改上账信息
                    TpmAuditDetailCustomerEntity auditDetailCustomerEntity = auditDetailCustomerEntityMap.get(tpmAccountDetailEntity.getAuditDetailCode() + "-" + tpmAccountDetailEntity.getCustomerCode());
                    if(auditDetailCustomerEntity != null){
                        BigDecimal use = tpmAccountDetailEntity.getAccountAmount().add(auditDetailCustomerEntity.getFeeUsed());
                        auditDetailCustomerEntity.setFeeUsed(use);
                        if(use.compareTo(auditDetailCustomerEntity.getAuditApplyAmount()) < 0){
                            auditDetailCustomerEntity.setIsAllFeeUsed(GlobalWhetherEnum.NO.getCode());
                        }else {
                            auditDetailCustomerEntity.setIsAllFeeUsed(GlobalWhetherEnum.YES.getCode());
                        }
                    }
                }
                if (o.getAuditApplyAmount().compareTo(all) > 0) {
                    o.setIsAllFeeUsed(GlobalWhetherEnum.NO.getCode());
                } else {
                    o.setIsAllFeeUsed(GlobalWhetherEnum.YES.getCode());
                }
                o.setFeeUsed(all);
            });
            auditDetailCustomerService.updateBatchById(allCustomerEntities);
            auditDetailService.updateBatchById(tpmAuditDetailEntities);
        }
    }

    /**
     * 酒类 更新核销明细的已上账金额和是否完全上账字段
     *
     * @param saveType
     * @param accountDetailEntities
     */
    public void updateLiqueurAuditDetails(String saveType, List<TpmAccountDetailEntity> accountDetailEntities) {
        if (StringUtils.equals(AccountStatusEnum.SUBMIT.getCode(), saveType)) {
            Map<String, TpmAccountDetailEntity> map = Maps.newHashMap();
            List<String> auditAttachCodes = Lists.newArrayList();
            accountDetailEntities.stream().forEach(o -> {
                map.put(o.getAttachCode(), o);
                auditAttachCodes.add(o.getAttachCode());
            });
            //查询核销附加信息
            List<TpmLiqueurAuditAttachEntity> tpmLiqueurAuditAttachEntities = liqueurAuditAttachMapper.selectList(Wrappers.lambdaQuery(TpmLiqueurAuditAttachEntity.class).in(TpmLiqueurAuditAttachEntity::getAttachCode, auditAttachCodes));
            tpmLiqueurAuditAttachEntities.stream().forEach(o -> {
                TpmAccountDetailEntity tpmAccountDetailEntity = map.get(o.getAttachCode());
                BigDecimal add = tpmAccountDetailEntity.getAccountAmount().add(Optional.ofNullable(o.getFeeUsed()).orElse(BigDecimal.ZERO));
                if (o.getAuditAmount().compareTo(add) == 1) {
                    o.setIsAllFeeUsed(GlobalWhetherEnum.NO.getCode());
                } else {
                    o.setIsAllFeeUsed(GlobalWhetherEnum.YES.getCode());
                }
                o.setFeeUsed(add);
            });
            liqueurAuditAttachService.updateBatchById(tpmLiqueurAuditAttachEntities);
        }
    }

    /**
     * 保存产品数据
     *
     * @param needSaveProductList
     * @param groupId
     */
    private void saveProductInfo(List<TpmAccountProductReqVo> needSaveProductList, String groupId) {
        if (CollectionUtils.isNotEmpty(needSaveProductList)) {
            List<String> ids = needSaveProductList.stream().filter(o -> StringUtils.isNotBlank(o.getId())).map(TpmAccountProductReqVo::getId).collect(Collectors.toList());
            tpmAccountProductMapper.delete(Wrappers.lambdaQuery(TpmAccountProductEntity.class).eq(TpmAccountProductEntity::getGroupId, groupId).notIn(CollectionUtils.isNotEmpty(ids), TpmAccountProductEntity::getId, ids));
            List<TpmAccountProductEntity> productEntities = needSaveProductList.stream().map(o -> {
                TpmAccountProductEntity productEntity = CrmBeanUtil.copy(o, TpmAccountProductEntity.class);
                productEntity.setGroupId(groupId);
                return productEntity;
            }).collect(Collectors.toList());
            accountProductService.saveOrUpdateBatch(productEntities);
        } else {
            //如果新增的时候有产品,编辑的时候没有了,那就整体删一次
            tpmAccountProductMapper.delete(Wrappers.lambdaQuery(TpmAccountProductEntity.class).eq(TpmAccountProductEntity::getGroupId, groupId));
        }
    }

    /**
     * 保存发票数据
     *
     * @param accountInvoiceReqVos
     * @param groupId
     */
    private void saveInvoice(List<TpmAccountInvoiceReqVo> accountInvoiceReqVos,
                             String groupId) {

        if (CollectionUtils.isNotEmpty(accountInvoiceReqVos)) {
            //查数据字典
            List<String> dictCodes = Lists.newArrayList();
            dictCodes.add(TpmGlobalDictConstants.VAT);
            Map<String, Map<String, String>> map = DictUtil.getDictValueMapsByCodes(dictCodes);
            accountInvoiceReqVos.stream().forEach(o -> {
                o.setTotalInvoiceAmount(o.getInvoiceAmount().add(o.getTaxAmount()));
                o.setTaxRate(Optional.ofNullable(map.get(TpmGlobalDictConstants.VAT)).orElse(Maps.newHashMap()).get(o.getTaxCode()));
            });
            List<String> ids = accountInvoiceReqVos.stream().filter(o -> StringUtils.isNotBlank(o.getId())).map(TpmAccountInvoiceReqVo::getId).collect(Collectors.toList());
            tpmAccountInvoiceMapper.delete(Wrappers.lambdaQuery(TpmAccountInvoiceEntity.class).eq(TpmAccountInvoiceEntity::getGroupId, groupId).notIn(CollectionUtils.isNotEmpty(ids), TpmAccountInvoiceEntity::getId, ids));
            List<TpmAccountInvoiceEntity> entities = accountInvoiceReqVos.stream().map(o -> {
                TpmAccountInvoiceEntity entity = CrmBeanUtil.copy(o, TpmAccountInvoiceEntity.class);
                entity.setGroupId(groupId);
                return entity;
            }).collect(Collectors.toList());
            accountInvoiceService.saveOrUpdateBatch(entities);
        } else {
            //如果新增的时候有发票,编辑的时候没有了,那就整体删一次
            tpmAccountInvoiceMapper.delete(Wrappers.lambdaQuery(TpmAccountInvoiceEntity.class).eq(TpmAccountInvoiceEntity::getGroupId, groupId));
        }

    }

    /**
     * 更新
     *
     * @param accountDetailMainReqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    @Klock(keys = {"accountDetail", "#accountDetailMainReqVo.lockUserName"}, waitTime = 0, leaseTime = 5)
    public void update(TpmAccountDetailMainReqVo accountDetailMainReqVo) {
        accountServiceHelper.saveCheck(accountDetailMainReqVo);
        Object menuCodeObj = ThreadLocalUtil.getObj(GlobalParam.MENU_CODE);
        String groupId = accountDetailMainReqVo.getAccountDetailVos().get(0).getGroupId();
        AssertUtils.isNotEmpty(groupId, "分组编码不能为空");
        List<TpmAccountDetailEntity> detailEntities = tpmAccountDetailMapper.selectList(new LambdaQueryWrapper<TpmAccountDetailEntity>()
                .eq(TpmAccountDetailEntity::getGroupId, groupId));
        Map<String, TpmAccountDetailEntity> detailEntityMap = detailEntities.stream().collect(Collectors.toMap(TpmAccountDetailEntity::getId, Function.identity()));
        List<TpmAccountDetailReqVo> accountDetailReqVos = accountDetailMainReqVo.getAccountDetailVos();
        List<TpmAccountInvoiceReqVo> accountInvoiceReqVos = accountDetailMainReqVo.getAccountInvoiceVos();
        List<String> accountDetailIds = accountDetailReqVos.stream()
                .filter(o -> StringUtils.isNotBlank(o.getId()))
                .map(TpmAccountDetailReqVo::getId).collect(Collectors.toList());

        //先将数据库需要删除数据删除
        tpmAccountDetailMapper.delete(new LambdaQueryWrapper<TpmAccountDetailEntity>()
                .eq(TpmAccountDetailEntity::getGroupId, groupId).notIn(CollectionUtils.isNotEmpty(accountDetailIds), TpmAccountDetailEntity::getId, accountDetailIds));
        List<TpmAccountDetailEntity> accountDetailEntities = Lists.newArrayList();
        List<TpmAccountFileVo> fileVos=Lists.newArrayList();
        accountDetailReqVos.stream().forEach(o -> {
        TpmAccountDetailEntity entity = CrmBeanUtil.copy(o, TpmAccountDetailEntity.class);
        if (CollectionUtils.isNotEmpty(o.getReplenishmentProductList())) {
            String productNames = o.getReplenishmentProductList().stream().map(TpmAccountProductReqVo::getProductName).collect(Collectors.joining(","));
            entity.setProductNames(productNames);
        }
        entity.setGroupId(groupId);
        accountDetailEntities.add(entity);
            if(CollectionUtils.isNotEmpty(o.getActivityFileList())){
                o.getActivityFileList().forEach(r->{
                    r.setAccountCode(o.getAccountCode());
                    r.setGroupId(o.getGroupId());
                    fileVos.add(r);
                });
            }
    });
        //更新费用上账表信息
        this.saveOrUpdateBatch(accountDetailEntities);
        //保存产品数
        this.saveProductInfo(accountDetailMainReqVo.getNeedSaveProductList(), groupId);
        //保存发票数据
        this.saveInvoice(accountInvoiceReqVos, groupId);
        //保存附件
        this.saveFiles(fileVos);
        //更新核销明细的已上账金额和是否完全上账字段
        this.updateAuditDetails(accountDetailMainReqVo.getSaveType(), accountDetailEntities);
        this.batchAccountToFeePool(accountDetailMainReqVo);

        //保存日志
        if (CollectionUtil.listNotEmptyNotSizeZero(accountDetailIds)) {
            accountDetailEntities.forEach(a -> {
                TpmAccountDetailRespVo newData = CrmBeanUtil.copy(a, TpmAccountDetailRespVo.class);
                if (accountDetailIds.contains(a.getId())) {
                    //添加更新日志
                    TpmAccountDetailRespVo oldData = CrmBeanUtil.copy(detailEntityMap.get(a.getId()), TpmAccountDetailRespVo.class);
                    crmLogSendUtil.sendForUpdate(menuCodeObj.toString(), newData.getId(), newData.getAccountCode(), oldData, newData);
                } else {
                    //添加新增日志
                    crmLogSendUtil.sendForAdd(menuCodeObj.toString(), newData.getId(), newData.getAccountCode(), newData);
                }
            });
            detailEntities.forEach(a -> {
                if (!accountDetailIds.contains(a.getId())) {
                    //添加删除日志
                    TpmAccountDetailRespVo oldData = CrmBeanUtil.copy(detailEntityMap.get(a.getId()), TpmAccountDetailRespVo.class);
                    crmLogSendUtil.sendForDel(menuCodeObj.toString(), oldData.getId(), oldData.getAccountCode(), oldData);
                }
            });
        }
    }

    /**
     * 删除
     *
     * @param ids id集合
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteBatch(List<String> ids) {

        AssertUtils.isNotEmpty(ids, "id集合不能为空");

        List<TpmAccountDetailEntity> accountDetailEntities = tpmAccountDetailMapper.selectBatchIds(ids);
        if (CollectionUtils.isEmpty(accountDetailEntities)) {
            return;
        }
        List<String> accountCodes = Lists.newArrayList();
        Set<String> groupIds = Sets.newHashSet();
        accountDetailEntities.forEach(o -> {
            AssertUtils.isTrue(StringUtils.equals(o.getApproveStatus(), AccountStatusEnum.TEMP.getCode()), "只有暂存状态的数据允许删除");
            accountCodes.add(o.getAccountCode());
            groupIds.add(o.getGroupId());
        });
        tpmAccountDetailMapper.deleteBatchIds(ids);
        //删除产品数据
        tpmAccountProductMapper.delete(Wrappers.lambdaQuery(TpmAccountProductEntity.class).in(TpmAccountProductEntity::getAccountCode, accountCodes));
        //删除附件
        accountFileMapper.delete(Wrappers.lambdaQuery(TpmAccountFileEntity.class).in(TpmAccountFileEntity::getAccountCode,accountCodes));
        //删除发票数据
        List<TpmAccountInvoiceEntity> invoiceEntities = tpmAccountInvoiceMapper.selectList(Wrappers.lambdaQuery(TpmAccountInvoiceEntity.class).in(TpmAccountInvoiceEntity::getGroupId, groupIds));
        if (CollectionUtils.isNotEmpty(invoiceEntities)) {
            List<String> needDeleteInvoiceIds = Lists.newArrayList();
            List<TpmAccountInvoiceEntity> needUpdateInvoices = Lists.newArrayList();
            invoiceEntities.stream().forEach(o -> {
                List<String> codes = JSON.parseArray(o.getAccountCodes(), String.class);
                codes.removeAll(accountCodes);
                if (CollectionUtils.isEmpty(codes)) {
                    needDeleteInvoiceIds.add(o.getId());
                } else {
                    //如果移除后的集合长度和移除之前的长度一样,说明不需要处理,否则更新发票关联的上账单集合
                    if (codes.size() != JSON.parseArray(o.getAccountCodes(), String.class).size()) {
                        o.setAccountCodes(JSON.toJSONString(codes));
                        needUpdateInvoices.add(o);
                    }
                }
            });
            if (CollectionUtils.isNotEmpty(needDeleteInvoiceIds)) {
                tpmAccountInvoiceMapper.delete(Wrappers.lambdaQuery(TpmAccountInvoiceEntity.class).in(TpmAccountInvoiceEntity::getId, needDeleteInvoiceIds));
            }
            if (CollectionUtils.isNotEmpty(needUpdateInvoices)) {
                accountInvoiceService.updateBatchById(needUpdateInvoices);
            }
        }
        //日志
        Object menuCodeObj = ThreadLocalUtil.getObj(GlobalParam.MENU_CODE);
        for (TpmAccountDetailEntity oldData:accountDetailEntities) {
            crmLogSendUtil.sendForDel(menuCodeObj.toString(),oldData.getId(),oldData.getAccountCode(),oldData);
        }


    }

    /**
     * 启用
     *
     * @param ids
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void enableBatch(List<String> ids) {
        //设置状态为启用
        List<TpmAccountDetailEntity> tpmAccountDetailEntities = tpmAccountDetailMapper.selectBatchIds(ids);
        if (CollectionUtils.isNotEmpty(tpmAccountDetailEntities)) {
            tpmAccountDetailEntities.forEach(o -> {
                o.setEnableStatus(CrmEnableStatusEnum.ENABLE.getCode());
            });
        }
        this.updateBatchById(tpmAccountDetailEntities);
    }

    /**
     * 禁用
     *
     * @param ids
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void disableBatch(List<String> ids) {
        //设置状态为禁用
        List<TpmAccountDetailEntity> tpmAccountDetailEntities = tpmAccountDetailMapper.selectBatchIds(ids);
        if (CollectionUtils.isNotEmpty(tpmAccountDetailEntities)) {
            tpmAccountDetailEntities.forEach(o -> {
                o.setEnableStatus(CrmEnableStatusEnum.DISABLE.getCode());
            });
        }
        this.updateBatchById(tpmAccountDetailEntities);
    }

    /**
     * 批量修改支付方式
     *
     * @param updatePayTypeReqVo
     */
    @Override
    public void batchUpdatePayType(TpmAccountUpdatePayTypeReqVo updatePayTypeReqVo) {
//        accountServiceHelper.batchUpdatePayTypeCheck(updatePayTypeReqVo);
//        List<String> ids = updatePayTypeReqVo.getIds();
//        ids.forEach(id -> {
//            TpmAccountDetailEntity tpmAccountDetailEntity = new TpmAccountDetailEntity();
//            tpmAccountDetailEntity.setPayType(updatePayTypeReqVo.getPayType());
//            tpmAccountDetailEntity.setPayTypeName(updatePayTypeReqVo.getPayTypeName());
//            tpmAccountDetailEntity.setId(id);
//            tpmAccountDetailMapper.updateById(tpmAccountDetailEntity);
//        });
    }

    /**
     * 根据支付方式类型和活动细类集合查询支付方式
     *
     * @param typeAndFineVo
     * @return
     */
    @Override
    public List<DictItemVo> payTypeByTypeAndFine(TpmPayTypeByTypeAndFineVo typeAndFineVo) {
        AssertUtils.isNotEmpty(typeAndFineVo.getPayTypeType(), "支付方式类型不能为空");
        AssertUtils.isNotEmpty(typeAndFineVo.getFineCodes(), "活动细类编码集合不能为空");
        Set<String> collect = typeAndFineVo.getFineCodes().stream().collect(Collectors.toSet());
        List<TpmCostTypeFineEntity> tpmCostTypeFineEntities = tpmCostTypeFineMapper.selectList(Wrappers.<TpmCostTypeFineEntity>lambdaQuery()
                .in(TpmCostTypeFineEntity::getFineCode, collect));
        Set<DictItemVo> payTypeSet = Sets.newConcurrentHashSet();
        if (CollectionUtil.listNotEmpty(tpmCostTypeFineEntities)) {
            Map<String, String> map = iTpmAuditService.queryDictValueMap();
            List<String> payTypeCodes = JSON.parseArray(tpmCostTypeFineEntities.get(0).getPayTypeList(), String.class);
            for (TpmCostTypeFineEntity tpmCostTypeFineEntity : tpmCostTypeFineEntities) {
                String payTypeList = tpmCostTypeFineEntity.getPayTypeList();
                List<String> strings = JSON.parseArray(payTypeList, String.class);
                payTypeCodes.retainAll(strings);
            }
            List<DictItemVo> payTypes = iTpmAuditService.getPayTypes(JSON.toJSONString(payTypeCodes), map);
            payTypeSet.addAll(payTypes);
        }
        if (CollectionUtils.isEmpty(payTypeSet)) {
            return Lists.newArrayList();
        }
        return payTypeByType(typeAndFineVo.getPayTypeType(), payTypeSet);
    }

    /**
     * 根据支付方式类型筛选出支付方式
     *
     * @param payTypeType
     * @param payTypeSet
     * @return
     */
    public List<DictItemVo> payTypeByType(String payTypeType, Set<DictItemVo> payTypeSet) {
        //需要发票：1 现金、2 转预付款 、3 账扣
        //费用池：4 货补 、5 折扣、7 票折
        List<DictItemVo> resultList;
        if (StringUtils.equals(ActPayTypeTypeEnum.NEED_INVOICE.getCode(), payTypeType)) {
            resultList = payTypeSet.stream().filter(o -> StringUtils.equals(o.getDictValue(), ActPayTypeEnum.CASH.getCode()) ||
                    StringUtils.equals(o.getDictValue(), ActPayTypeEnum.FORWARD_PAYMENT.getCode()) ||
                    StringUtils.equals(o.getDictValue(), ActPayTypeEnum.ACCOUNT_DEDUCTION.getCode())
            ).collect(Collectors.toList());
        } else if (StringUtils.equals(ActPayTypeTypeEnum.DEFAULT_FEE_POOL.getCode(), payTypeType)) {
            resultList = payTypeSet.stream().filter(o -> StringUtils.equals(o.getDictValue(), ActPayTypeEnum.REPLENISHMENT.getCode()) ||
                    StringUtils.equals(o.getDictValue(), ActPayTypeEnum.DISCOUNT.getCode()) ||
                    StringUtils.equals(o.getDictValue(), ActPayTypeEnum.TICKET_DISCOUNT.getCode())
            ).collect(Collectors.toList());
        } else {
            throw new BusinessException("支付方式类型不存在");
        }
        return resultList;
    }
}
