package com.biz.crm.act.service.impl;

import com.baomidou.mybatisplus.core.mapper.BaseMapper;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.act.mapper.TpmActDetailMapper;
import com.biz.crm.act.model.TpmActDetailEntity;
import com.biz.crm.act.service.ITpmActDetailService;
import com.biz.crm.advancepay.service.impl.ActAdvancePayServiceHelper;
import com.biz.crm.common.PageResult;
import com.biz.crm.common.TpmGlobalDictConstants;
import com.biz.crm.eunm.CrmDelFlagEnum;
import com.biz.crm.eunm.CrmEnableStatusEnum;
import com.biz.crm.eunm.tpm.ActAdvancePayTypeEnum;
import com.biz.crm.nebular.tpm.act.req.TpmActDetailReportReqVo;
import com.biz.crm.nebular.tpm.act.req.TpmActDetailReqVo;
import com.biz.crm.nebular.tpm.act.resp.TpmActDetailReportRespVo;
import com.biz.crm.nebular.tpm.act.resp.TpmActDetailRespVo;
import com.biz.crm.util.*;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.math.BigDecimal;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;

/**
 * 活动明细信息表;接口实现
 *
 * @author huanglong
 * @date 2020-09-18 13:46:33
 */
@Slf4j
@Service
@ConditionalOnMissingBean(name="TpmActDetailServiceExpandImpl")
public class TpmActDetailServiceImpl<M extends BaseMapper<T>,T> extends ServiceImpl<TpmActDetailMapper, TpmActDetailEntity> implements ITpmActDetailService {

    @Resource
    private TpmActDetailMapper tpmActDetailMapper;

    @Autowired
    private ActAdvancePayServiceHelper advancePayServiceHelper;

    @Resource
    private ActServiceHelper serviceHelper;

    /**
     * 列表
     * @param reqVo
     * @return
     */
    @Override
    public PageResult<TpmActDetailRespVo> findList(TpmActDetailReqVo reqVo){
        Page<TpmActDetailRespVo> page = PageUtil.buildPage(reqVo.getPageNum(), reqVo.getPageSize());
        reqVo.setSelectedCodeList(serviceHelper.dealSelectedCodeList(reqVo.getSelectedCode(),reqVo.getSelectedCodeList()));
        List<TpmActDetailRespVo> list = tpmActDetailMapper.findList(page, reqVo);
        return PageResult.<TpmActDetailRespVo>builder()
                .data(list)
                .count(page.getTotal())
                .build();
    }

    /**
     * 查询
     * @param id
     * @return tpmActDetailRespVo
     */
    @Override
    public TpmActDetailRespVo query(String id){
        return null;
    }

    /**
     * 新增
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void save(TpmActDetailReqVo reqVo){
        TpmActDetailEntity entity = CrmBeanUtil.copy(reqVo,TpmActDetailEntity.class);
        this.save(entity);
    }

    /**
     * 更新
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void update(TpmActDetailReqVo reqVo){
        TpmActDetailEntity entity = this.getById(reqVo.getId());
        this.updateById(entity);
    }

    /**
     * 删除
     * @param ids id集合
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteBatch(List<String> ids){
        List<TpmActDetailEntity> tpmActDetailEntities = tpmActDetailMapper.selectBatchIds(ids);
        if(CollectionUtils.isNotEmpty(tpmActDetailEntities)){
            tpmActDetailEntities.forEach(o -> {
                 o.setDelFlag(CrmDelFlagEnum.DELETE.getCode());
            });
        }
        this.updateBatchById(tpmActDetailEntities);
    }

    /**
     * 启用
     * @param ids id集合
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void enableBatch(List<String> ids){
        //设置状态为启用
        List<TpmActDetailEntity> tpmActDetailEntities = tpmActDetailMapper.selectBatchIds(ids);
        if(CollectionUtils.isNotEmpty(tpmActDetailEntities)){
            tpmActDetailEntities.forEach(o -> {
                o.setEnableStatus(CrmEnableStatusEnum.ENABLE.getCode());
            });
        }
        this.updateBatchById(tpmActDetailEntities);
    }

    /**
     * 禁用
     * @param ids id集合
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void disableBatch(List<String> ids){
        //设置状态为禁用
        List<TpmActDetailEntity> tpmActDetailEntities = tpmActDetailMapper.selectBatchIds(ids);
        if(CollectionUtils.isNotEmpty(tpmActDetailEntities)){
                tpmActDetailEntities.forEach(o -> {
                o.setEnableStatus(CrmEnableStatusEnum.DISABLE.getCode());
            });
        }
        this.updateBatchById(tpmActDetailEntities);
    }

    /**
     * 活动预付分页查询活动明细列表
     * @param reqVo
     * @return
     */
    @Override
    public PageResult<TpmActDetailRespVo> pageForAdvancePay(TpmActDetailReqVo reqVo) {
        AssertUtils.isNotEmpty(reqVo.getAdvanceType(),"活动预付类型不能为空");
        if(StringUtils.equals(ActAdvancePayTypeEnum.ACT.getCode(),reqVo.getAdvanceType())){
            AssertUtils.isNotEmpty(reqVo.getActCode(),"请先选择活动");
        }
        if(StringUtils.equals(ActAdvancePayTypeEnum.CUSTOMER.getCode(),reqVo.getAdvanceType())){
            AssertUtils.isNotEmpty(reqVo.getCustomerCode(),"请先选择客户");
        }
        Page<TpmActDetailRespVo> page = PageUtil.buildPage(reqVo.getPageNum(), reqVo.getPageSize());
        List<TpmActDetailRespVo> data = tpmActDetailMapper.pageForAdvancePay(page,reqVo);
        if(data.size()>0){
            //获取所有的明细编码,查询明细编码对应的已申请预付金额
            List<String> codes = data.stream().map(TpmActDetailRespVo::getActDetailCode).collect(Collectors.toList());
            Map<String, BigDecimal> map = advancePayServiceHelper.getActDetailsAdvanceAmount(codes);
            data.forEach(o->{
                o.setAlreadyApplyAdvanceTotalAmount(Optional.ofNullable(map.get(o.getActDetailCode())).orElse(BigDecimal.ZERO));
            });
        }
        return PageResult.<TpmActDetailRespVo>builder()
                .data(data)
                .count(page.getTotal())
                .build();
    }

    @Override
    public PageResult<TpmActDetailReportRespVo> actDetailReport(TpmActDetailReportReqVo reqVo) {
        Page<TpmActDetailReportRespVo> page = PageUtil.buildPage(reqVo.getPageNum(), reqVo.getPageSize());
        List<TpmActDetailReportRespVo> data = tpmActDetailMapper.actDetailReport(page,reqVo);
        convertData(data);
        return PageResult.<TpmActDetailReportRespVo>builder()
                .data(data)
                .count(page.getTotal())
                .build();
    }

    private void convertData(List<TpmActDetailReportRespVo> data){
        if(CollectionUtil.listNotEmpty(data)){
            Map<String, Map<String, String>> map = DictUtil.getDictValueMapsByCodes(Lists.newArrayList(TpmGlobalDictConstants.ACT_APPROVE_STATUS, TpmGlobalDictConstants.ACCOUNT_STATUS));
            data.forEach(o -> {
                o.setApproveStatus(Optional.ofNullable(map.get(TpmGlobalDictConstants.ACT_APPROVE_STATUS)).orElse(Maps.newHashMap()).get(o.getApproveStatus()));
                o.setAuditApproveStatus(Optional.ofNullable(map.get(TpmGlobalDictConstants.ACT_APPROVE_STATUS)).orElse(Maps.newHashMap()).get(o.getAuditApproveStatus()));
                o.setAccountStatus(Optional.ofNullable(map.get(TpmGlobalDictConstants.ACCOUNT_STATUS)).orElse(Maps.newHashMap()).get(o.getAccountStatus()));
            });
        }
    }
}
