package com.biz.crm.advancepay.service.impl;

import com.alibaba.fastjson.JSON;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.mapper.BaseMapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.biz.crm.act.model.TpmActEntity;
import com.biz.crm.act.service.impl.ActServiceHelper;
import com.biz.crm.activiti.act.TaActBaseFeign;
import com.biz.crm.advancepay.mapper.TpmActAdvancePayDetailMapper;
import com.biz.crm.advancepay.model.TpmActAdvancePayDetailEntity;
import com.biz.crm.advancepay.service.ITpmActAdvancePayDetailService;
import com.biz.crm.annotation.Klock;
import com.biz.crm.base.BusinessException;
import com.biz.crm.base.config.ThreadLocalUtil;
import com.biz.crm.common.GlobalParam;
import com.biz.crm.crmlog.handle.util.CrmLogSendUtil;
import com.biz.crm.eunm.activiti.Indicator;
import com.biz.crm.eunm.tpm.ActAdvancePayApproveStatusEnum;
import com.biz.crm.eunm.tpm.ActApproveStatusEnum;
import com.biz.crm.eunm.tpm.ActSaveTypeEnum;
import com.biz.crm.eunm.tpm.TpmSaveTypeEnum;
import com.biz.crm.exception.tpm.ActAdvancePayException;
import com.biz.crm.exception.tpm.FeeBudgetException;
import com.biz.crm.exception.tpm.FiscalYearSettingException;
import com.biz.crm.nebular.activiti.act.ActivitiCallBackVo;
import com.biz.crm.nebular.activiti.act.req.StartProcessReqVo;
import com.biz.crm.nebular.activiti.common.ProcessCommonVo;
import com.biz.crm.nebular.tpm.act.resp.TpmActRespVo;
import com.biz.crm.nebular.tpm.advancepay.req.TpmActAdvancePayDetailReqVo;
import com.biz.crm.nebular.tpm.advancepay.resp.TpmActAdvancePayDetailRespVo;
import com.biz.crm.util.*;
import com.google.common.collect.Lists;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.stereotype.Service;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.common.PageResult;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.advancepay.model.TpmActAdvancePayEntity;
import com.biz.crm.advancepay.mapper.TpmActAdvancePayMapper;
import com.biz.crm.advancepay.service.ITpmActAdvancePayService;
import org.apache.commons.collections4.CollectionUtils;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

import org.apache.commons.collections4.MapUtils;
import lombok.extern.slf4j.Slf4j;
import org.springframework.transaction.annotation.Transactional;
import com.biz.crm.nebular.tpm.advancepay.resp.TpmActAdvancePayRespVo;
import com.biz.crm.nebular.tpm.advancepay.req.TpmActAdvancePayReqVo;
import com.biz.crm.eunm.CrmDelFlagEnum;
import com.biz.crm.eunm.CrmEnableStatusEnum;

import javax.annotation.Resource;

/**
 * 活动预付主表 接口实现
 *
 * @author huanglong
 * @date 2020-09-27 10:46:08
 */
@Slf4j
@Service
@ConditionalOnMissingBean(name="TpmActAdvancePayServiceExpandImpl")
public class TpmActAdvancePayServiceImpl<M extends BaseMapper<T>,T> extends ServiceImpl<TpmActAdvancePayMapper, TpmActAdvancePayEntity> implements ITpmActAdvancePayService {

    @Resource
    private TpmActAdvancePayMapper tpmActAdvancePayMapper;
    @Autowired
    private ActAdvancePayServiceHelper serviceHelper;
    @Resource
    private TpmActAdvancePayDetailMapper detailMapper;
    @Autowired
    private ITpmActAdvancePayDetailService detailService;

    @Autowired
    private TaActBaseFeign activityFeign;

    @Autowired
    private CrmLogSendUtil crmLogSendUtil;

    @Resource
    private ActServiceHelper actServiceHelper;

    /**
     * 列表
     * @param reqVo
     * @return
     */
    @Override
    public PageResult<TpmActAdvancePayRespVo> findList(TpmActAdvancePayReqVo reqVo){
        Page<TpmActAdvancePayRespVo> page = PageUtil.buildPage(reqVo.getPageNum(), reqVo.getPageSize());
        reqVo.setSelectedCodeList(actServiceHelper.dealSelectedCodeList(reqVo.getSelectedCode(),reqVo.getSelectedCodeList()));
        List<TpmActAdvancePayRespVo> list = tpmActAdvancePayMapper.findList(page, reqVo);
        serviceHelper.convertListDate(list);
        return PageResult.<TpmActAdvancePayRespVo>builder()
                .data(list)
                .count(page.getTotal())
                .build();
    }

    /**
     * 查询
     * @param id
     * @return tpmActAdvancePayRespVo
     */
    @Override
    public TpmActAdvancePayRespVo query(String id){
        AssertUtils.isNotEmpty(id, "id不能为空");
        TpmActAdvancePayReqVo tpmActAdvancePayReqVo = new TpmActAdvancePayReqVo();
        tpmActAdvancePayReqVo.setId(id);
        List<TpmActAdvancePayRespVo> data = this.findList(tpmActAdvancePayReqVo).getData();
        if(CollectionUtils.isEmpty(data)){
            return new TpmActAdvancePayRespVo();
        }
        TpmActAdvancePayRespVo respVo = data.get(0);
        //查询明细数据
        TpmActAdvancePayDetailReqVo detailReqVo = new TpmActAdvancePayDetailReqVo();
        detailReqVo.setPageSize(-1);
        detailReqVo.setAdvanceCode(respVo.getAdvanceCode());
        List<TpmActAdvancePayDetailRespVo> detailRespVos = detailService.findList(detailReqVo).getData();
        respVo.setDetailVos(detailRespVos);
        return respVo;
    }

    /**
     * 新增
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    @Klock(keys = {"pay","#reqVo.lockUserName"},waitTime = 0,leaseTime = 5)
    public void save(TpmActAdvancePayReqVo reqVo){
        serviceHelper.saveCheck(reqVo);
        AssertUtils.isTrue(ActSaveTypeEnum.getCreateTypes().contains(reqVo.getSaveType()),"保存类型值错误");
        TpmActAdvancePayEntity entity = CrmBeanUtil.copy(reqVo,TpmActAdvancePayEntity.class);
        this.save(entity);
        //保存明细数据
        List<TpmActAdvancePayDetailReqVo> detailVos = reqVo.getDetailVos();
        List<TpmActAdvancePayDetailEntity> payDetailEntities = this.convertDetails(detailVos);
        detailService.saveBatch(payDetailEntities);
        //调用审批流接口
        this.sendToActivity(entity,reqVo);

        //日志
        TpmActAdvancePayRespVo newData = CrmBeanUtil.copy(entity, TpmActAdvancePayRespVo.class);
        Object menuCodeObj = ThreadLocalUtil.getObj(GlobalParam.MENU_CODE);
        crmLogSendUtil.sendForAdd(menuCodeObj.toString(),newData.getId(),newData.getBeginDateSecond(),newData);
    }
    /**
     * 发起审批
     * @param entity
     * @param reqVo
     */
    public void sendToActivity(TpmActAdvancePayEntity entity,TpmActAdvancePayReqVo reqVo){
        //保存类型:1新增暂存,2,新增并提交审批,3,修改暂存,4修改并提交审批,5,提交审批
        if(StringUtils.equals(ActApproveStatusEnum.APPROVING.getCode(),entity.getApproveStatus())){
            //提交审批
            StartProcessReqVo processReqVo=serviceHelper.buildStartProcessData(entity,reqVo);
            String processNo = ActivityUtils.startProcess(processReqVo);
            entity.setProcessCode(processNo);
            this.saveOrUpdate(entity);
        }
    }
    /**
     * 更新
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    @Klock(keys = {"pay","#reqVo.lockUserName"},waitTime = 0,leaseTime = 5)
    public void update(TpmActAdvancePayReqVo reqVo){
        AssertUtils.isNotEmpty(reqVo.getId(),"数据主键不能为空");
        TpmActAdvancePayEntity entity = this.getById(reqVo.getId());
        AssertUtils.isNotNull(entity, ActAdvancePayException.DATA_NOT_EXIST);
        AssertUtils.isTrue(ActAdvancePayApproveStatusEnum.getCanUpdateStatus().contains(entity.getApproveStatus()),"当前数据不允许修改");
        serviceHelper.saveCheck(reqVo);
        TpmActAdvancePayEntity oldEntity = this.getById(reqVo.getId());
        TpmActAdvancePayRespVo oldData = CrmBeanUtil.copy(oldEntity, TpmActAdvancePayRespVo.class);
        if(Objects.isNull(oldEntity)){
            throw new BusinessException(FiscalYearSettingException.DATA_NOT_EXIST);
        }
        AssertUtils.isTrue(ActSaveTypeEnum.getUpdateTypes().contains(reqVo.getSaveType()),"保存类型值错误");
        TpmActAdvancePayEntity updateEntity = CrmBeanUtil.copy(reqVo,TpmActAdvancePayEntity.class);
        this.updateById(updateEntity);
        //保存明细数据
        List<TpmActAdvancePayDetailReqVo> detailVos = reqVo.getDetailVos();
        List<String> detailIds = detailVos.stream().map(TpmActAdvancePayDetailReqVo::getId).collect(Collectors.toList());
        //将数据库需要删除的数据删除
        if(CollectionUtils.isNotEmpty(detailIds)){
            detailMapper.delete(new LambdaQueryWrapper<TpmActAdvancePayDetailEntity>().eq(TpmActAdvancePayDetailEntity::getAdvanceCode,entity.getAdvanceCode()).notIn(TpmActAdvancePayDetailEntity::getId,detailIds));
        }
        List<TpmActAdvancePayDetailEntity> payDetailEntities = this.convertDetails(detailVos);
        detailService.saveOrUpdateBatch(payDetailEntities);
        //调用审批流接口
        this.sendToActivity(updateEntity,reqVo);
        //日志
        Object menuCodeObj = ThreadLocalUtil.getObj(GlobalParam.MENU_CODE);
        TpmActAdvancePayRespVo newData = CrmBeanUtil.copy(entity, TpmActAdvancePayRespVo.class);
        crmLogSendUtil.sendForUpdate(menuCodeObj.toString(),newData.getId(),newData.getBeginDateSecond(),oldData,newData);
    }

    /**
     * 明细vo转实体
     * @param detailVos
     * @return
     */
    public List<TpmActAdvancePayDetailEntity> convertDetails(List<TpmActAdvancePayDetailReqVo> detailVos){
        List<TpmActAdvancePayDetailEntity> payDetailEntities = detailVos.stream().map(o -> {
            TpmActAdvancePayDetailEntity detailEntity = new TpmActAdvancePayDetailEntity();
            CrmBeanUtil.copyProperties(o, detailEntity);
            return detailEntity;
        }).collect(Collectors.toList());
        return payDetailEntities;
    }
    /**
     * 删除
     * @param ids id集合
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteBatch(List<String> ids){
        AssertUtils.isNotEmpty(ids,"请选您需要删除的数据");
        List<TpmActAdvancePayEntity> entities = tpmActAdvancePayMapper.selectBatchIds(ids);
        if(CollectionUtils.isNotEmpty(entities)){
            //只有待提交的数据可以执行删除操作
            List<String> codes = Lists.newArrayList();
            entities.stream().forEach(o->{
                codes.add(o.getAdvanceCode());
                AssertUtils.isTrue(StringUtils.equals(ActAdvancePayApproveStatusEnum.CREATE.getCode(),o.getApproveStatus()),"只有待提交的数据可以删除");
            });
            tpmActAdvancePayMapper.deleteBatchIds(ids);
            detailMapper.delete(Wrappers.lambdaQuery(TpmActAdvancePayDetailEntity.class).in(TpmActAdvancePayDetailEntity::getAdvanceCode,codes));
            //日志
            Object menuCodeObj = ThreadLocalUtil.getObj(GlobalParam.MENU_CODE);
            for (TpmActAdvancePayEntity oldData:entities) {
                crmLogSendUtil.sendForDel(menuCodeObj.toString(),oldData.getId(),oldData.getAdvanceCode(),oldData);
            }
        }
    }

    /**
     * 启用
     * @param ids id集合
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void enableBatch(List<String> ids){
        //设置状态为启用
        List<TpmActAdvancePayEntity> tpmActAdvancePayEntities = tpmActAdvancePayMapper.selectBatchIds(ids);
        if(CollectionUtils.isNotEmpty(tpmActAdvancePayEntities)){
            tpmActAdvancePayEntities.forEach(o -> {
                o.setEnableStatus(CrmEnableStatusEnum.ENABLE.getCode());
            });
        }
        this.updateBatchById(tpmActAdvancePayEntities);
    }

    /**
     * 禁用
     * @param ids id集合
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void disableBatch(List<String> ids){
        //设置状态为禁用
        List<TpmActAdvancePayEntity> tpmActAdvancePayEntities = tpmActAdvancePayMapper.selectBatchIds(ids);
        if(CollectionUtils.isNotEmpty(tpmActAdvancePayEntities)){
                tpmActAdvancePayEntities.forEach(o -> {
                o.setEnableStatus(CrmEnableStatusEnum.DISABLE.getCode());
            });
        }
        this.updateBatchById(tpmActAdvancePayEntities);
    }

    /**
     * 提交审批
     * @param reqVo
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    @Klock(keys = {"pay","#reqVo.lockUserName"},waitTime = 0,leaseTime = 5)
    public void approve(TpmActAdvancePayReqVo reqVo) {
        AssertUtils.isNotEmpty(reqVo.getId(),"数据主键不能为空");
        AssertUtils.isNotEmpty(reqVo.getWorkFlowKey(),"请选择流程");
        TpmActAdvancePayEntity entity = this.getById(reqVo.getId());
        AssertUtils.isNotNull(entity, ActAdvancePayException.DATA_NOT_EXIST);
        reqVo.setSaveType(ActSaveTypeEnum.APPROVE.getCode());
        //获取可以提交审批的状态
        Set<String> canApproveStatus = ActAdvancePayApproveStatusEnum.getCanApproveStatus();
        AssertUtils.isTrue(canApproveStatus.contains(entity.getApproveStatus()),"仅状态为“待提交、审批驳回、流程追回”的单据可提交审批");
        serviceHelper.saveCheck(reqVo);
        TpmActAdvancePayEntity updateEntity = CrmBeanUtil.copy(reqVo,TpmActAdvancePayEntity.class);
        this.updateById(updateEntity);
        //保存明细数据
        List<TpmActAdvancePayDetailReqVo> detailVos = reqVo.getDetailVos();
        List<TpmActAdvancePayDetailEntity> payDetailEntities = this.convertDetails(detailVos);
        detailService.saveOrUpdateBatch(payDetailEntities);
        //调用审批流接口
        this.sendToActivity(updateEntity,reqVo);
    }

    /**
     * 预付审批回调
     * @param activitiCallBackVo
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void activityCallBack(ActivitiCallBackVo activitiCallBackVo) {
        String formNo = activitiCallBackVo.getFormNo();
        AssertUtils.isNotEmpty(formNo,"业务编码不能为空");
        TpmActAdvancePayEntity entity = tpmActAdvancePayMapper.selectById(formNo);
        //审批通过
        if(Indicator.CON_BPM_DOING.getCode()==activitiCallBackVo.getProcessState()){
            entity.setApproveStatus(ActAdvancePayApproveStatusEnum.APPROVED.getCode());
        }
        //审批驳回
        if(Indicator.CON_BPM_PASS.getCode()==activitiCallBackVo.getProcessState()){
            entity.setApproveStatus(ActAdvancePayApproveStatusEnum.REJECTED.getCode());
        }
        //流程追回
        if(Indicator.CON_BPM_REJECT.getCode()==activitiCallBackVo.getProcessState()){
            entity.setApproveStatus(ActApproveStatusEnum.INTERRUPT.getCode());
        }
        this.updateById(entity);
    }
}
