package com.biz.crm.budgetgeneraterule.service.impl;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.biz.crm.base.ApiResultUtil;
import com.biz.crm.base.BusinessException;
import com.biz.crm.budgetgeneraterule.mapper.TpmBudgetGenerateRuleMapper;
import com.biz.crm.budgetgeneraterule.model.TpmBudgetGenerateRuleEntity;
import com.biz.crm.budgetsubjects.mapper.TpmBudgetSubjectsMapper;
import com.biz.crm.budgetsubjects.model.TpmBudgetSubjectsEntity;
import com.biz.crm.common.TpmGlobalDictConstants;
import com.biz.crm.eunm.CrmEnableStatusEnum;
import com.biz.crm.eunm.YesNoEnum;
import com.biz.crm.eunm.tpm.BudgetSubjectsControlTypeEnum;
import com.biz.crm.eunm.tpm.FeeBudgetTypeEnum;
import com.biz.crm.mdm.customer.MdmCustomerMsgFeign;
import com.biz.crm.mdm.org.MdmOrgFeign;
import com.biz.crm.mdm.terminal.MdmTerminalFeign;
import com.biz.crm.nebular.mdm.customer.MdmCustomerMsgRespVo;
import com.biz.crm.nebular.mdm.humanarea.MdmCustomerOrgSearchReqVo;
import com.biz.crm.nebular.mdm.org.req.MdmOrgReqVo;
import com.biz.crm.nebular.mdm.org.resp.MdmOrgRespVo;
import com.biz.crm.nebular.mdm.terminal.MdmCustomerTerminalVo;
import com.biz.crm.nebular.mdm.terminal.MdmTerminalVo;
import com.biz.crm.nebular.tpm.budgetgeneraterule.req.TpmBudgetGenerateRuleReqVo;
import com.biz.crm.nebular.tpm.budgetgeneraterule.resp.TpmBudgetGenerateRuleRespVo;
import com.biz.crm.util.AssertUtils;
import com.biz.crm.util.DictUtil;
import com.biz.crm.util.UserRedis;
import com.biz.crm.util.UserUtils;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.collections4.ListUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.math.BigDecimal;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.function.Function;
import java.util.stream.Collectors;

import static java.util.stream.Collectors.toMap;

/**
 * @author yb
 * @version 1.0
 * @date 2021/12/10 11:01
 * @description
 */
@Service
public class TpmBudgetGenerateRuleHelper {

    @Resource
    private TpmBudgetGenerateRuleMapper tpmBudgetGenerateRuleMapper;

    @Autowired
    private MdmOrgFeign mdmOrgFeign;

    @Resource
    private MdmCustomerMsgFeign mdmCustomerMsgFeign;

    @Autowired
    private MdmTerminalFeign terminalFeign;

    @Resource
    private TpmBudgetSubjectsMapper budgetSubjectsMapper;


    public Map<String, Map<String, String>> getDicts(){
        List<String> dictCodes= Lists.newArrayList();
        dictCodes.add(TpmGlobalDictConstants.BUDGET_YEAR);
        dictCodes.add(TpmGlobalDictConstants.BUDGET_MONTH_TRUE);
        dictCodes.add(TpmGlobalDictConstants.CHANNEL);
        dictCodes.add(TpmGlobalDictConstants.FEE_BUDGET_TYPE);
        dictCodes.add(TpmGlobalDictConstants.YES_OR_NO);
        //查询字典数据
        return DictUtil.getDictValueMapsByCodes(dictCodes);
    }

    /**
     * 转换列表数据
     * @param list
     */
    public void convertListDate(List<TpmBudgetGenerateRuleRespVo> list) {
        if(CollectionUtils.isNotEmpty(list)){
            //获取需要的字典集合
            Map<String, Map<String, String>> map = this.getDicts();
            list.forEach(o->{
                //启用禁用状态
                if(StringUtils.isNotEmpty(o.getEnableStatus())){
                    o.setEnableStatusName(Optional.ofNullable(map.get(TpmGlobalDictConstants.ENABLE_STATUS)).orElse(Maps.newHashMap()).get(o.getEnableStatus()));
                }
                //年份
                if(StringUtils.isNotEmpty(o.getYear())){
                    o.setYearName(Optional.ofNullable(map.get(TpmGlobalDictConstants.BUDGET_YEAR)).orElse(Maps.newHashMap()).get(o.getYear()));
                }
                //月份
                if(StringUtils.isNotEmpty(o.getMonth())) {
                    o.setMonthName(Optional.ofNullable(map.get(TpmGlobalDictConstants.BUDGET_MONTH_TRUE)).orElse(Maps.newHashMap()).get(o.getMonth()));
                }
                //渠道
                if(StringUtils.isNotEmpty(o.getChannel())){
                    o.setChannelName(Optional.ofNullable(map.get(TpmGlobalDictConstants.CHANNEL)).orElse(Maps.newHashMap()).get(o.getChannel()));
                }
                //费用预算类型
                if(StringUtils.isNotEmpty(o.getFeeBudgetType())){
                    o.setFeeBudgetTypeName(Optional.ofNullable(map.get(TpmGlobalDictConstants.FEE_BUDGET_TYPE)).orElse(Maps.newHashMap()).get(o.getFeeBudgetType()));
                }
                //费用预算类型
                if(o.getIsGenerate() != null){
                    o.setIsGenerateName(Optional.ofNullable(map.get(TpmGlobalDictConstants.YES_OR_NO)).orElse(Maps.newHashMap()).get(o.getIsGenerate()));
                }
            });
        }
    }

    public void checkData(TpmBudgetGenerateRuleReqVo reqVo){
        AssertUtils.isNotEmpty(reqVo.getRatio().toString(), "比例不能为空");
        AssertUtils.isNotEmpty(reqVo.getYear(),"年份不能为空");
        AssertUtils.isNotEmpty(reqVo.getMonth(),"月份不能为空");
        if(reqVo.getRatio().compareTo(BigDecimal.ZERO) < 0 || reqVo.getRatio().compareTo(new BigDecimal(100)) > 0 ){
            throw new BusinessException("比例应在0~100之间");
        }
        AssertUtils.isNotEmpty(reqVo.getFeeBudgetType(), "预算生成维度不能为空");
        TpmBudgetSubjectsEntity subjectsEntity = budgetSubjectsMapper.selectOne(new LambdaQueryWrapper<TpmBudgetSubjectsEntity>().eq(TpmBudgetSubjectsEntity::getBudgetSubjectsCode, reqVo.getBudgetSubjectCode()));
        AssertUtils.isNotNull(subjectsEntity,"预算科目数据不存在");
        AssertUtils.isTrue(StringUtils.equals(CrmEnableStatusEnum.ENABLE.getCode(),subjectsEntity.getEnableStatus()),"您选择预算科目已被禁用");
        reqVo.setBudgetSubjectName(subjectsEntity.getBudgetSubjectsName());
        if (StringUtils.equals(reqVo.getFeeBudgetType(), FeeBudgetTypeEnum.CHANNEL.getCode())) {
            if(reqVo.getChannel() != null) {
                if (!Optional.ofNullable(getDicts().get(TpmGlobalDictConstants.CHANNEL)).orElse(Maps.newHashMap())
                        .containsKey(reqVo.getChannel())) {
                    throw new BusinessException("渠道类型不正确");
                }
            }
            clearCustomerInfo(reqVo);
            clearTerminalInfo(reqVo);
        }else if(StringUtils.equals(reqVo.getFeeBudgetType(), FeeBudgetTypeEnum.CUSTOMER.getCode())){
            if (reqVo.getCustomerCode() != null){
                checkCustomer(reqVo);
            }
            clearTerminalInfo(reqVo);
            clearChannelInfo(reqVo);
        }else if(StringUtils.equals(reqVo.getFeeBudgetType(), FeeBudgetTypeEnum.TERMINAL.getCode())){
            if(reqVo.getTerminalCode() != null){
                AssertUtils.isNotEmpty(reqVo.getCustomerCode(), "客户不能为空");
                checkCustomer(reqVo);
                // 通过客户
                MdmCustomerTerminalVo mdmCustomerTerminalVo = new MdmCustomerTerminalVo();
                mdmCustomerTerminalVo.setCustomerCodeList(Lists.newArrayList(reqVo.getCustomerCode()));
                MdmTerminalVo mdmTerminalVo =ApiResultUtil.objResult(terminalFeign.query("",reqVo.getTerminalCode()));
                AssertUtils.isNotNull(mdmTerminalVo,"门店不存在");
                List<MdmTerminalVo> mdmTerminalVos =ApiResultUtil.objResult(terminalFeign.customerTerminalList(mdmCustomerTerminalVo));
                Map<String,MdmTerminalVo> mdmTerminalVoMap = mdmTerminalVos.stream().collect(toMap(MdmTerminalVo::getTerminalCode,Function.identity()));
                if (!mdmTerminalVoMap.containsKey(reqVo.getTerminalCode())) {
                    throw new BusinessException("门店不属于所选客户");
                }
                reqVo.setTerminalName(mdmTerminalVoMap.get(reqVo.getTerminalCode()).getTerminalName());
            }
        }else{
            if(reqVo.getOrgCode() != null){
                checkOrg(reqVo);
            }
            clearCustomerInfo(reqVo);
            clearTerminalInfo(reqVo);
            clearChannelInfo(reqVo);
        }
        checkIsExist(reqVo);
    }

    private void checkOrg(TpmBudgetGenerateRuleReqVo reqVo){
        MdmOrgRespVo mdmOrgRespVo =ApiResultUtil.objResult(mdmOrgFeign.detail(reqVo.getOrgCode()));
        AssertUtils.isNotNull(mdmOrgRespVo,"组织不存在");
        MdmOrgReqVo mdmOrgReqVo = new MdmOrgReqVo();
        mdmOrgReqVo.setOrgCode(UserUtils.getUser().getOrgcode());
        List<MdmOrgRespVo> orgRespVos = ApiResultUtil.objResult(mdmOrgFeign.findAllChildrenOrgList(mdmOrgReqVo));
        Map<String ,MdmOrgRespVo> orgRespVoMap = orgRespVos.stream().collect(Collectors.toMap(MdmOrgRespVo::getOrgCode,Function.identity()));
        if (!orgRespVoMap.containsKey(reqVo.getOrgCode())){
            throw new BusinessException("只能选择自己的组织或下级组织");
        }
    }

    private void checkCustomer(TpmBudgetGenerateRuleReqVo reqVo){
        AssertUtils.isNotEmpty(reqVo.getOrgCode(), "组织不能为空");
        checkOrg(reqVo);
        MdmCustomerMsgRespVo customerMsgRespVo = ApiResultUtil.objResult(mdmCustomerMsgFeign.query("",reqVo.getCustomerCode()));
        AssertUtils.isNotNull(customerMsgRespVo,"客户不存在");
        MdmCustomerOrgSearchReqVo mdmCustomerOrgSearchReqVo  = new MdmCustomerOrgSearchReqVo();
        mdmCustomerOrgSearchReqVo.setOrgCodeList(Lists.newArrayList(reqVo.getOrgCode()));
        Map<String, MdmCustomerMsgRespVo> customerMsgRespVoMap
                = ApiResultUtil.objResult(mdmCustomerMsgFeign.findCurrentAndSubCustomerList(mdmCustomerOrgSearchReqVo)).stream()
                .collect(toMap(MdmCustomerMsgRespVo::getCustomerCode, Function.identity()));
        if (!customerMsgRespVoMap.containsKey(reqVo.getCustomerCode())){
            throw new BusinessException("客户不属于所选组织");
        }
        MdmCustomerMsgRespVo mdmCustomerMsgRespVo = customerMsgRespVoMap.get(reqVo.getCustomerCode());
        reqVo.setCustomerName(mdmCustomerMsgRespVo.getCustomerName());
    }


    private void clearCustomerInfo(TpmBudgetGenerateRuleReqVo reqVo){
        reqVo.setCustomerCode(null);
        reqVo.setCustomerName(null);
    }

    private void clearTerminalInfo(TpmBudgetGenerateRuleReqVo reqVo){
        reqVo.setTerminalCode(null);
        reqVo.setTerminalName(null);
    }

    private void clearChannelInfo(TpmBudgetGenerateRuleReqVo reqVo){
        reqVo.setChannel(null);
    }
    
    public void checkIsExist(TpmBudgetGenerateRuleReqVo reqVo){
        LambdaQueryWrapper <TpmBudgetGenerateRuleEntity> wrapper = Wrappers.lambdaQuery(TpmBudgetGenerateRuleEntity.class)
                .eq(TpmBudgetGenerateRuleEntity::getBudgetSubjectCode, reqVo.getBudgetSubjectCode())
                .eq(TpmBudgetGenerateRuleEntity::getYear, reqVo.getYear())
                .eq(TpmBudgetGenerateRuleEntity::getMonth, reqVo.getMonth())
                .eq(TpmBudgetGenerateRuleEntity::getFeeBudgetType, reqVo.getFeeBudgetType());
        if(reqVo.getOrgCode() != null){
            wrapper.eq(TpmBudgetGenerateRuleEntity::getOrgCode, reqVo.getOrgCode());
        }else {
            wrapper.isNull(TpmBudgetGenerateRuleEntity::getOrgCode);
        }
        if(reqVo.getCustomerCode() != null){
            wrapper.eq(TpmBudgetGenerateRuleEntity::getCustomerCode, reqVo.getOrgCode());
        }else {
            wrapper.isNull(TpmBudgetGenerateRuleEntity::getCustomerCode);
        }
        if(reqVo.getTerminalCode() != null){
            wrapper.eq(TpmBudgetGenerateRuleEntity::getTerminalCode, reqVo.getOrgCode());
        }else {
            wrapper.isNull(TpmBudgetGenerateRuleEntity::getTerminalCode);
        }
        if(reqVo.getId() != null){
            wrapper.ne(TpmBudgetGenerateRuleEntity::getId, reqVo.getId());
        }
        int count = tpmBudgetGenerateRuleMapper.selectCount(wrapper);
        if(count > 0){
            throw new BusinessException("已存在相同数据，不允许重复提交");
        }
    }
}
