package com.biz.crm.budgetgeneraterule.service.impl;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.mapper.BaseMapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.base.BusinessException;
import com.biz.crm.base.config.ThreadLocalUtil;
import com.biz.crm.budgetgeneraterule.mapper.TpmBudgetGenerateRuleMapper;
import com.biz.crm.budgetgeneraterule.model.TpmBudgetGenerateRuleEntity;
import com.biz.crm.budgetgeneraterule.service.ITpmBudgetGenerateRuleService;
import com.biz.crm.common.GlobalParam;
import com.biz.crm.common.PageResult;
import com.biz.crm.crmlog.handle.util.CrmLogSendUtil;
import com.biz.crm.eunm.CodeRuleEnum;
import com.biz.crm.eunm.CrmDelFlagEnum;
import com.biz.crm.eunm.CrmEnableStatusEnum;
import com.biz.crm.eunm.YesNoEnum;
import com.biz.crm.eunm.tpm.FeeBudgetDetailTypeEnum;
import com.biz.crm.feebudget.service.ITpmFeeBudgetService;
import com.biz.crm.income.mapper.TpmIncomeBudgetMapper;
import com.biz.crm.income.model.TpmIncomeBudgetEntity;
import com.biz.crm.nebular.kms.confadmin.resp.KmsDirectStoreRespVo;
import com.biz.crm.nebular.tpm.budgetgeneraterule.req.TpmBudgetGenerateRuleReqVo;
import com.biz.crm.nebular.tpm.budgetgeneraterule.resp.TpmBudgetGenerateRuleRespVo;
import com.biz.crm.nebular.tpm.feebudget.req.TpmFeeBudgetReqVo;
import com.biz.crm.util.*;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.List;

/**
 * 接口实现
 *
 * @author gavin
 * @date 2021-12-10 10:32:54
 */
@Slf4j
@Service
@ConditionalOnMissingBean(name = "TpmBudgetGenerateRuleServiceExpandImpl")
public class TpmBudgetGenerateRuleServiceImpl<M extends BaseMapper<T>, T> extends ServiceImpl<TpmBudgetGenerateRuleMapper, TpmBudgetGenerateRuleEntity> implements ITpmBudgetGenerateRuleService {

    @Resource
    private TpmBudgetGenerateRuleMapper tpmBudgetGenerateRuleMapper;

    @Resource
    private TpmBudgetGenerateRuleHelper tpmBudgetGenerateRuleHelper;

    @Resource
    private CrmLogSendUtil crmLogSendUtil;

    @Resource
    private TpmIncomeBudgetMapper tpmIncomeBudgetMapper;

    @Resource
    private ITpmFeeBudgetService tpmFeeBudgetService;

    /**
     * 列表
     *
     * @param reqVo
     * @return
     */
    @Override
    public PageResult<TpmBudgetGenerateRuleRespVo> findList(TpmBudgetGenerateRuleReqVo reqVo) {
        Page<TpmBudgetGenerateRuleRespVo> page = PageUtil.buildPage(reqVo.getPageNum(), reqVo.getPageSize());
        List<TpmBudgetGenerateRuleRespVo> list = tpmBudgetGenerateRuleMapper.findList(page, reqVo);
        tpmBudgetGenerateRuleHelper.convertListDate(list);
        return PageResult.<TpmBudgetGenerateRuleRespVo>builder()
                .data(list)
                .count(page.getTotal())
                .build();
    }

    /**
     * 查询
     *
     * @param id
     * @return tpmBudgetGenerateRuleRespVo
     */
    @Override
    public TpmBudgetGenerateRuleRespVo query(String id) {
        TpmBudgetGenerateRuleEntity entity = tpmBudgetGenerateRuleMapper.selectById(id);
        return CrmBeanUtil.copy(entity, TpmBudgetGenerateRuleRespVo.class);
    }

    /**
     * 新增
     *
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void save(TpmBudgetGenerateRuleReqVo reqVo) {
        tpmBudgetGenerateRuleHelper.checkData(reqVo);
        TpmBudgetGenerateRuleEntity entity = CrmBeanUtil.copy(reqVo, TpmBudgetGenerateRuleEntity.class);
        entity.setBudgetRuleCode(CodeUtil.generateCode(CodeRuleEnum.BUDGET_RULE_CODE.getCode()));
        StringBuilder msg = new StringBuilder();
        generateFeeBudget(entity, msg);
        this.save(entity);
        checkMsg(msg);
        Object menuCodeObj = ThreadLocalUtil.getObj(GlobalParam.MENU_CODE);
        TpmBudgetGenerateRuleRespVo newData = CrmBeanUtil.copy(entity, TpmBudgetGenerateRuleRespVo.class);
        crmLogSendUtil.sendForAdd(menuCodeObj.toString(), newData.getId(), newData.getId(), newData);
    }

    /**
     * 生成费用预算
     */
    private void generateFeeBudget(TpmBudgetGenerateRuleEntity entity, StringBuilder msg){
        //不生成费用预算直接返回
        if(entity.getIsGenerate().equals(YesNoEnum.yesNoEnum.ZERO.getValue())){
            return;
        }
        LambdaQueryWrapper<TpmIncomeBudgetEntity> wrapper = Wrappers.lambdaQuery(TpmIncomeBudgetEntity.class)
                .eq(TpmIncomeBudgetEntity::getFeeBudgetType, entity.getFeeBudgetType())
                .eq(TpmIncomeBudgetEntity::getBudgetYear, entity.getYear())
                .eq(TpmIncomeBudgetEntity::getBudgetMonth, entity.getMonth());
        if(entity.getOrgCode() != null){
            wrapper.eq(TpmIncomeBudgetEntity::getOrgCode, entity.getOrgCode());
        }
        if(entity.getCustomerCode() != null){
            wrapper.eq(TpmIncomeBudgetEntity::getCustomerCode, entity.getCustomerCode());
        }
        if(entity.getTerminalCode() != null){
            wrapper.eq(TpmIncomeBudgetEntity::getTerminalCode, entity.getTerminalCode());
        }
        if(entity.getChannel() != null){
            wrapper.eq(TpmIncomeBudgetEntity::getChannel, entity.getChannel());
        }
        //按照维度,年分,月分查找收入预算并生成相应的费用预算
        List<TpmIncomeBudgetEntity> incomeBudgetEntities = tpmIncomeBudgetMapper.selectList(wrapper);
        if(CollectionUtils.isNotEmpty(incomeBudgetEntities)){
            String userName = UserUtils.getUser().getUsername();
            incomeBudgetEntities.forEach(tpmIncomeBudgetEntity -> {
                TpmFeeBudgetReqVo feeBudgetReqVo = CrmBeanUtil.copy(tpmIncomeBudgetEntity, TpmFeeBudgetReqVo.class);
                feeBudgetReqVo.setId(null);
                feeBudgetReqVo.setBudgetQuater(String.valueOf(DateUtil.getQuaterByMonth(Integer.parseInt(feeBudgetReqVo.getBudgetMonth()))));
                feeBudgetReqVo.setInitAmount(tpmIncomeBudgetEntity.getInitAmount().multiply(entity.getRatio()).divide(new BigDecimal(100)));
                feeBudgetReqVo.setLockUserName(userName);
                feeBudgetReqVo.setFeeBudgetDetailType(FeeBudgetDetailTypeEnum.INIT.getCode());
                feeBudgetReqVo.setBudgetSubjectsCode(entity.getBudgetSubjectCode());
                feeBudgetReqVo.setBudgetSubjectsName(entity.getBudgetSubjectName());
                feeBudgetReqVo.setIsIncomeGenerate(true);
                tpmFeeBudgetService.save(feeBudgetReqVo);
            });
        }else {
            msg.append("规则:").append(entity.getBudgetRuleCode()).append("未找到对应维度收入预算;");
        }
    }

    /**
     * 更新
     *
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void update(TpmBudgetGenerateRuleReqVo reqVo) {
        tpmBudgetGenerateRuleHelper.checkData(reqVo);
        TpmBudgetGenerateRuleEntity entity = this.getById(reqVo.getId());
        if(entity.getIsGenerate().equals(YesNoEnum.yesNoEnum.ONE.getValue())){
            throw new BusinessException("该规则已生成费用预算，不允许编辑");
        }
        TpmBudgetGenerateRuleRespVo oldObject = CrmBeanUtil.copy(entity, TpmBudgetGenerateRuleRespVo.class);
        StringBuilder msg = new StringBuilder();
        CrmBeanUtil.copyProperties(reqVo, entity);
        generateFeeBudget(entity, msg);
        checkMsg(msg);
        TpmBudgetGenerateRuleRespVo newObject = CrmBeanUtil.copy(entity, TpmBudgetGenerateRuleRespVo.class);;
        this.updateById(entity);
        crmLogSendUtil.sendForUpdate(ThreadLocalUtil.getObj(GlobalParam.MENU_CODE).toString(),
               reqVo.getId(),reqVo.getId(),oldObject,newObject);
    }

    /**
     * 删除
     *
     * @param ids
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteBatch(List<String> ids) {
        List<TpmBudgetGenerateRuleEntity> tpmBudgetGenerateRuleEntities = tpmBudgetGenerateRuleMapper.selectBatchIds(ids);
        List<TpmBudgetGenerateRuleRespVo> old = new ArrayList<>();
        if (CollectionUtils.isNotEmpty(tpmBudgetGenerateRuleEntities)) {
            tpmBudgetGenerateRuleEntities.forEach(o -> {
                if (o.getIsGenerate().equals(YesNoEnum.yesNoEnum.ONE.getValue())) {
                    throw new BusinessException("所选数据已生成费用预算，不允许删除");
                }
                old.add(CrmBeanUtil.copy(o, TpmBudgetGenerateRuleRespVo.class));
            });
        }
        removeByIds(ids);
        old.forEach(o -> {
            crmLogSendUtil.sendForDel(ThreadLocalUtil.getObj(GlobalParam.MENU_CODE).toString(),
                    o.getId(),o.getId(),CrmBeanUtil.copy(o, TpmBudgetGenerateRuleRespVo.class));
        });
    }

    /**
     * 启用
     *
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void enableBatch(TpmBudgetGenerateRuleReqVo reqVo) {
        //设置状态为启用
        List<TpmBudgetGenerateRuleEntity> tpmBudgetGenerateRuleEntities = tpmBudgetGenerateRuleMapper.selectBatchIds(reqVo.getIds());
        if (CollectionUtils.isNotEmpty(tpmBudgetGenerateRuleEntities)) {
            tpmBudgetGenerateRuleEntities.forEach(o -> {
                o.setEnableStatus(CrmEnableStatusEnum.ENABLE.getCode());
            });
        }
        this.updateBatchById(tpmBudgetGenerateRuleEntities);
    }

    /**
     * 禁用
     *
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void disableBatch(TpmBudgetGenerateRuleReqVo reqVo) {
        //设置状态为禁用
        List<TpmBudgetGenerateRuleEntity> tpmBudgetGenerateRuleEntities = tpmBudgetGenerateRuleMapper.selectBatchIds(reqVo.getIds());
        if (CollectionUtils.isNotEmpty(tpmBudgetGenerateRuleEntities)) {
            tpmBudgetGenerateRuleEntities.forEach(o -> {
                o.setEnableStatus(CrmEnableStatusEnum.DISABLE.getCode());
            });
        }
        this.updateBatchById(tpmBudgetGenerateRuleEntities);
    }

    private void checkMsg(StringBuilder msg){
        if(StringUtils.isNotBlank(msg.toString())){
            throw new BusinessException(msg.toString());
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void batchGenerateFeeBudget(List<String> ids) {
        List<TpmBudgetGenerateRuleEntity> entities = tpmBudgetGenerateRuleMapper.selectBatchIds(ids);
        StringBuilder msg = new StringBuilder();
        entities.forEach(entity -> {
            if(entity.getIsGenerate().equals(YesNoEnum.yesNoEnum.ZERO.getValue())){
                entity.setIsGenerate(YesNoEnum.yesNoEnum.ONE.getValue());
                generateFeeBudget(entity, msg);
            }
        });
        updateBatchById(entities);
        checkMsg(msg);
    }
}
