package com.biz.crm.budgetsubjects.service.impl;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.biz.crm.CrmCodeRuleConstants;
import com.biz.crm.base.BaseServiceHelper;
import com.biz.crm.budgetsubjects.mapper.TpmBudgetSubjectsMapper;
import com.biz.crm.budgetsubjects.model.TpmBudgetSubjectsEntity;
import com.biz.crm.common.TpmGlobalDictConstants;
import com.biz.crm.eunm.CrmDelFlagEnum;
import com.biz.crm.eunm.CrmEnableStatusEnum;
import com.biz.crm.eunm.GlobalWhetherEnum;
import com.biz.crm.exception.tpm.BudgetSubjectsException;
import com.biz.crm.exception.tpm.CostTypeCategoriesException;
import com.biz.crm.nebular.tpm.budgetsubjects.req.TpmBudgetSubjectsReqVo;
import com.biz.crm.nebular.tpm.budgetsubjects.resp.TpmBudgetSubjectsRespVo;
import com.biz.crm.util.AssertUtils;
import com.biz.crm.util.CodeUtil;
import com.biz.crm.util.DateUtil;
import com.biz.crm.util.DictUtil;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.rocketmq.common.filter.impl.Op;
import org.springframework.stereotype.Service;
import javax.annotation.Resource;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * @Project crm
 * @ClassName BudgetSubjectsServiceHelper
 * @Author HuangLong
 * @Date 2020/9/8 14:02
 * @Description 预算科目工具类
 */
@Slf4j
@Service
public class BudgetSubjectsServiceHelper extends BaseServiceHelper {

    @Resource
    private TpmBudgetSubjectsMapper mapper;

    public Map<String, Map<String, String>> getDicts(){
        List<String> dictCodes= Lists.newArrayList();
        dictCodes.add(TpmGlobalDictConstants.ENABLE_STATUS);
        dictCodes.add(TpmGlobalDictConstants.CONTROL_TYPE);
        dictCodes.add(TpmGlobalDictConstants.SUBJECTS_TYPE);
        dictCodes.add(TpmGlobalDictConstants.SUBJECTS_GROUP);
        dictCodes.add(TpmGlobalDictConstants.IS_ROLL);
        dictCodes.add(TpmGlobalDictConstants.IS_ACT_BUDGET);
        //查询字典数据
        return DictUtil.getDictValueMapsByCodes(dictCodes);
    }

    /**
     * 转换列表数据
     * @param list
     */
    public void convertListDate(List<TpmBudgetSubjectsRespVo> list) {
        if(CollectionUtils.isNotEmpty(list)){
            //获取需要的字典集合
            Map<String, Map<String, String>> map = this.getDicts();
            list.forEach(o->{
                //启用禁用状态
                if(StringUtils.isNotEmpty(o.getEnableStatus())){
                    o.setEnableStatusName(Optional.ofNullable(map.get(TpmGlobalDictConstants.ENABLE_STATUS)).orElse(Maps.newHashMap()).get(o.getEnableStatus()));
                }
                if(StringUtils.isNotEmpty(o.getSubjectsGroupCode())){
                    o.setSubjectsGroupName(Optional.ofNullable(map.get(TpmGlobalDictConstants.SUBJECTS_GROUP)).orElse(Maps.newHashMap()).get(o.getSubjectsGroupCode()));
                }
                if(StringUtils.isNotEmpty(o.getSubjectsTypeCode())){
                    o.setSubjectsTypeName(Optional.ofNullable(map.get(TpmGlobalDictConstants.SUBJECTS_TYPE)).orElse(Maps.newHashMap()).get(o.getSubjectsTypeCode()));
                }
                if(StringUtils.isNotEmpty(o.getIsRoll())){
                    o.setIsRollName(Optional.ofNullable(map.get(TpmGlobalDictConstants.IS_ROLL)).orElse(Maps.newHashMap()).get(o.getIsRoll()));
                }
                if(StringUtils.isNotEmpty(o.getControlType())){
                    o.setControlTypeName(Optional.ofNullable(map.get(TpmGlobalDictConstants.CONTROL_TYPE)).orElse(Maps.newHashMap()).get(o.getControlType()));
                }
                if (StringUtils.isNotEmpty(o.getIsActBudget())){
                    o.setIsActBudgetDesc(Optional.ofNullable(map.get(TpmGlobalDictConstants.IS_ACT_BUDGET)).orElse(Maps.newHashMap()).get(o.getIsActBudget()));
                }
            });
        }
    }
    /**
     * 通过查询条件获取wrapper
     * @return
     */
    public LambdaQueryWrapper<TpmBudgetSubjectsEntity> getQueryWrapper(TpmBudgetSubjectsReqVo reqVo){
        LambdaQueryWrapper<TpmBudgetSubjectsEntity> wrapper = new LambdaQueryWrapper<>();
        //科目名称
        wrapper.like(StringUtils.isNotEmpty(reqVo.getBudgetSubjectsName()),TpmBudgetSubjectsEntity::getBudgetSubjectsName,reqVo.getBudgetSubjectsName());
        //科目编码
        wrapper.like(StringUtils.isNotEmpty(reqVo.getBudgetSubjectsCode()),TpmBudgetSubjectsEntity::getBudgetSubjectsCode,reqVo.getBudgetSubjectsCode());
        //科目类型
        wrapper.eq(StringUtils.isNotEmpty(reqVo.getSubjectsTypeCode()),TpmBudgetSubjectsEntity::getSubjectsTypeCode,reqVo.getSubjectsTypeCode());
        //科目分组
        wrapper.eq(StringUtils.isNotEmpty(reqVo.getSubjectsGroupCode()),TpmBudgetSubjectsEntity::getSubjectsGroupCode,reqVo.getSubjectsGroupCode());
        //控制类型
        wrapper.eq(StringUtils.isNotEmpty(reqVo.getControlType()),TpmBudgetSubjectsEntity::getControlType,reqVo.getControlType());
        //启用禁用
        wrapper.eq(StringUtils.isNotEmpty(reqVo.getEnableStatus()),TpmBudgetSubjectsEntity::getEnableStatus,reqVo.getEnableStatus());
        return wrapper;
    }
    /**
     * 修改财年数据(启用,禁用)
     * @param reqVo
     * @param entity
     */
    public void updateBySelectAll(TpmBudgetSubjectsReqVo reqVo,TpmBudgetSubjectsEntity entity){
        LambdaQueryWrapper<TpmBudgetSubjectsEntity> queryWrapper = this.getQueryWrapper(reqVo);
        //TODO  这里必须加上数据权限的限制!!!!
        mapper.update(entity, queryWrapper);
    }
    /**
     * 新增,编辑,校验
     * @param reqVo
     */
    public void saveCheck(TpmBudgetSubjectsReqVo reqVo){
        //校验数据不为空
        AssertUtils.isNotEmpty(reqVo.getControlType(),"请选择控制类型");
        if(StringUtils.isEmpty(reqVo.getBudgetSubjectsCode())){
            reqVo.setBudgetSubjectsCode(CodeUtil.createOneCode(CrmCodeRuleConstants.BUDGET_SUBJECTS));
        }
        if(StringUtils.isEmpty(reqVo.getEnableStatus())){
            reqVo.setEnableStatus(CrmEnableStatusEnum.ENABLE.getCode());
        }
        AssertUtils.isNotEmpty(reqVo.getBudgetSubjectsName(),"请输入预算科目名称");
        // 设置是否为活动预算 默认为否
        if (StringUtils.isBlank(reqVo.getIsActBudget())){
            reqVo.setIsActBudget(GlobalWhetherEnum.YES.getCode());
        }
        if (StringUtils.isNotBlank(reqVo.getParentSubjectsCode())){
            AssertUtils.isTrue(!StringUtils.equals(reqVo.getParentSubjectsCode(),reqVo.getBudgetSubjectsCode()),"上级预算科目不能与当前预算科目相同");
            TpmBudgetSubjectsEntity parentSubjects = mapper.selectOne(Wrappers.lambdaQuery(TpmBudgetSubjectsEntity.class).eq(TpmBudgetSubjectsEntity::getBudgetSubjectsCode, reqVo.getParentSubjectsCode()));
            AssertUtils.isNotNull(parentSubjects,"你选择的上级预算科目不存在");
        }
        this.checkDataExist(reqVo);
    }

    /**
     * 校验数据是否已经存在
     * @param reqVo
     * @return
     */
    public void checkDataExist(TpmBudgetSubjectsReqVo reqVo ) {
        LambdaQueryWrapper<TpmBudgetSubjectsEntity> wrapper = new LambdaQueryWrapper<TpmBudgetSubjectsEntity>().eq(TpmBudgetSubjectsEntity::getBudgetSubjectsCode,reqVo.getBudgetSubjectsCode());
        if(StringUtils.isNotEmpty(reqVo.getId())){
            wrapper.ne(TpmBudgetSubjectsEntity::getId,reqVo.getId());
        }
        Integer count = mapper.selectCount(wrapper);
        AssertUtils.isTrue(count==0, BudgetSubjectsException.DATA_DUPLICATE);
    }

}
