package com.biz.crm.budgetsubjects.service.impl;

import com.baomidou.mybatisplus.core.mapper.BaseMapper;
import com.baomidou.mybatisplus.core.toolkit.ObjectUtils;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.aop.CrmLog;
import com.biz.crm.base.ApiResultUtil;
import com.biz.crm.base.BusinessException;
import com.biz.crm.base.config.ThreadLocalUtil;
import com.biz.crm.budgetsubjects.mapper.TpmBudgetSubjectsMapper;
import com.biz.crm.budgetsubjects.model.TpmBudgetSubjectsEntity;
import com.biz.crm.budgetsubjects.service.ITpmBudgetSubjectsService;
import com.biz.crm.common.GlobalParam;
import com.biz.crm.common.PageResult;
import com.biz.crm.crmlog.handle.util.CrmLogSendUtil;
import com.biz.crm.eunm.CrmEnableStatusEnum;
import com.biz.crm.exception.CommonException;
import com.biz.crm.exception.tpm.BudgetSubjectsException;
import com.biz.crm.nebular.tpm.budgetsubjects.req.TpmBudgetSubjectsReqVo;
import com.biz.crm.nebular.tpm.budgetsubjects.resp.TpmBudgetSubjectsRespVo;
import com.biz.crm.util.AssertUtils;
import com.biz.crm.util.CrmBeanUtil;
import com.biz.crm.util.PageUtil;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.sql.Wrapper;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * 预算科目 接口实现
 *
 * @author huang.long
 * @date 2020-09-09 13:40:31
 */
@Slf4j
@Service
@ConditionalOnMissingBean(name = "TpmBudgetSubjectsServiceExpandImpl")
public class TpmBudgetSubjectsServiceImpl<M extends BaseMapper<T>, T> extends ServiceImpl<TpmBudgetSubjectsMapper, TpmBudgetSubjectsEntity> implements ITpmBudgetSubjectsService {

    @Resource
    private TpmBudgetSubjectsMapper tpmBudgetSubjectsMapper;
    @Autowired
    private BudgetSubjectsServiceHelper serviceHelper;
    @Autowired
    private CrmLogSendUtil crmLogSendUtil;

    /**
     * 列表
     *
     * @param tpmBudgetSubjectsReqVo
     * @return
     */
    @Override
    @CrmLog
    public PageResult<TpmBudgetSubjectsRespVo> findList(TpmBudgetSubjectsReqVo tpmBudgetSubjectsReqVo) {
        Page<TpmBudgetSubjectsRespVo> page = PageUtil.buildPage(tpmBudgetSubjectsReqVo.getPageNum(), tpmBudgetSubjectsReqVo.getPageSize());
        tpmBudgetSubjectsReqVo.setSelectedCodeList(serviceHelper.dealSelectedCodeList(tpmBudgetSubjectsReqVo.getSelectedCode(),tpmBudgetSubjectsReqVo.getSelectedCodeList()));
        List<TpmBudgetSubjectsRespVo> list = tpmBudgetSubjectsMapper.findList(page, tpmBudgetSubjectsReqVo);
        serviceHelper.convertListDate(list);
        return PageResult.<TpmBudgetSubjectsRespVo>builder()
                .data(list)
                .count(page.getTotal())
                .build();
    }

    /**
     * 查询
     *
     * @param id
     * @return tpmBudgetSubjectsRespVo
     */
    @Override
    @CrmLog
    public TpmBudgetSubjectsRespVo query(String id) {
        AssertUtils.isNotEmpty(id, "id不能为空");
        TpmBudgetSubjectsReqVo tpmBudgetSubjectsReqVo = new TpmBudgetSubjectsReqVo();
        tpmBudgetSubjectsReqVo.setId(id);
        List<TpmBudgetSubjectsRespVo> list = this.findList(tpmBudgetSubjectsReqVo).getData();
        if (CollectionUtils.isEmpty(list)) {
//            throw new BusinessException(BudgetSubjectsException.DATA_NOT_EXIST);
            return new TpmBudgetSubjectsRespVo();
        }
        return list.get(0);
    }

    /**
     * 新增
     *
     * @param tpmBudgetSubjectsReqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    @CrmLog
    public void save(TpmBudgetSubjectsReqVo tpmBudgetSubjectsReqVo) {
        serviceHelper.saveCheck(tpmBudgetSubjectsReqVo);
        TpmBudgetSubjectsEntity entity = CrmBeanUtil.copy(tpmBudgetSubjectsReqVo, TpmBudgetSubjectsEntity.class);
        this.save(entity);
        //日志
        //获取菜单码
        Object menuCodeObj = ThreadLocalUtil.getObj(GlobalParam.MENU_CODE);
        TpmBudgetSubjectsRespVo newData = CrmBeanUtil.copy(entity, TpmBudgetSubjectsRespVo.class);
        //发送添加日志
        crmLogSendUtil.sendForAdd(menuCodeObj.toString(),newData.getId(),newData.getBudgetSubjectsCode(),newData);

    }

    /**
     * 更新
     *
     * @param tpmBudgetSubjectsReqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    @CrmLog
    public void update(TpmBudgetSubjectsReqVo tpmBudgetSubjectsReqVo) {
        if (StringUtils.isEmpty(tpmBudgetSubjectsReqVo.getId())) {
            throw new BusinessException("数据主键不能为空");
        }
        TpmBudgetSubjectsEntity entity = this.getById(tpmBudgetSubjectsReqVo.getId());
        //修改前的对象
        TpmBudgetSubjectsRespVo oldData = CrmBeanUtil.copy(entity,TpmBudgetSubjectsRespVo.class);
        if (Objects.isNull(entity)) {
            throw new BusinessException(BudgetSubjectsException.DATA_NOT_EXIST);
        }
        serviceHelper.saveCheck(tpmBudgetSubjectsReqVo);
        CrmBeanUtil.copyProperties(tpmBudgetSubjectsReqVo, entity);
        //进行修改操作
        this.updateById(entity);
        //获取菜单码
        Object menuCodeObj = ThreadLocalUtil.getObj(GlobalParam.MENU_CODE);
        TpmBudgetSubjectsRespVo newData = CrmBeanUtil.copy(entity,TpmBudgetSubjectsRespVo.class);
        //发送修改日志
        crmLogSendUtil.sendForUpdate(menuCodeObj.toString(),newData.getId(),newData.getBudgetSubjectsCode(),oldData,newData);
    }

    /**
     * 删除
     *
     * @param ids id集合
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    @CrmLog
    public void deleteBatch(List<String> ids) {
        AssertUtils.isNotEmpty(ids, CommonException.IDS_NULL);
        //预算科目删除时,需要判断预算科目有没有关联的费用预算
        List<TpmBudgetSubjectsRespVo> list = tpmBudgetSubjectsMapper.countRelatedNum(ids);
        AssertUtils.isNotEmpty(list, "您选择的预算科目已被删除");
        //如有关联的下级科目，则不允许删除（需先解除上下级关系）
        Set<String> budgetSubjectsCodes = Sets.newHashSet();
        list.forEach(o -> {
            AssertUtils.isTrue(o.getNum() == 0, "预算科目:" + o.getBudgetSubjectsCode() + "已被费用预算关联,不能删除");
            budgetSubjectsCodes.add(o.getBudgetSubjectsCode());
        });
        List<TpmBudgetSubjectsEntity> tpmBudgetSubjectsEntityList = tpmBudgetSubjectsMapper.selectList(Wrappers.lambdaQuery(TpmBudgetSubjectsEntity.class).in(TpmBudgetSubjectsEntity::getParentSubjectsCode, budgetSubjectsCodes));
        //存在下级科目的情况
        if (CollectionUtils.isNotEmpty(tpmBudgetSubjectsEntityList)){
            String parentSubjectsCodes = "";
            for(int i=0;i<tpmBudgetSubjectsEntityList.size();i++){
                TpmBudgetSubjectsEntity budgetSubjectsEntity = tpmBudgetSubjectsEntityList.get(i);
                if (!parentSubjectsCodes.contains(budgetSubjectsEntity.getParentSubjectsCode())){
                    if (StringUtils.isNotEmpty(parentSubjectsCodes)){
                        parentSubjectsCodes = parentSubjectsCodes +"、"+ budgetSubjectsEntity.getParentSubjectsCode();
                    }else {
                        parentSubjectsCodes = budgetSubjectsEntity.getParentSubjectsCode();
                    }
                }
            }
            AssertUtils.isTrue(false,"科目编码："+parentSubjectsCodes+"有关联的下级科目，不允许删除（需先解除上下级关系）");
        }

        //删除前数据
        PageResult<TpmBudgetSubjectsRespVo> oldDateList = this.findList(new TpmBudgetSubjectsReqVo().setIds(ids));
        tpmBudgetSubjectsMapper.deleteBatchIds(ids);
        //获取菜单码
        Object menuCodeObj = ThreadLocalUtil.getObj(GlobalParam.MENU_CODE);
        //发送删除日志
        for (TpmBudgetSubjectsRespVo oldData:oldDateList.getData()) {
            crmLogSendUtil.sendForDel(menuCodeObj.toString(),oldData.getId(),oldData.getBudgetSubjectsCode(),oldData);
        }
    }

    /**
     * 启用
     *
     * @param ids id集合
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    @CrmLog
    public void enableBatch(List<String> ids) {
        //设置状态为启用
        AssertUtils.isNotEmpty(ids, CommonException.IDS_NULL);
        List<TpmBudgetSubjectsEntity> entities = tpmBudgetSubjectsMapper.selectBatchIds(ids);
        //日志
        List<TpmBudgetSubjectsRespVo> oldDataList = CrmBeanUtil.copyList(entities, TpmBudgetSubjectsRespVo.class);
        if (CollectionUtils.isNotEmpty(entities)) {
            entities.forEach(o -> {
                o.setEnableStatus(CrmEnableStatusEnum.ENABLE.getCode());
            });
        }
        this.updateBatchById(entities);
        //获取菜单码
        Object menuCodeObj = ThreadLocalUtil.getObj(GlobalParam.MENU_CODE);
        List<TpmBudgetSubjectsRespVo> newDataList = CrmBeanUtil.copyList(entities, TpmBudgetSubjectsRespVo.class);
        for (int i = 0;i < newDataList.size();i++) {
            crmLogSendUtil.sendForUpdate(menuCodeObj.toString(),newDataList.get(i).getId(),newDataList.get(i).getBudgetSubjectsCode(),oldDataList.get(i),newDataList.get(i));
        }


    }

    /**
     * 禁用
     *
     * @param ids id集合
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    @CrmLog
    public void disableBatch(List<String> ids) {
        //设置状态为禁用
        AssertUtils.isNotEmpty(ids, CommonException.IDS_NULL);
        List<TpmBudgetSubjectsEntity> entities = tpmBudgetSubjectsMapper.selectBatchIds(ids);
        //日志老数据
        List<TpmBudgetSubjectsRespVo> oldDataList = CrmBeanUtil.copyList(entities, TpmBudgetSubjectsRespVo.class);
        if (CollectionUtils.isNotEmpty(entities)) {
            entities.forEach(o -> {
                o.setEnableStatus(CrmEnableStatusEnum.DISABLE.getCode());
            });
        }
        this.updateBatchById(entities);
        //日志
        List<TpmBudgetSubjectsRespVo> newDataList = CrmBeanUtil.copyList(entities, TpmBudgetSubjectsRespVo.class);
        Object menuCodeObj = ThreadLocalUtil.getObj(GlobalParam.MENU_CODE);
        for (int i = 0;i < newDataList.size();i++) {
            crmLogSendUtil.sendForUpdate(menuCodeObj.toString(),newDataList.get(i).getId(),newDataList.get(i).getBudgetSubjectsCode(),oldDataList.get(i),newDataList.get(i));
        }

    }

    @Override
    public List<TpmBudgetSubjectsRespVo> findAllBudgets(TpmBudgetSubjectsReqVo reqVo) {
        return this.findList(reqVo).getData();
    }
}
