package com.biz.crm.invoicepool.service.impl;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.biz.crm.base.BusinessException;
import com.biz.crm.eunm.tpm.InvoiceTypeEnum;
import com.biz.crm.invoicepool.mapper.TpmInvoicePoolMapper;
import com.biz.crm.invoicepool.model.TpmInvoicePoolEntity;
import com.biz.crm.nebular.tpm.audit.req.TpmAuditInvoiceReqVo;
import com.biz.crm.nebular.tpm.invoicepool.req.TpmInvoicePoolReqVo;
import com.biz.crm.util.AssertUtils;
import com.biz.crm.util.NumberUtil;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.math.BigDecimal;

/**
 * @author yb
 * @version 1.0
 * @date 2021/12/21 14:34
 * @description
 */
@Service
public class TpmInvoiceHelper {

    @Resource
    private TpmInvoicePoolMapper tpmInvoicePoolMapper;


    public void checkInvoice(TpmInvoicePoolReqVo o){
        AssertUtils.isNotEmpty(o.getInvoiceType(), "发票类型不能为空");
        AssertUtils.isNotEmpty(o.getInvoiceCode(), "发票代码不能为空");
        AssertUtils.isNotEmpty(o.getInvoiceNumber(), "发票号码不能为空");
        NumberUtil.checkNumber(o.getTotalAmount(), "税价合计", BigDecimal.ZERO, null);
        if(InvoiceTypeEnum.SPECIAL_VAT.getCode().equals(o.getInvoiceType()) || InvoiceTypeEnum.ORDINARY_VAT.getCode().equals(o.getInvoiceType())){
            NumberUtil.checkNumber(o.getAmount(), "不含税金额", BigDecimal.ZERO, null);
            NumberUtil.checkNumber(o.getTaxRate(), "税率", BigDecimal.ZERO, null);
            NumberUtil.checkNumber(o.getTaxAmount(), "税额", BigDecimal.ZERO, null);
            AssertUtils.isTrue(o.getTotalAmount().multiply(o.getTaxRate()).divide(new BigDecimal(100)).compareTo(o.getTaxAmount()) == 0, "税额应等于税率乘税价合计");
            AssertUtils.isTrue(o.getTotalAmount().compareTo(o.getAmount().add(o.getTaxAmount())) == 0, "不含金额加税额应等于税价合计");
            AssertUtils.isNotEmpty(o.getPurchaserName(), "购买方名称不能为空");
            AssertUtils.isNotEmpty(o.getPurchaserTaxpayerNumber(), "购买方税号不能为空");
            AssertUtils.isNotEmpty(o.getSellerName(), "销售方名称不能为空");
            AssertUtils.isNotEmpty(o.getSellerTaxpayerNumber(), "销售方税号不能为空");
            AssertUtils.isNotEmpty(o.getSellerAccount(), "销售方开行号及账户不能为空");
            AssertUtils.isNotEmpty(o.getSellerInfo(), "销售方地址电话不能为空");
            if(InvoiceTypeEnum.SPECIAL_VAT.getCode().equals(o.getInvoiceType())){
                AssertUtils.isNotEmpty(o.getPurchaserAccount(), "购买方开行号及账户不能为空");
                AssertUtils.isNotEmpty(o.getPurchaserInfo(), "购买方地址电话不能为空");
            }
        }else if(!InvoiceTypeEnum.STABLE.getCode().equals(o.getInvoiceType())) {
            throw new BusinessException("未找到的发票类型");
        }
    }


    public void checkInvoicePool(TpmInvoicePoolReqVo o){
        checkInvoice(o);
        LambdaQueryWrapper<TpmInvoicePoolEntity> wrapper = Wrappers.lambdaQuery(TpmInvoicePoolEntity.class)
                .eq(TpmInvoicePoolEntity::getInvoiceNumber, o.getInvoiceNumber());
        if(o.getId() != null){
            wrapper.ne(TpmInvoicePoolEntity::getId, o.getId());
        }
        if(tpmInvoicePoolMapper.selectOne(wrapper) != null){
            throw new BusinessException("已存在相同发票号码的发票,请勿重复上传");
        }
    }

    public void checkAuditInvoice(TpmAuditInvoiceReqVo o){
        NumberUtil.checkNumber(o.getUsedAmount(), "已使用金额", BigDecimal.ZERO, null);
        NumberUtil.checkNumber(o.getAvailableAmount(), "可用余额", BigDecimal.ZERO, null);
        AssertUtils.isTrue(o.getTotalAmount().compareTo(o.getUsedAmount().add(o.getAvailableAmount())) == 0, "已使用金额加可用余额应等于税价合计");
        NumberUtil.checkNumber(o.getUseAmount(), "本次使用金额", BigDecimal.ZERO, null);
        checkInvoice(o);
        TpmInvoicePoolEntity check = tpmInvoicePoolMapper.selectOne(Wrappers.lambdaQuery(TpmInvoicePoolEntity.class).eq(TpmInvoicePoolEntity::getInvoiceNumber, o.getInvoiceNumber()));
        BigDecimal availableAmount = o.getTotalAmount();
        if(check != null ){
            if(check.getAvailableAmount().compareTo(BigDecimal.ZERO) <= 0) {
                throw new BusinessException("发票" + o.getInvoiceNumber() + "可用余额为0,请重新上传");
            }
            availableAmount = check.getAvailableAmount();
        }
        if(o.getUseAmount().compareTo(availableAmount) > 0){
            throw new BusinessException("发票" + o.getInvoiceNumber() + "使用金额大于可用余额");
        }
    }
}
