package com.biz.crm.audit.service.impl;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.biz.crm.CrmCodeRuleConstants;
import com.biz.crm.act.mapper.TpmActDetailMapper;
import com.biz.crm.act.mapper.TpmActMapper;
import com.biz.crm.act.model.TpmActEntity;
import com.biz.crm.act.service.impl.ActServiceHelper;
import com.biz.crm.audit.mapper.TpmAuditMapper;
import com.biz.crm.audit.model.TpmAuditEntity;
import com.biz.crm.base.BaseServiceHelper;
import com.biz.crm.base.BusinessException;
import com.biz.crm.common.TpmGlobalDictConstants;
import com.biz.crm.costtypefine.mapper.TpmCostTypeFineMapper;
import com.biz.crm.costtypefine.model.TpmCostTypeFineEntity;
import com.biz.crm.eunm.GlobalWhetherEnum;
import com.biz.crm.eunm.tpm.*;
import com.biz.crm.exception.tpm.AuditException;
import com.biz.crm.invoicepool.mapper.TpmInvoicePoolMapper;
import com.biz.crm.invoicepool.service.impl.TpmInvoiceHelper;
import com.biz.crm.nebular.activiti.act.req.StartProcessReqVo;
import com.biz.crm.nebular.activiti.act.req.TaActFileReqVo;
import com.biz.crm.nebular.tpm.act.req.TpmActDetailReqVo;
import com.biz.crm.nebular.tpm.act.resp.TpmActDetailRespVo;
import com.biz.crm.nebular.tpm.audit.TpmAuditDetailReplenishmentProductVo;
import com.biz.crm.nebular.tpm.audit.req.*;
import com.biz.crm.nebular.tpm.audit.resp.TpmAuditFileRespVo;
import com.biz.crm.nebular.tpm.audit.resp.TpmAuditRespVo;
import com.biz.crm.util.*;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import lombok.extern.slf4j.Slf4j;
import oracle.net.aso.o;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.math.BigDecimal;
import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * @Project crm
 * @ClassName FeeBudgetServiceHelper
 * @Author HuangLong
 * @Date 2020/9/8 14:02
 * @Description 核销申请工具类
 */
@Slf4j
@Service
public class AuditServiceHelper extends BaseServiceHelper {

    @Resource
    private TpmAuditMapper auditMapper;

    @Resource
    private TpmActMapper tpmActMapper;

    @Resource
    private TpmActDetailMapper tpmActDetailMapper;

    @Resource
    private TpmCostTypeFineMapper tpmCostTypeFineMapper;

    @Resource
    private TpmInvoicePoolMapper tpmInvoicePoolMapper;

    @Autowired
    private ActServiceHelper helper;

    @Autowired
    private TpmInvoiceHelper invoiceHelper;

    /**
     * 转换列表数据
     * @param list
     */
    public void convertListDate(List<TpmAuditRespVo> list) {
        if(CollectionUtils.isNotEmpty(list)){
            list.forEach(o->{
                //审批状态
                if(StringUtils.isNotEmpty(o.getApproveStatus())){
                    o.setApproveStatusName(ActAdvancePayApproveStatusEnum.getStatusName(o.getApproveStatus()));
                }
            });
        }
    }

    /**
     * 删除检查
     * @param ids
     */
    public void deleteCheck(List<TpmAuditEntity> tpmAuditEntities,List<String> ids){
        tpmAuditEntities.forEach(tpmAuditEntity -> {
            AssertUtils.isTrue(
            ActApproveStatusEnum.CREATE.getCode().equals(tpmAuditEntity.getApproveStatus())||
                    ActApproveStatusEnum.REJECTED.getCode().equals(tpmAuditEntity.getApproveStatus())||
                    ActApproveStatusEnum.INTERRUPT.getCode().equals(tpmAuditEntity.getApproveStatus()
                    ),"已提交数据不能删除");
        });
    }

    /**
     * 新增,编辑,校验
     * @param reqVo
     */
    public void saveCheck(TpmAuditReqVo reqVo){
        //校验主表数据
        if(StringUtils.isNotEmpty(reqVo.getId())){
            TpmAuditEntity tpmAuditEntity = auditMapper.selectById(reqVo.getId());
            AssertUtils.isNotNull(tpmAuditEntity,AuditException.DATA_NOT_EXIST);
            if(!ActApproveStatusEnum.getCanUpdateStatus().contains(tpmAuditEntity.getApproveStatus())){
                throw new BusinessException("当前数据状态不允许修改操作");
            }
            reqVo.setAuditCode(tpmAuditEntity.getAuditCode());
        }else {
            reqVo.setAuditCode(CodeUtil.createOneCode(CrmCodeRuleConstants.AUDIT));
        }
        AssertUtils.isNotEmpty(reqVo.getAuditName(),"核销申请名称不能为空");
        AssertUtils.isNotEmpty(reqVo.getSaveType(),"保存类型不能为空");
        if(ActSaveTypeEnum.getSendApproveTypes().contains(reqVo.getSaveType())){
            this.approvingCheck(reqVo);
            reqVo.setApproveStatus(ActApproveStatusEnum.APPROVING.getCode());
        }else {
            //审批驳回的数据编辑时,如果不提交审批,状态保持不变
            if(StringUtils.isEmpty(reqVo.getApproveStatus())){
                reqVo.setApproveStatus(ActApproveStatusEnum.CREATE.getCode());
            }
        }
        //校验活动信息
        this.saveActCheckDetails(reqVo);
        //校验明细信息
        this.saveCheckDetails(reqVo);
        //校验核销资料数据
        this.saveCheckAuditFiles(reqVo);
        //核销发票数据
        this.saveCheckInvoice(reqVo);
        //校验编码的重复
        this.checkDataExist(reqVo);
        //校验货补产品
        this.checkProductInfo(reqVo);
    }

    /**
     * 校验核销资料
     * @param reqVo
     */
    private void saveCheckAuditFiles(TpmAuditReqVo reqVo) {
        List<TpmAuditFileRespVo> auditFileRespVos = reqVo.getAuditFileRespVos();
        Set<String> fineCodes = reqVo.getDetailVos().stream().map(TpmAuditDetailReqVo::getFineCode).collect(Collectors.toSet());
        if(CollectionUtils.isNotEmpty(auditFileRespVos)){
            auditFileRespVos.forEach(o->{
                AssertUtils.isNotNull(o.getAuditFileType(),"核销资料类型不能为空");
                o.setAuditCode(reqVo.getAuditCode());
                if(Objects.equals(1,o.getAuditFileType())){
                    AssertUtils.isNotEmpty(o.getExampleCode(),"核销资料的核销采集示例编码不能为空");
                    AssertUtils.isNotEmpty(o.getFineCode(),"核销资料的细类编码不能为空");
                    AssertUtils.isTrue(fineCodes.contains(o.getFineCode()),"核销资料对应的细类编码"+o.getFineCode()+",不在核销明细中");
                }
                AssertUtils.isNotEmpty(o.getAddressUrl(),"核销资料的文件地址不能为空");
            });
        }
    }

    private void saveCheckInvoice(TpmAuditReqVo reqVo){
        List<TpmAuditInvoiceReqVo> auditInvoiceReqVos = reqVo.getAuditInvoiceReqVos();
        if(CollectionUtil.listNotEmpty(auditInvoiceReqVos)){
            auditInvoiceReqVos.forEach(o -> {
                invoiceHelper.checkAuditInvoice(o);
                o.setAuditCode(reqVo.getAuditCode());
            });
        }
    }

    /**
     * 保存时校验选择的活动数据
     * @param reqVo
     */
    private void saveActCheckDetails(TpmAuditReqVo reqVo) {
        //校验活动信息
        List<TpmAuditActReqVo> tpmAuditActReqVos = reqVo.getTpmAuditActReqVos();
        AssertUtils.isNotEmpty(tpmAuditActReqVos,"核销申请活动信息不能为空");
        Set<String> actCodes = tpmAuditActReqVos.stream().map(TpmAuditActReqVo::getActCode).collect(Collectors.toSet());
        AssertUtils.isTrue(tpmAuditActReqVos.size()==CollectionUtils.size(actCodes),"请不要选择重复的活动");
        List<TpmActEntity> tpmActEntities = tpmActMapper.selectList(Wrappers.<TpmActEntity>lambdaQuery().in(TpmActEntity::getActCode, actCodes));
        Map<String, TpmActEntity> map = tpmActEntities.stream().collect(Collectors.toMap(TpmActEntity::getActCode, Function.identity()));
        AssertUtils.isTrue(CollectionUtils.size(tpmActEntities) == tpmAuditActReqVos.size(),"活动数据异常");
        tpmAuditActReqVos.forEach(tpmAuditActReqVo -> {
            TpmActEntity entity = map.get(tpmAuditActReqVo.getActCode());
            AssertUtils.isNotNull(entity,"活动编码:"+tpmAuditActReqVo.getActCode()+"数据不存在");
            AssertUtils.isNotEmpty(tpmAuditActReqVo.getActName(),"活动名称不能为空");
            AssertUtils.isTrue(StringUtils.equals(ActApproveStatusEnum.APPROVED.getCode(),entity.getApproveStatus()),"审批未通过的活动:"+entity.getActCode()+"不能核销");
            AssertUtils.isTrue(StringUtils.equals(GlobalWhetherEnum.NO.getCode(),entity.getIsAllAudit()),"完成核销的活动不能再次核销");
            tpmAuditActReqVo.setAuditCode(reqVo.getAuditCode());
        });
    }

    /**
     * 保存时校验核销明细数据
     * @param reqVo
     */
    private void saveCheckDetails(TpmAuditReqVo reqVo) {
        //校验明细表数据
        List<TpmAuditDetailReqVo> detailVos = reqVo.getDetailVos();
        Set<String> actCodes = reqVo.getTpmAuditActReqVos().stream().map(TpmAuditActReqVo::getActCode).collect(Collectors.toSet());
        AssertUtils.isNotEmpty(detailVos,"核销申请活动明细信息不能为空");
        Set<String> actDetailCodes = Sets.newHashSet();
        Set<String> fineCodes= Sets.newHashSet();
        //获取支付方式类型map
        Map<String,String> payTypeTypeMap = helper.getPayTypeTypeMap(TpmGlobalDictConstants.PAY_TYPE,TpmGlobalDictConstants.PAY_TYPE_TYPE);
        detailVos.stream().forEach(o->{
            AssertUtils.isNotEmpty(o.getActDetailCode(),"活动明细编码不能为空");
            AssertUtils.isNotEmpty(o.getFineCode(),"细类编码不能为空");
            AssertUtils.isTrue(actCodes.contains(o.getActCode()),"活动明细编码:"+o.getActDetailCode()+"不在选定的核销活动范围内");
            actDetailCodes.add(o.getActDetailCode());
            fineCodes.add(o.getFineCode());
            if(StringUtils.isNotEmpty(o.getPayType())){
                o.setPayTypeType(payTypeTypeMap.get(o.getPayType()));
            }
        });
        AssertUtils.isTrue(detailVos.size()==CollectionUtils.size(actDetailCodes),"请不要选择重复的活动明细");
        List<TpmCostTypeFineEntity> fineEntities = tpmCostTypeFineMapper.selectList(Wrappers.<TpmCostTypeFineEntity>lambdaQuery()
                .in(TpmCostTypeFineEntity::getFineCode, fineCodes));
        AssertUtils.isNotEmpty(fineEntities,"活动细类数据异常");
        Map<String, TpmCostTypeFineEntity> fineEntityMap = fineEntities.stream().collect(Collectors.toMap(TpmCostTypeFineEntity::getFineCode, Function.identity()));
        TpmActDetailReqVo tpmActDetailReqVo = new TpmActDetailReqVo();
        tpmActDetailReqVo.setActCodes(Lists.newArrayList(actCodes));
        tpmActDetailReqVo.setIsExpiration(false);
        List<TpmActDetailRespVo> actDetailList = tpmActDetailMapper.findActDetailList(PageUtil.buildPage(reqVo.getPageNum(), -1), tpmActDetailReqVo);
        AssertUtils.isNotEmpty(actDetailList,"您选择的活动明细数据发生变更,请重新选择数据");
        Map<String, TpmActDetailRespVo> actDetailMap = actDetailList.stream().collect(Collectors.toMap(TpmActDetailRespVo::getActDetailCode, Function.identity()));
        //需要占用预算的明细集合
        List<TpmAuditDetailReqVo> occupyFeeBudgetDetails=Lists.newArrayList();
        //获取支付方式的map集合
        Map<String, String> payTypesMap = super.getPayTypesMap();
        for (int i = 0; i < detailVos.size(); i++) {
            TpmAuditDetailReqVo detailReqVo = detailVos.get(i);
            AssertUtils.isNotEmpty(detailReqVo.getPayType(),"支付方式不能为空");
            AssertUtils.isNotEmpty(detailReqVo.getPayTypeName(),"支付方式名称不能为空");
            AssertUtils.isNotNull(detailReqVo.getAuditApplyAmount(),"本次核销金额不能为空");
            TpmActDetailRespVo tpmActDetail = actDetailMap.get(detailReqVo.getActDetailCode());
            AssertUtils.isNotNull(tpmActDetail,"活动明细编码:"+detailReqVo.getActDetailCode()+"不存在,或者不能进行核销操作");
            detailReqVo.setAuditCode(reqVo.getAuditCode());
            if(Objects.isNull(detailReqVo.getFeeUsed())){
                detailReqVo.setFeeUsed(BigDecimal.ZERO);
            }
            if(StringUtils.isBlank(detailReqVo.getIsAllFeeUsed())){
                detailReqVo.setIsAllFeeUsed(GlobalWhetherEnum.NO.getCode());
            }
            if(StringUtils.isEmpty(detailReqVo.getAuditDetailCode())){
                detailReqVo.setAuditDetailCode(CodeUtil.createOneCode(CrmCodeRuleConstants.AUDIT_DETAIL));
            }
            //核销维度只到组织
            if(isSpecial(detailReqVo)){
                AssertUtils.isNotEmpty(detailReqVo.getAuditDetailCustomerReqVos(), "支付方式为货补、折扣、转预付款/账扣时必须选择客户作为核销的对象");
                BigDecimal sum = BigDecimal.ZERO;
                for (TpmAuditDetailCustomerReqVo auditDetailCustomerReqVo : detailReqVo.getAuditDetailCustomerReqVos()) {
                    auditDetailCustomerReqVo.setAuditDetailCode(detailReqVo.getAuditDetailCode());
                    auditDetailCustomerReqVo.setIsAllFeeUsed(GlobalWhetherEnum.NO.getCode());
                    auditDetailCustomerReqVo.setFeeUsed(BigDecimal.ZERO);
                    sum = sum.add(auditDetailCustomerReqVo.getAuditApplyAmount());
                }
                AssertUtils.isTrue(detailReqVo.getAuditApplyAmount().compareTo(sum) == 0, "客户核销金额汇总与对应明细核销申请金额不一致");
            }
            detailReqVo.setPayTypeName(payTypesMap.get(detailReqVo.getPayType()));
            //校验每行的本次核销金额是否超额核销比控制，（本次核销金额+累计已核销金额）/申请金额不能超过活动细类的（1+超额核销比），
            // 如超出则不允许提交，并提示第**行超过允许最大核销金额**，其中最大核销金额为申请金额*（1+超额核销比/100）-已核销金额
            //获取超额核销比例
            detailReqVo.setAlreadyAuditAmount(Optional.ofNullable(tpmActDetail.getAuditAmount()).orElse(BigDecimal.ZERO));
            detailReqVo.setActDetailApplyAmount(tpmActDetail.getApplyAmount());
            TpmCostTypeFineEntity typeFineEntity = fineEntityMap.get(detailReqVo.getFineCode());
            AssertUtils.isNotNull(typeFineEntity,"活动细类:"+detailReqVo.getFineCode()+"不存在");
            BigDecimal extraAuditRatio = typeFineEntity.getExtraAuditRatio();
            //获取申请金额
            BigDecimal addAmount = detailReqVo.getAuditApplyAmount().add(detailReqVo.getAlreadyAuditAmount());
            //获取最大核销金额
            BigDecimal maxAuditAmount = (Optional.ofNullable(extraAuditRatio).orElse(BigDecimal.ZERO).multiply(tpmActDetail.getApplyAmount()).divide(BigDecimal.valueOf(100))).add(tpmActDetail.getApplyAmount());
            AssertUtils.isTrue(addAmount.compareTo(maxAuditAmount) < 1,"第"+(i+1)+"行超过允许最大核销金额:"+maxAuditAmount.subtract(detailReqVo.getAlreadyAuditAmount()));
            if(StringUtils.equals(GlobalWhetherEnum.NO.getCode(),typeFineEntity.getIsAllowRepeatAudit())){
                detailReqVo.setIsAllAudit(GlobalWhetherEnum.YES.getCode());
            }
            if(StringUtils.equals(GlobalWhetherEnum.YES.getCode(),typeFineEntity.getIsAllowRepeatAudit())){
                //允许多次核销,但是超额核销比例为空,如果核销金额等于活动明细的费用申请金额,完全核销标识也必须为 Y ,
                if(Objects.isNull(typeFineEntity.getExtraAuditRatio())&&addAmount.compareTo(tpmActDetail.getApplyAmount())==0){
                    detailReqVo.setIsAllAudit(GlobalWhetherEnum.YES.getCode());
                }
                if(Objects.nonNull(typeFineEntity.getExtraAuditRatio())&&addAmount.compareTo(maxAuditAmount)==0){
                    detailReqVo.setIsAllAudit(GlobalWhetherEnum.YES.getCode());
                }
            }
            if(addAmount.compareTo(tpmActDetail.getApplyAmount())==1){
                reqVo.setOccupyFeeBudgetFlag(true);
                occupyFeeBudgetDetails.add(detailReqVo);
            }
            //如果核销明细里面传了核销资料的文件集合,就把文件转换后放在最外层
            if(CollectionUtils.isNotEmpty(detailReqVo.getAuditFileRespVos())){
                detailReqVo.getAuditFileRespVos().forEach(o->{
                    o.setAuditDetailCode(detailReqVo.getAuditDetailCode());
                    Optional.ofNullable(reqVo.getAuditFileRespVos()).orElse(Lists.newArrayList()).add(o);
                });
            }
        }
        reqVo.setOccupyFeeBudgetDetails(occupyFeeBudgetDetails);
    }

    public boolean isSpecial(TpmAuditDetailReqVo detailReqVo){
        return detailReqVo.getCustomerCode() == null && (ActPayTypeEnum.FORWARD_PAYMENT.getCode().equals(detailReqVo.getPayType()) || ActPayTypeEnum.REPLENISHMENT.getCode().equals(detailReqVo.getPayType()) || ActPayTypeEnum.DISCOUNT.getCode().equals(detailReqVo.getPayType()));
    }

    /**
     * 保存时校验核销编码是否已经存在
     * @param reqVo
     */
    private void checkDataExist(TpmAuditReqVo reqVo) {
        LambdaQueryWrapper<TpmAuditEntity> wrapper = new LambdaQueryWrapper<TpmAuditEntity>().eq(TpmAuditEntity::getAuditCode,reqVo.getAuditCode());
        if(StringUtils.isNotEmpty(reqVo.getId())){
            wrapper.ne(TpmAuditEntity::getId,reqVo.getId());
        }
        Integer count = auditMapper.selectCount(wrapper);
        AssertUtils.isTrue(count==0, AuditException.DATA_CODE_DUPLICATE);
    }

    /**
     * 发起审批
     * @param reqVo
     */
    public void sendToActivity(TpmAuditReqVo reqVo){
        //保存类型:1新增暂存,2,新增并提交审批,3,修改暂存,4修改并提交审批,5,提交审批
    }

    /**
     * 审批驳回的校验
     * @param reqVo
     */
    public TpmAuditEntity rejectAndInterruptCheck(TpmAuditReqVo reqVo) {
        AssertUtils.isNotEmpty(reqVo.getId(),"id不能为空");
        TpmAuditEntity entity = auditMapper.selectOne(new LambdaQueryWrapper<TpmAuditEntity>().eq(TpmAuditEntity::getId,reqVo.getId()));
        if(!StringUtils.equals(ActApproveStatusEnum.APPROVING.getCode(),entity.getApproveStatus())){
            return null;
        }
        if(StringUtils.equals(ActApproveStatusEnum.REJECTED.getCode(),reqVo.getApproveStatus())){
            entity.setApproveStatus(ActApproveStatusEnum.REJECTED.getCode());
        }
        if(StringUtils.equals(ActApproveStatusEnum.INTERRUPT.getCode(),reqVo.getApproveStatus())){
            entity.setApproveStatus(ActApproveStatusEnum.INTERRUPT.getCode());
        }
        return entity;
    }

    /**
     * 组装提交审批流的参数
     * @param entity
     * @param reqVo
     * @return
     */
    public StartProcessReqVo buildStartProcessData(TpmAuditEntity entity, TpmAuditReqVo reqVo) {
        UserRedis user = UserUtils.getUser();
        StartProcessReqVo startProcessReqVo = new StartProcessReqVo();
        startProcessReqVo.setCallBackFeign("TpmAuditCallBackFeign");
        startProcessReqVo.setProcessKey(reqVo.getWorkFlowKey());
        startProcessReqVo.setUserCode(user.getUsername());
        startProcessReqVo.setPosCode(user.getPoscode());
        startProcessReqVo.setTitle(reqVo.getTitle());
        startProcessReqVo.setFormNo(entity.getId());
        startProcessReqVo.setRemark(reqVo.getApproveRemarks());
        startProcessReqVo.setSignTicket(System.currentTimeMillis()+"");
        startProcessReqVo.setBusinessCode(entity.getAuditCode());
        List<TaActFileReqVo> taActFileReqVos = Optional.ofNullable(reqVo.getActivityFileReqVos()).orElse(Lists.newArrayList()).stream().map(o -> {
            TaActFileReqVo taActFileReqVo = new TaActFileReqVo();
            taActFileReqVo.setObjectName(o.getObjectName());
            taActFileReqVo.setFileAddress(o.getAddressUrl());
            return taActFileReqVo;
        }).collect(Collectors.toList());
        if(CollectionUtils.isEmpty(taActFileReqVos)){
            taActFileReqVos = Optional.ofNullable(reqVo.getActivityFileList()).orElse(Lists.newArrayList()).stream().map(o -> {
                TaActFileReqVo taActFileReqVo = new TaActFileReqVo();
                taActFileReqVo.setObjectName(o.getObjectName());
                taActFileReqVo.setFileAddress(o.getFileAddress());
                return taActFileReqVo;
            }).collect(Collectors.toList());
        }
        startProcessReqVo.setFileList(taActFileReqVos);
        return startProcessReqVo;
    }

    /**
     * 核销提交审批校验
     * @param tpmAuditReqVo
     * @return
     */
    public void approveCheck(TpmAuditReqVo tpmAuditReqVo) {
        //活动关闭为单条操作,参数接收为id
        AssertUtils.isNotEmpty(tpmAuditReqVo.getWorkFlowKey(),"请选择流程");
        AssertUtils.isNotEmpty(tpmAuditReqVo.getId(),"id不能为空");
        //提交审批的数据审批状态必须为 待提交、审批驳回、流程追回
        TpmAuditEntity entity = auditMapper.selectOne(new LambdaQueryWrapper<TpmAuditEntity>().eq(TpmAuditEntity::getId, tpmAuditReqVo.getId()));
        //获取可以提交审批的状态
        Set<String> canApproveStatus = ActApproveStatusEnum.getCanApproveStatus();
        AssertUtils.isTrue(canApproveStatus.contains(entity.getApproveStatus()),"仅状态为“待提交、审批驳回、流程追回”的单据可提交审批");
        tpmAuditReqVo.setSaveType(ActSaveTypeEnum.APPROVE.getCode());
    }

    /**
     * 核销提交审批时，活动明细的核销申请在审批中的话，不允许提交审批
     * @param reqVo
     * @return
     */
    public void approvingCheck(TpmAuditReqVo reqVo){
        List<TpmAuditDetailReqVo> detailVos = reqVo.getDetailVos();
            List<String> actDetailCodes = detailVos.stream().map(TpmAuditDetailReqVo::getActDetailCode).collect(Collectors.toList());
            List<String> actDetailCodeList = auditMapper.findApprovingByActDetailCode(actDetailCodes);
            if (CollectionUtils.isNotEmpty(actDetailCodeList)){
                throw new BusinessException("活动明细："+actDetailCodeList.toString()+",对应的核销正在审核中,当前核销申请不能提交审批");
            }
    }

    /**
     * 支付类型为货补时验证货补商品
     * @param reqVo
     */
    private void checkProductInfo(TpmAuditReqVo reqVo) {
        //当上账数据的支付方式类型为费用池并且当前行的支付方式为货补(字典编码"4")时才允许填写产品(产品为多选)
        //需要保存的货补产品数据
        List<TpmAuditDetailReplenishmentProductVo> needSaveProductList= Lists.newArrayList();
        reqVo.getDetailVos().forEach(detail->{
            //if(StringUtils.equals(ActPayTypeEnum.REPLENISHMENT.getCode(),detail.getPayType())){
//                if(StringUtils.isEmpty(detail.getProductLevelCode())&&CollectionUtils.isEmpty(detail.getReplenishmentProductList())){
//                    throw new BusinessException("支付方式为货补的上账明细,货补产品层级和货补产品不能同时为空");
//                }
                List<TpmAuditDetailReplenishmentProductVo> productInfoReqVos = detail.getReplenishmentProductList();
                if(CollectionUtils.isNotEmpty(productInfoReqVos)){
                    Set<String> collect = Sets.newHashSet();
                    productInfoReqVos.forEach(o->{
                        AssertUtils.isNotEmpty(o.getProductCode(),"核销明细编码:"+detail.getAuditDetailCode()+"货补产品编码不能为空");
                        AssertUtils.isNotEmpty(o.getProductName(),"核销明细编码:"+detail.getAuditDetailCode()+"货补产品名称不能为空");
                        o.setAuditCode(reqVo.getAuditCode());
                        o.setAuditDetailCode(detail.getAuditDetailCode());
                        AssertUtils.isTrue(!collect.contains(o.getProductCode()),"核销明细编码:"+detail.getAuditDetailCode()+"货补产品重复");
                        collect.add(o.getProductCode());
                    });
                    needSaveProductList.addAll(productInfoReqVos);
                }
//            }else {
//                if(StringUtils.isNotEmpty(detail.getProductLevelCode())||CollectionUtils.isEmpty(detail.getReplenishmentProductList())){
//                    throw new BusinessException("核销明细编码:"+detail.getAuditDetailCode()+"不能选择货补产品层级或者货补产品");
//                }
//            }
        });
        reqVo.setAuditDetailReplenishmentProductVos(needSaveProductList);
    }
}
