package com.biz.crm.feebudget.service.impl;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.biz.crm.CrmCodeRuleConstants;
import com.biz.crm.base.ApiResultUtil;
import com.biz.crm.base.BaseServiceHelper;
import com.biz.crm.base.BusinessException;
import com.biz.crm.budgetsubjects.mapper.TpmBudgetSubjectsMapper;
import com.biz.crm.budgetsubjects.model.TpmBudgetSubjectsEntity;
import com.biz.crm.common.TpmGlobalDictConstants;
import com.biz.crm.eunm.CrmEnableStatusEnum;
import com.biz.crm.eunm.GlobalWhetherEnum;
import com.biz.crm.eunm.tpm.BudgetSubjectsControlTypeEnum;
import com.biz.crm.eunm.tpm.FeeBudgetDetailTypeEnum;
import com.biz.crm.eunm.tpm.FeeBudgetTypeEnum;
import com.biz.crm.exception.tpm.FeeBudgetException;
import com.biz.crm.feebudget.mapper.TpmFeeBudgetControlMapper;
import com.biz.crm.feebudget.mapper.TpmFeeBudgetDetailsMapper;
import com.biz.crm.feebudget.mapper.TpmFeeBudgetMapper;
import com.biz.crm.feebudget.model.OperateBudgetControlReqVo;
import com.biz.crm.feebudget.model.TpmFeeBudgetControlEntity;
import com.biz.crm.feebudget.model.TpmFeeBudgetDetailsEntity;
import com.biz.crm.feebudget.model.TpmFeeBudgetEntity;
import com.biz.crm.mdm.customer.MdmCustomerMsgFeign;
import com.biz.crm.mdm.org.MdmOrgFeign;
import com.biz.crm.mdm.product.MdmProductFeign;
import com.biz.crm.mdm.terminal.MdmTerminalFeign;
import com.biz.crm.nebular.mdm.customer.MdmCustomerMsgReqVo;
import com.biz.crm.nebular.mdm.customer.MdmCustomerMsgRespVo;
import com.biz.crm.nebular.mdm.customer.MdmCustomerMsgSelectRespVo;
import com.biz.crm.nebular.mdm.dict.resp.DictDataVo;
import com.biz.crm.nebular.mdm.org.req.MdmOrgReqVo;
import com.biz.crm.nebular.mdm.org.resp.MdmOrgRespVo;
import com.biz.crm.nebular.mdm.product.resp.MdmProductRespVo;
import com.biz.crm.nebular.mdm.productlevel.resp.MdmProductLevelRespVo;
import com.biz.crm.nebular.mdm.terminal.MdmCustomerTerminalVo;
import com.biz.crm.nebular.mdm.terminal.MdmTerminalVo;
import com.biz.crm.nebular.tpm.feebudget.req.TpmFeeBudgetDistributionReqVo;
import com.biz.crm.nebular.tpm.feebudget.req.TpmFeeBudgetReqVo;
import com.biz.crm.nebular.tpm.feebudget.resp.TpmFeeBudgetRespVo;
import com.biz.crm.util.*;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.collections4.MapUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.math.BigDecimal;
import java.util.*;

/**
 * @Project crm
 * @ClassName FeeBudgetServiceHelper
 * @Author HuangLong
 * @Date 2020/9/8 14:02
 * @Description 费用预算工具类
 */
@Slf4j
@Service
public class FeeBudgetServiceHelper extends BaseServiceHelper {

    @Resource
    private TpmFeeBudgetMapper mapper;

    @Resource
    private TpmFeeBudgetDetailsMapper detailsMapper;

    @Resource
    private TpmBudgetSubjectsMapper budgetSubjectsMapper;

    @Resource
    private MdmCustomerMsgFeign customerFeign;

    @Resource
    private TpmFeeBudgetControlMapper controlMapper;

    @Resource
    private MdmOrgFeign orgFeign;
    @Autowired
    private MdmProductFeign productFeign;
    @Autowired
    private MdmTerminalFeign terminalFeign;


    public Map<String, Map<String, String>> getDicts() {
        List<String> dictCodes = Lists.newArrayList();
        dictCodes.add(TpmGlobalDictConstants.FISCAL_YEAR);
        dictCodes.add(TpmGlobalDictConstants.FISCAL_MONTH);
        dictCodes.add(TpmGlobalDictConstants.FISCAL_QUATER);
        dictCodes.add(TpmGlobalDictConstants.ENABLE_STATUS);
        dictCodes.add(TpmGlobalDictConstants.CHANNEL);
        dictCodes.add(TpmGlobalDictConstants.FEE_BUDGET_TYPE);
        dictCodes.add(TpmGlobalDictConstants.CONTROL_TYPE);
        //查询字典数据
        return DictUtil.getDictValueMapsByCodes(dictCodes);
    }

    /**
     * 转换列表数据
     *
     * @param list
     */
    public void convertListDate(List<TpmFeeBudgetRespVo> list) {
        if (CollectionUtils.isNotEmpty(list)) {
            //获取需要的字典集合
            Map<String, Map<String, String>> map = this.getDicts();
            list.forEach(o -> {
                //启用禁用状态
                if (StringUtils.isNotEmpty(o.getEnableStatus())) {
                    o.setEnableStatusName(Optional.ofNullable(map.get(TpmGlobalDictConstants.ENABLE_STATUS)).orElse(Maps.newHashMap()).get(o.getEnableStatus()));
                }
                //年份
                if (StringUtils.isNotEmpty(o.getBudgetYear())) {
                    o.setBudgetYearName(Optional.ofNullable(map.get(TpmGlobalDictConstants.FISCAL_YEAR)).orElse(Maps.newHashMap()).get(o.getBudgetYear()));
                }
                //月份
                if (StringUtils.isNotEmpty(o.getBudgetMonth())) {
                    o.setBudgetMonthName(Optional.ofNullable(map.get(TpmGlobalDictConstants.FISCAL_MONTH)).orElse(Maps.newHashMap()).get(o.getBudgetMonth()));
                }
                //季度
                if (StringUtils.isNotEmpty(o.getBudgetQuater())) {
                    o.setBudgetQuaterName(Optional.ofNullable(map.get(TpmGlobalDictConstants.FISCAL_QUATER)).orElse(Maps.newHashMap()).get(o.getBudgetQuater()));
                }
                //渠道
                if (StringUtils.isNotEmpty(o.getChannel())) {
                    o.setChannelName(Optional.ofNullable(map.get(TpmGlobalDictConstants.CHANNEL)).orElse(Maps.newHashMap()).get(o.getChannel()));
                }
                //费用预算类型
                if (StringUtils.isNotEmpty(o.getFeeBudgetType())) {
                    o.setFeeBudgetTypeName(Optional.ofNullable(map.get(TpmGlobalDictConstants.FEE_BUDGET_TYPE)).orElse(Maps.newHashMap()).get(o.getFeeBudgetType()));
                }
            });
        }
    }

    /**
     * 修改数据(启用,禁用)
     *
     * @param reqVo
     * @param entity
     */
    public void updateBySelectAll(TpmFeeBudgetReqVo reqVo, TpmFeeBudgetEntity entity) {
        //TODO  这里必须加上数据权限的限制!!!!
        mapper.updateBySelectAll(entity, reqVo);
    }

    public void saveCheck(TpmFeeBudgetReqVo reqVo) {
        commonCheck(reqVo);
        AssertUtils.isTrue(getCount(reqVo) == 0, "已经存在当前维度的费用预算,请勿重复添加");
    }

    public boolean incomeGenerateCheck(TpmFeeBudgetReqVo reqVo) {
        commonCheck(reqVo);
        return getCount(reqVo) == 0;
    }

    /**
     * 新增,编辑,校验
     *
     * @param reqVo
     */
    public void commonCheck(TpmFeeBudgetReqVo reqVo) {
        //校验数据不为空
        if (StringUtils.isEmpty(reqVo.getFeeBudgetCode())) {
            reqVo.setFeeBudgetCode(CodeUtil.createOneCode(CrmCodeRuleConstants.FEE_BUDGET));
        }
        AssertUtils.isNotEmpty(reqVo.getFeeBudgetCode(), "费用预算编码不能为空");
        AssertUtils.isNotEmpty(reqVo.getFeeBudgetType(), "费用预算类型");
        AssertUtils.isNotEmpty(reqVo.getBudgetYear(), "请选择年份");
        AssertUtils.isNotEmpty(reqVo.getBudgetMonth(), "请选择月份");
        AssertUtils.isNotEmpty(reqVo.getBudgetQuater(), "请选择季度");
        AssertUtils.isNotNull(reqVo.getInitAmount(), "请输入期初金额");
        AssertUtils.isNotEmpty(reqVo.getBudgetSubjectsCode(), "请选择预算科目");
        //设置季度
        if (StringUtils.isNotEmpty(reqVo.getBudgetMonth())) {
            int quaterByMonth = DateUtil.getQuaterByMonth(Integer.valueOf(reqVo.getBudgetMonth()));
            reqVo.setBudgetQuater(String.valueOf(quaterByMonth));
        }
        //如果传了组织编码,就用组织编码把组织的数据查出来
        MdmOrgRespVo orgRespVo = new MdmOrgRespVo();
        if (StringUtils.isNotEmpty(reqVo.getOrgCode())) {
            orgRespVo = orgFeign.detail(reqVo.getOrgCode()).getResult();
            AssertUtils.isNotNull(orgRespVo, "您选择的组织不存在");
            AssertUtils.isTrue(StringUtils.equals(CrmEnableStatusEnum.ENABLE.getCode(), orgRespVo.getEnableStatus()), "您选择的组织尚未启用");
            reqVo.setOrgType(Optional.ofNullable(orgRespVo.getOrgType()).orElse(""));
            reqVo.setOrgName(Optional.ofNullable(orgRespVo.getOrgName()).orElse(""));
            reqVo.setOrgTypeName(Optional.ofNullable(orgRespVo.getOrgTypeName()).orElse(""));
        }
        if (StringUtils.isNotEmpty(reqVo.getProductLevelCode())) {
            MdmProductLevelRespVo productLevelByCode = ProductLevelUtil.getProductLevelByCode(reqVo.getProductLevelCode());
            AssertUtils.isNotNull(productLevelByCode, "您选择的产品层级不存在");
            AssertUtils.isTrue(StringUtils.equals(CrmEnableStatusEnum.ENABLE.getCode(), productLevelByCode.getEnableStatus()), "您选择的产品层级已被禁用");
            reqVo.setProductLevelName(productLevelByCode.getProductLevelName());
        }
        if (StringUtils.isNotEmpty(reqVo.getProductCode())) {
            MdmProductRespVo mdmProductRespVo = ApiResultUtil.objResult(productFeign.detail(null, reqVo.getProductCode()), true);
            AssertUtils.isNotNull(mdmProductRespVo, "您选择的产品不存在");
            AssertUtils.isTrue(StringUtils.equals(CrmEnableStatusEnum.ENABLE.getCode(), mdmProductRespVo.getEnableStatus()), "您选择的产品已被禁用");
            AssertUtils.isTrue(StringUtils.equals(GlobalWhetherEnum.YES.getCode(), mdmProductRespVo.getIsShelf()), "您选择的产品已下架");
            if (StringUtils.isNotEmpty(reqVo.getProductLevelCode())) {
                AssertUtils.isTrue(StringUtils.equals(reqVo.getProductLevelCode(), mdmProductRespVo.getProductLevelCode()), "您选择的产品层级和产品不存在关联关系");
                reqVo.setProductName(mdmProductRespVo.getProductName());
            }
        }
        //部门预算
        if (StringUtils.equals(FeeBudgetTypeEnum.DEPARTMENT.getCode(), reqVo.getFeeBudgetType())) {
            AssertUtils.isNotEmpty(reqVo.getOrgCode(), "请选择组织");
        }
        //客户预算
        if (StringUtils.equals(FeeBudgetTypeEnum.CUSTOMER.getCode(), reqVo.getFeeBudgetType())) {
            AssertUtils.isNotEmpty(reqVo.getCustomerCode(), "请选择客户");
            MdmCustomerMsgReqVo customerMsgReqVo = new MdmCustomerMsgReqVo();
            customerMsgReqVo.setCustomerCode(reqVo.getCustomerCode());
            List<MdmCustomerMsgRespVo> customerMsgRespVos = customerFeign.list(customerMsgReqVo).getResult().getData();
            if (CollectionUtils.isNotEmpty(customerMsgRespVos)) {
                MdmCustomerMsgRespVo customerMsgRespVo = customerMsgRespVos.get(0);
                AssertUtils.isNotNull(customerMsgRespVo, "您选择的客户不存在");
                AssertUtils.isTrue(StringUtils.equals(CrmEnableStatusEnum.ENABLE.getCode(), customerMsgRespVo.getEnableStatus()), "您选择的客户已被禁用");
                AssertUtils.isNotEmpty(customerMsgRespVo.getOrgCode(), "您选择的客户组织数据不完整");
                reqVo.setCustomerName(customerMsgRespVo.getCustomerName());
                reqVo.setCustomerOrgCode(customerMsgRespVo.getCustomerOrgCode());
                reqVo.setCustomerOrgName(customerMsgRespVo.getCustomerOrgName());
                reqVo.setOrgCode(customerMsgRespVo.getOrgCode());
                reqVo.setOrgName(customerMsgRespVo.getOrgName());
            }
        }
        //渠道预算
        if (StringUtils.equals(FeeBudgetTypeEnum.CHANNEL.getCode(), reqVo.getFeeBudgetType())) {
            AssertUtils.isNotEmpty(reqVo.getOrgCode(), "请选择组织");
            AssertUtils.isNotEmpty(reqVo.getChannel(), "请选择渠道");
        }
        //门店预算
        if (StringUtils.equals(FeeBudgetTypeEnum.TERMINAL.getCode(), reqVo.getFeeBudgetType())) {
            AssertUtils.isNotEmpty(reqVo.getCustomerCode(), "请选择客户");
            AssertUtils.isNotEmpty(reqVo.getTerminalCode(), "请选择门店");
        }
        reqVo.setEnableStatus(CrmEnableStatusEnum.ENABLE.getCode());
        //校验预算科目
        TpmBudgetSubjectsEntity subjectsEntity = budgetSubjectsMapper.selectOne(new LambdaQueryWrapper<TpmBudgetSubjectsEntity>().eq(TpmBudgetSubjectsEntity::getBudgetSubjectsCode, reqVo.getBudgetSubjectsCode()));
        AssertUtils.isNotNull(subjectsEntity, "预算科目数据不存在");
        AssertUtils.isTrue(StringUtils.equals(CrmEnableStatusEnum.ENABLE.getCode(), subjectsEntity.getEnableStatus()), "您选择预算科目已被禁用");
        if (StringUtils.equals(BudgetSubjectsControlTypeEnum.MONTH.getCode(), subjectsEntity.getControlType())) {
            AssertUtils.isNotEmpty(reqVo.getBudgetMonth(), "请选择月份");
        }
        reqVo.setBudgetSubjectsControlType(subjectsEntity.getControlType());
    }

    /**
     * 获取异常描述
     *
     * @param reqVo
     * @return
     */
    public String getErrorMsg(TpmFeeBudgetReqVo reqVo) {
        StringBuffer msg = new StringBuffer("已存在,");
        msg.append(" 年份:").append(reqVo.getBudgetYear());
        msg.append(",月份:").append(Optional.ofNullable(reqVo.getBudgetMonth()).orElse(""));
        msg.append(",客户编码:").append(reqVo.getCustomerCode());
        msg.append(",预算科目编码:").append(reqVo.getBudgetSubjectsCode());
        msg.append(",产品层级编码:").append(reqVo.getProductLevelCode());
        msg.append(",产品编码:").append(reqVo.getProductCode());
        msg.append(",组织编码:").append(reqVo.getOrgCode());
        msg.append(",渠道:").append(reqVo.getChannel());
        msg.append(",门店编码:").append(reqVo.getTerminalCode());
        msg.append(",的费用预算数据");
        return msg.toString();
    }

    private int getCount(TpmFeeBudgetReqVo reqVo) {
        LambdaQueryWrapper<TpmFeeBudgetEntity> wrapper = new LambdaQueryWrapper<TpmFeeBudgetEntity>().eq(TpmFeeBudgetEntity::getFeeBudgetCode, reqVo.getFeeBudgetCode());
        if (StringUtils.isNotEmpty(reqVo.getId())) {
            wrapper.ne(TpmFeeBudgetEntity::getId, reqVo.getId());
        }
        TpmFeeBudgetEntity entity = mapper.selectOne(wrapper);
        AssertUtils.isTrue(Objects.isNull(entity), FeeBudgetException.DATA_CODE_DUPLICATE);
        //根据维度校验重复数据
        LambdaQueryWrapper<TpmFeeBudgetEntity> queryWrapper = new LambdaQueryWrapper<TpmFeeBudgetEntity>();
        if (StringUtils.isNotEmpty(reqVo.getId())) {
            queryWrapper.ne(TpmFeeBudgetEntity::getId, reqVo.getId());
        }
        queryWrapper.eq(TpmFeeBudgetEntity::getBudgetYear, reqVo.getBudgetYear());
        if (StringUtils.isNotEmpty(reqVo.getBudgetMonth())) {
            queryWrapper.eq(TpmFeeBudgetEntity::getBudgetMonth, reqVo.getBudgetMonth());
        } else {
            queryWrapper.isNull(TpmFeeBudgetEntity::getBudgetMonth);
        }
        if (StringUtils.isNotEmpty(reqVo.getBudgetQuater())) {
            queryWrapper.eq(TpmFeeBudgetEntity::getBudgetQuater, reqVo.getBudgetQuater());
        } else {
            queryWrapper.isNull(TpmFeeBudgetEntity::getBudgetQuater);
        }
        //客户编码
        if (StringUtils.isNotEmpty(reqVo.getCustomerCode())) {
            queryWrapper.eq(TpmFeeBudgetEntity::getCustomerCode, reqVo.getCustomerCode());
        } else {
            queryWrapper.isNull(TpmFeeBudgetEntity::getCustomerCode);
        }
        //预算科目编码
        queryWrapper.eq(TpmFeeBudgetEntity::getBudgetSubjectsCode, reqVo.getBudgetSubjectsCode());
        //产品层级
        if (StringUtils.isNotEmpty(reqVo.getProductLevelCode())) {
            queryWrapper.eq(TpmFeeBudgetEntity::getProductLevelCode, reqVo.getProductLevelCode());
        } else {
            queryWrapper.isNull(TpmFeeBudgetEntity::getProductLevelCode);
        }
        //产品编码
        if (StringUtils.isNotEmpty(reqVo.getProductCode())) {
            queryWrapper.eq(TpmFeeBudgetEntity::getProductCode, reqVo.getProductCode());
        } else {
            queryWrapper.isNull(TpmFeeBudgetEntity::getProductCode);
        }
        //企业组织编码
        if (StringUtils.isNotEmpty(reqVo.getOrgCode())) {
            queryWrapper.eq(TpmFeeBudgetEntity::getOrgCode, reqVo.getOrgCode());
        } else {
            queryWrapper.isNull(TpmFeeBudgetEntity::getOrgCode);
        }
        //渠道
        if (StringUtils.isNotEmpty(reqVo.getChannel())) {
            queryWrapper.eq(TpmFeeBudgetEntity::getChannel, reqVo.getChannel());
        } else {
            queryWrapper.isNull(TpmFeeBudgetEntity::getChannel);
        }
        //门店编码
        if (StringUtils.isNotEmpty(reqVo.getTerminalCode())) {
            queryWrapper.eq(TpmFeeBudgetEntity::getTerminalCode, reqVo.getTerminalCode());
        } else {
            queryWrapper.isNull(TpmFeeBudgetEntity::getTerminalCode);
        }
        if (com.biz.crm.util.StringUtils.isNotEmpty(reqVo.getExt1())) {
            queryWrapper.eq(TpmFeeBudgetEntity::getExt1, reqVo.getExt1());
        }
        Integer count = mapper.selectCount(queryWrapper);
        return count;
    }

    /**
     * 新增,修改,转换费用预算明细的方法
     *
     * @param entity
     * @return
     */
    public TpmFeeBudgetDetailsEntity transformDetail(TpmFeeBudgetEntity entity, TpmFeeBudgetReqVo reqVo) {
        TpmFeeBudgetDetailsEntity detailsEntity = new TpmFeeBudgetDetailsEntity();
        CrmBeanUtil.copyProperties(entity, detailsEntity);
        detailsEntity.setId(null);
        detailsEntity.setBusinessCode(detailsEntity.getFeeBudgetCode());
        FeeBudgetDetailTypeEnum detailTypeEnum = null;
        //修改后余额
        BigDecimal afterAmount = BigDecimal.ZERO;
        //修改前余额
        BigDecimal beforAmount = BigDecimal.ZERO;
        //操作金额
        BigDecimal feeAmount = BigDecimal.ZERO;
        //新增或者修改
        if (StringUtils.equals(FeeBudgetDetailTypeEnum.INIT.getCode(), reqVo.getFeeBudgetDetailType())) {
            //期初
            detailTypeEnum = FeeBudgetDetailTypeEnum.INIT;
            beforAmount = BigDecimal.ZERO;
            afterAmount = entity.getInitAmount();
            feeAmount = entity.getInitAmount();
        } else if (StringUtils.equals(FeeBudgetDetailTypeEnum.ADDITIONAL.getCode(), reqVo.getFeeBudgetDetailType())) {
            //追加
            afterAmount = entity.getCanUseAmount();
            beforAmount = entity.getCanUseAmount().subtract(reqVo.getChangeAmount());
            feeAmount = reqVo.getChangeAmount();
            detailTypeEnum = FeeBudgetDetailTypeEnum.ADDITIONAL;
        } else if (StringUtils.equals(FeeBudgetDetailTypeEnum.CUT_OUT.getCode(), reqVo.getFeeBudgetDetailType())) {
            //削减
            afterAmount = entity.getCanUseAmount();
            beforAmount = entity.getCanUseAmount().add(reqVo.getChangeAmount());
            feeAmount = BigDecimal.ZERO.subtract(reqVo.getChangeAmount());
            detailTypeEnum = FeeBudgetDetailTypeEnum.CUT_OUT;
        } else if (StringUtils.equals(FeeBudgetDetailTypeEnum.ADJUST_IN.getCode(), reqVo.getFeeBudgetDetailType())) {
            //调入
            afterAmount = entity.getCanUseAmount();
            feeAmount = reqVo.getChangeAmount();
            beforAmount = entity.getCanUseAmount().subtract(reqVo.getChangeAmount());
            detailTypeEnum = FeeBudgetDetailTypeEnum.ADJUST_IN;
            //调入方存调出方的编码
            detailsEntity.setBusinessCode(reqVo.getFeeBudgetCode());
        } else if (StringUtils.equals(FeeBudgetDetailTypeEnum.ADJUST_OUT.getCode(), reqVo.getFeeBudgetDetailType())) {
            //调出
            afterAmount = entity.getCanUseAmount();
            feeAmount = BigDecimal.ZERO.subtract(reqVo.getChangeAmount());
            beforAmount = entity.getCanUseAmount().add(reqVo.getChangeAmount());
            detailTypeEnum = FeeBudgetDetailTypeEnum.ADJUST_OUT;
            //调出方存调入方的编码
            detailsEntity.setBusinessCode(reqVo.getAdjustInFeeBudgetCode());
        }else {
            //划拨
            detailTypeEnum = FeeBudgetDetailTypeEnum.DISTRIBUTION;
            beforAmount = BigDecimal.ZERO;
            afterAmount = entity.getInitAmount();
            feeAmount = entity.getInitAmount();
        }
        detailsEntity.setBusinessRemarks(reqVo.getBusinessRemarks());
        detailsEntity.setFeeBudgetDetailType(detailTypeEnum.getCode());
        detailsEntity.setFeeBudgetDetailTypeName(detailTypeEnum.getDes());
        detailsEntity.setMonth(entity.getBudgetMonth());
        detailsEntity.setYear(entity.getBudgetYear());
        detailsEntity.setAfterAmount(afterAmount);
        detailsEntity.setFeeAmount(feeAmount);
        detailsEntity.setBeforAmount(beforAmount);
        return detailsEntity;
    }

    /**
     * 新增修改时保存明细数据
     */
    public void createOrUpdateSaveDetail(TpmFeeBudgetEntity entity, TpmFeeBudgetReqVo reqVo) {
        //如果传入的实体id不为空,说明是修改,修改需要先把原来的初始化明细数据删除
        if (StringUtils.isNotEmpty(entity.getId())) {
            detailsMapper.delete(new LambdaQueryWrapper<TpmFeeBudgetDetailsEntity>().eq(TpmFeeBudgetDetailsEntity::getFeeBudgetCode, entity.getFeeBudgetCode()));
        }
        TpmFeeBudgetDetailsEntity detailsEntity = this.transformDetail(entity, reqVo);
        detailsMapper.insert(detailsEntity);
    }

    /**
     * 变更,调整时保存明细数据
     */
    public void changeOrAdjustSaveDetail(TpmFeeBudgetEntity entity, TpmFeeBudgetReqVo reqVo) {
        TpmFeeBudgetDetailsEntity detailsEntity = this.transformDetail(entity, reqVo);
        detailsMapper.insert(detailsEntity);
    }

    /**
     * 变更校验
     *
     * @param reqVo
     */
    public void changeCheck(TpmFeeBudgetReqVo reqVo) {
        //1、变更金额必须是正数；
        //2、追加时用可用余额+变更金额
        AssertUtils.isNotNull(reqVo.getChangeAmount(), "变更金额不能为空");
        AssertUtils.isTrue(reqVo.getChangeAmount().compareTo(BigDecimal.ZERO) > 0, "变更金额必须大于0");
        AssertUtils.isNotEmpty(reqVo.getFeeBudgetDetailType(), "变更类型不能为空");
        AssertUtils.isTrue(StringUtils.equals(FeeBudgetDetailTypeEnum.ADDITIONAL.getCode(), reqVo.getFeeBudgetDetailType()) || StringUtils.equals(FeeBudgetDetailTypeEnum.CUT_OUT.getCode(), reqVo.getFeeBudgetDetailType()), "费用预算变更只能选择追加或者削减");
        //查询一下预算科目的控制类型
        TpmBudgetSubjectsEntity subjectsEntity = budgetSubjectsMapper.selectOne(new LambdaQueryWrapper<TpmBudgetSubjectsEntity>().eq(TpmBudgetSubjectsEntity::getBudgetSubjectsCode, reqVo.getBudgetSubjectsCode()));
        reqVo.setBudgetSubjectsControlType(subjectsEntity.getControlType());
    }

    /**
     * 费用预算调整检验
     *
     * @param reqVo
     */
    public void adjustCheck(TpmFeeBudgetReqVo reqVo) {
        /*1、选择调入方时需要除开当前调出方；
        2、保存时校验：调出方的调整后余额不能为负（保存时计算一次，不能用页面上计算的数据）
        3、调出金额必须是正数*/
        //如果控制类型为不控制 金额可为负
        AssertUtils.isNotEmpty(reqVo.getFeeBudgetDetailType(), "请选择变更类型");
        AssertUtils.isNotEmpty(reqVo.getAdjustInFeeBudgetId(), "请选择调入方");
        AssertUtils.isTrue(!StringUtils.equals(reqVo.getId(), reqVo.getAdjustInFeeBudgetId()), "调入方不能与调出方一致");
        AssertUtils.isNotNull(reqVo.getChangeAmount(), "请输入调出金额");
    }

    /**
     * 计算调整金额,调整后金额,已使用金额
     *
     * @param entity
     */
    public void computeAmount(TpmFeeBudgetEntity entity, TpmFeeBudgetReqVo reqVo) {
        if (StringUtils.equals(FeeBudgetDetailTypeEnum.CUT_OUT.getCode(), reqVo.getFeeBudgetDetailType()) || StringUtils.equals(FeeBudgetDetailTypeEnum.ADJUST_OUT.getCode(), reqVo.getFeeBudgetDetailType())) {
            entity.setAdjustTotalAmount(entity.getAdjustTotalAmount().subtract(reqVo.getChangeAmount()));
            entity.setAfterAdjustAmount(entity.getAfterAdjustAmount().subtract(reqVo.getChangeAmount()));
        } else if (StringUtils.equals(FeeBudgetDetailTypeEnum.ADDITIONAL.getCode(), reqVo.getFeeBudgetDetailType()) || StringUtils.equals(FeeBudgetDetailTypeEnum.ADJUST_IN.getCode(), reqVo.getFeeBudgetDetailType())) {
            entity.setAdjustTotalAmount(entity.getAdjustTotalAmount().add(reqVo.getChangeAmount()));
            entity.setAfterAdjustAmount(entity.getAfterAdjustAmount().add(reqVo.getChangeAmount()));
        }
    }

    /**
     * 费用预算新增时,操作维度的方法
     *
     * @param reqVo
     */
    public void createFeeBudget(OperateBudgetControlReqVo reqVo) {
        TpmFeeBudgetEntity entity = reqVo.getEntity();
        TpmFeeBudgetReqVo budgetReqVo = reqVo.getReqVo();
        String controlType = budgetReqVo.getBudgetSubjectsControlType();
        //1.获取数据字典里面配置的每个预算类型的控制维度
        Set<String> feeBudgetControls = this.getFeeBudgetControls(entity.getFeeBudgetType(), controlType);
        Map<String, Object> newQueryMap = convertDataToMap(feeBudgetControls, entity);
        TpmFeeBudgetControlEntity newControl = this.findFeeBudgetControlByParams(newQueryMap);
        if (Objects.isNull(newControl)) {
            newControl = this.saveNewControl(budgetReqVo, entity);
        } else {
            if (StringUtils.isNotEmpty(newControl.getFeeBudgetCodes())) {
                newControl.setFeeBudgetCodes(newControl.getFeeBudgetCodes() + "," + entity.getFeeBudgetCode());
            } else {
                newControl.setFeeBudgetCodes(entity.getFeeBudgetCode());
            }
            this.updateControl(newControl, BigDecimal.ZERO, budgetReqVo.getInitAmount());
        }
        entity.setControlId(newControl.getId());
    }

    /**
     * 费用预算编辑时,操作维度的方法
     *
     * @param reqVo
     */
    public void updateFeeBudget(OperateBudgetControlReqVo reqVo) {
        TpmFeeBudgetEntity entity = reqVo.getEntity();
        TpmFeeBudgetReqVo budgetReqVo = reqVo.getReqVo();
        String controlType = budgetReqVo.getBudgetSubjectsControlType();
        //1.获取数据字典里面配置的每个预算类型的控制维度
        Set<String> feeBudgetControls = this.getFeeBudgetControls(entity.getFeeBudgetType(), controlType);
        AssertUtils.isNotEmpty(entity.getControlId(), "费用预算数据异常,控制维度id为空");
        TpmFeeBudgetControlEntity oldControl = controlMapper.selectOne(new LambdaQueryWrapper<TpmFeeBudgetControlEntity>().eq(TpmFeeBudgetControlEntity::getId, entity.getControlId()));
        Map<String, Object> newQueryMap = convertDataToMap(feeBudgetControls, entity);
        TpmFeeBudgetControlEntity newControl = this.findFeeBudgetControlByParams(newQueryMap);
        //判断修改数据后的控制维度和原始的控制维度是否一致
        if (Objects.nonNull(newControl)) {
            if (StringUtils.equals(entity.getControlId(), newControl.getId())) {
                //如果修改后的数据不需要更改控制维度,再判断金额是否有改变,金额不变,不做任何操作,否则修改控制维度的金额
                if (budgetReqVo.getInitAmount().compareTo(budgetReqVo.getCanUseAmount()) != 0) {
                    //减去原始金额,加上新的金额
                    this.updateControl(oldControl, budgetReqVo.getCanUseAmount(), budgetReqVo.getInitAmount());
                }
            } else {
                //修改新的控制维度的金额
                newControl.setFeeBudgetCodes(newControl.getFeeBudgetCodes() + "," + entity.getFeeBudgetCode());
                this.updateControl(newControl, BigDecimal.ZERO, budgetReqVo.getInitAmount());
                //修改原来的控制维度金额
                String feeBudgetCodes = this.replaceFeeBudgetCodes(oldControl.getFeeBudgetCodes(), entity.getFeeBudgetCode());
                oldControl.setFeeBudgetCodes(feeBudgetCodes);
                this.updateControl(oldControl, budgetReqVo.getCanUseAmount(), BigDecimal.ZERO);
            }
        } else {
            //如果新的控制维度没有查询到,则需要新增
            newControl = this.saveNewControl(budgetReqVo, entity);
            String feeBudgetCodes = this.replaceFeeBudgetCodes(oldControl.getFeeBudgetCodes(), entity.getFeeBudgetCode());
            oldControl.setFeeBudgetCodes(feeBudgetCodes);
            this.updateControl(oldControl, budgetReqVo.getCanUseAmount(), BigDecimal.ZERO);
        }
        entity.setControlId(newControl.getId());
    }

    /**
     * 替换费用预算维度表的费用预算编码值
     *
     * @param originCodes
     * @param replaceCode
     * @return
     */
    public String replaceFeeBudgetCodes(String originCodes, String replaceCode) {
        if (StringUtils.equals(replaceCode, originCodes)) {
            return null;
        }
        if (originCodes.endsWith(replaceCode)) {
            return originCodes.replaceAll(replaceCode, "");
        }
        return originCodes.replaceAll(replaceCode + ",", "");
    }


    /**
     * 费用预算删除时,操作维度的方法
     *
     * @param reqVo
     */
    private void deleteFeeBudget(OperateBudgetControlReqVo reqVo) {
        this.updateControl(reqVo.getControlEntity(), reqVo.getControlUpdateAmount(), BigDecimal.ZERO);
    }

    /**
     * 新增一条费用预算控制维度
     *
     * @param reqVo
     * @param entity
     * @return
     */
    public TpmFeeBudgetControlEntity saveNewControl(TpmFeeBudgetReqVo reqVo, TpmFeeBudgetEntity entity) {
        //新增控制维度
        TpmFeeBudgetControlEntity newControl = new TpmFeeBudgetControlEntity();
        CrmBeanUtil.copyProperties(entity, newControl);
        //设置费用预算编码
        newControl.setFeeBudgetCodes(entity.getFeeBudgetCode());
        //获取selectKey
        String selectKey = this.getSelectKey(reqVo);
        newControl.setCanUseAmount(entity.getCanUseAmount());
        newControl.setSelectKey(selectKey);
        super.setPublicParamsNull(newControl);

        if (StringUtils.equals(BudgetSubjectsControlTypeEnum.QUATER.getCode(), reqVo.getBudgetSubjectsControlType())) {
            newControl.setBudgetMonth(null);
        }
        if (StringUtils.equals(BudgetSubjectsControlTypeEnum.YEAR.getCode(), reqVo.getBudgetSubjectsControlType())) {
            newControl.setBudgetMonth(null);
            newControl.setBudgetQuater(null);
        }
        controlMapper.insert(newControl);
        return newControl;
    }

    /**
     * 修改已经有的控制维度    可以在这里进行加锁控制
     *
     * @param controlEntity
     * @param oldAmount     费用预算原始的金额,费用预算新的剩余金额
     * @param newAmount
     */
    public void updateControl(TpmFeeBudgetControlEntity controlEntity, BigDecimal oldAmount, BigDecimal newAmount) {
        //修改控制维度数据,将控制维度的可用余额增加或减少
        //如果费用预算控制维度上的预算编码为空,直接把该费用预算控制维度删掉
        if (StringUtils.isEmpty(controlEntity.getFeeBudgetCodes())) {
            controlMapper.deleteById(controlEntity.getId());
        } else {
            controlEntity.setCanUseAmount(Optional.ofNullable(controlEntity.getCanUseAmount()).orElse(BigDecimal.ZERO).add(Optional.ofNullable(newAmount).orElse(BigDecimal.ZERO)).subtract(Optional.ofNullable(oldAmount).orElse(BigDecimal.ZERO)));
            controlMapper.updateById(controlEntity);
        }
    }

    /**
     * 费用预算新增,修改,调整,变更 时,更新或新增费用预算控制维度数据
     *
     * @param reqVo
     */
    public void saveFeeBudgetControlData(OperateBudgetControlReqVo reqVo) {

        AssertUtils.isNotNull(reqVo.getTypeEnum(), "操作类型不能为空");

        switch (reqVo.getTypeEnum()) {
            case NEW:
                this.createFeeBudget(reqVo);
                break;
            case UPDATE:
                this.updateFeeBudget(reqVo);
                break;
            case DELETE:
                this.deleteFeeBudget(reqVo);
                break;
            case ENABLE:
                this.enableFeeBudget(reqVo);
                break;
            case DISABLE:
                this.disableFeeBudget(reqVo);
                break;
            case CHANGE:
                this.adjustOrChangeFeeBudget(reqVo);
                break;
            case ADJUST_OUT:
                this.adjustOrChangeFeeBudget(reqVo);
                break;
            case ADJUST_IN:
                this.adjustOrChangeFeeBudget(reqVo);
                break;
            case USE:
                this.useOrReturnFeeBudget(reqVo);
                break;
            case RETURN:
                this.useOrReturnFeeBudget(reqVo);
                break;
            default:
                throw new BusinessException("尚未定义的费用维度操作类型");
        }
        //操作预算的时候,只需要对控制维度加锁就行了  // TODO: 2020/9/28
    }

    /**
     * 活动使用费用预算时
     *
     * @param reqVo
     */
    private void useOrReturnFeeBudget(OperateBudgetControlReqVo reqVo) {
        this.updateControl(reqVo.getControlEntity(), BigDecimal.ZERO, BigDecimal.ZERO);
    }

    /**
     * 调入,调出,变更时,费用预算更新维度
     *
     * @param reqVo
     */
    private void adjustOrChangeFeeBudget(OperateBudgetControlReqVo reqVo) {
        TpmFeeBudgetEntity entity = reqVo.getEntity();
        AssertUtils.isNotEmpty(entity.getControlId(), "费用预算编码:" + entity.getFeeBudgetCode() + "未关联到控制维度");
        TpmFeeBudgetControlEntity controlEntity = controlMapper.selectById(entity.getControlId());
        AssertUtils.isNotNull(controlEntity, "费用预算编码:" + entity.getFeeBudgetCode() + "关联的控制维度不存在");
        this.updateControl(controlEntity, BigDecimal.ZERO, reqVo.getAfterChangeCanUseAmount());
    }

    /**
     * 启用费用预算
     *
     * @param reqVo
     */
    private void enableFeeBudget(OperateBudgetControlReqVo reqVo) {
        TpmFeeBudgetEntity entity = reqVo.getEntity();
        AssertUtils.isNotEmpty(entity.getControlId(), "费用预算编码:" + entity.getFeeBudgetCode() + "未关联到控制维度");
        TpmFeeBudgetControlEntity controlEntity = controlMapper.selectById(entity.getControlId());
        AssertUtils.isNotNull(controlEntity, "费用预算编码:" + entity.getFeeBudgetCode() + "关联的控制维度不存在");
        this.updateControl(controlEntity, BigDecimal.ZERO, entity.getCanUseAmount());
    }

    /**
     * 禁用费用预算
     *
     * @param reqVo
     */
    private void disableFeeBudget(OperateBudgetControlReqVo reqVo) {
        TpmFeeBudgetEntity entity = reqVo.getEntity();
        AssertUtils.isNotEmpty(entity.getControlId(), "费用预算编码:" + entity.getFeeBudgetCode() + "未关联到控制维度");
        TpmFeeBudgetControlEntity controlEntity = controlMapper.selectById(entity.getControlId());
        AssertUtils.isNotNull(controlEntity, "费用预算编码:" + entity.getFeeBudgetCode() + "关联的控制维度不存在");
        this.updateControl(controlEntity, entity.getCanUseAmount(), BigDecimal.ZERO);
    }

    /**
     * 获取控制维度
     *
     * @param reqVo
     * @return
     */
    public String getSelectKey(TpmFeeBudgetReqVo reqVo) {
        String controlType = reqVo.getBudgetSubjectsControlType();
        StringBuffer selectKey = new StringBuffer();
        selectKey.append(reqVo.getBudgetYear());
        if (StringUtils.equals(BudgetSubjectsControlTypeEnum.YEAR.getCode(), controlType)) {
            //全年控制
        } else if (StringUtils.equals(BudgetSubjectsControlTypeEnum.QUATER.getCode(), controlType)) {
            //季度控制
            selectKey.append("-").append(reqVo.getBudgetQuater());
        } else {
            //不控制和按月控制
            if (StringUtils.isEmpty(reqVo.getBudgetMonth())) {
                selectKey.append("-").append(reqVo.getBudgetQuater());
            } else {
                selectKey.append("-").append(reqVo.getBudgetQuater()).append("-").append(reqVo.getBudgetMonth());
            }
        }
        return selectKey.toString();
    }

    /**
     * 获取控制维度查询参数
     *
     * @param feeBudgetControls
     * @param obj
     * @return
     */
    public Map<String, Object> convertDataToMap(Set<String> feeBudgetControls, Object obj) {
        JSONObject jsonObject = JSON.parseObject(JSON.toJSONString(obj));
        Map<String, Object> map = Maps.newHashMap();
        if (CollectionUtils.isNotEmpty(feeBudgetControls)) {
            feeBudgetControls.stream().forEach(o -> {
                String s = com.biz.crm.util.StringUtils.camelCaseName(o);
                Object x = jsonObject.get(s);
                if (Objects.isNull(x)) {
                    map.put(o, null);
                } else {
                    map.put(o, x);
                }
            });
        }
        return map;
    }

    /**
     * 通过控制维度查询控制维度数据
     *
     * @param map
     * @return
     */
    public TpmFeeBudgetControlEntity findFeeBudgetControlByParams(Map<String, Object> map) {
        QueryWrapper<TpmFeeBudgetControlEntity> wrapper = new QueryWrapper<>();
        AssertUtils.isTrue(MapUtils.isNotEmpty(map), "查询费用预算控制维度参数不能为空");
        for (String s : map.keySet()) {
            Object o = map.get(s);
            if (Objects.isNull(o)) {
                wrapper.isNull(s);
            } else {
                wrapper.eq(s, o);
            }
        }
        List<TpmFeeBudgetControlEntity> entities = controlMapper.selectList(wrapper);
        AssertUtils.isTrue(CollectionUtils.size(entities) <= 1, "当前控制维度的数据存在多条,请联系管理员");
        return CollectionUtils.isEmpty(entities) ? null : entities.get(0);
    }

    /**
     * 获取费用预算类型对应的控制维度
     *
     * @param feeBudgetType
     * @return
     */
    public Set<String> getFeeBudgetControls(String feeBudgetType, String controlType) {
        List<DictDataVo> childrenList = DictUtil.getChildrenList(TpmGlobalDictConstants.FEE_BUDGET_TYPE, feeBudgetType);
        AssertUtils.isNotEmpty(childrenList, "当前费用预算类型未配置控制维度");
        Set<String> set = Sets.newHashSet();
        childrenList.forEach(o -> {
            Map<String, String> extendMap = o.getExtendMap();
            AssertUtils.isNotEmpty(extendMap, "当前费用预算类型控制维度未设置属性字段");
            String s = Lists.newArrayList(extendMap.values()).get(0);
            AssertUtils.isNotEmpty(s, "当前费用预算类型控制维度设置的属性值异常,请检查");
            AssertUtils.isTrue(!set.contains(s), "当前费用预算类型控制维度设置的属性值重复");
//            //根据控制类型决定要不要加上年季度月份的条件
//            if(StringUtils.equals(BudgetSubjectsControlTypeEnum.NON.getCode(),controlType)||StringUtils.equals(BudgetSubjectsControlTypeEnum.MONTH.getCode(),controlType)){
//                if(StringUtils.equals(TpmGlobalDictConstants.BUDGET_YEAR,s)){
//                    set.add(s);
//                }
//                if(StringUtils.equals(TpmGlobalDictConstants.BUDGET_MONTH,s)){
//                    set.add(s);
//                }
//                if(StringUtils.equals(TpmGlobalDictConstants.BUDGET_QUATER,s)){
//                    set.add(s);
//                }
//            }else if(StringUtils.equals(BudgetSubjectsControlTypeEnum.QUATER.getCode(),controlType)){
//                if(StringUtils.equals(TpmGlobalDictConstants.BUDGET_YEAR,s)){
//                    set.add(s);
//                }
//                if(StringUtils.equals(TpmGlobalDictConstants.BUDGET_QUATER,s)){
//                    set.add(s);
//                }
//            }
//            if(StringUtils.equals(BudgetSubjectsControlTypeEnum.YEAR.getCode(),controlType)){
//                if(StringUtils.equals(TpmGlobalDictConstants.BUDGET_YEAR,s)){
//                    set.add(s);
//                }
//            }
            set.add(s);
        });
        return set;
    }

    /**
     * 检查费用预算划拨
     *
     * @param reqVos
     */
    public void checkDistribution(List<TpmFeeBudgetDistributionReqVo> reqVos) {
        Optional<TpmFeeBudgetDistributionReqVo> optional = reqVos.stream().filter(o -> o.getFeeBudgetCode() != null).findFirst();
        AssertUtils.isTrue(optional.isPresent(), "缺少费用预算编码");
        TpmFeeBudgetEntity entity = mapper.selectOne(Wrappers.lambdaQuery(TpmFeeBudgetEntity.class)
                .eq(TpmFeeBudgetEntity::getFeeBudgetCode, optional.get().getFeeBudgetCode()));
        BigDecimal all = BigDecimal.ZERO;
        List<MdmOrgRespVo> mdmOrgRespVos = new ArrayList<>();
        List<MdmTerminalVo> terminalVos = new ArrayList<>();
        List<MdmCustomerMsgSelectRespVo> selectRespVos = new ArrayList<>();
        setCheckNeedInfo(entity, mdmOrgRespVos, selectRespVos, terminalVos);
        for (TpmFeeBudgetDistributionReqVo vo : reqVos) {
            AssertUtils.isTrue(StringUtils.equals(vo.getFeeBudgetCode(), entity.getFeeBudgetCode()), "出现多个费用预算编码");
            if (entity.getFeeBudgetType().equals(FeeBudgetTypeEnum.DEPARTMENT.getCode())) {
                checkOrg(mdmOrgRespVos, vo);
            } else if (entity.getFeeBudgetType().equals(FeeBudgetTypeEnum.CUSTOMER.getCode())) {
                setOrgProperty(entity, vo);
                AssertUtils.isFalse(entity.getCustomerCode().equals(vo.getCustomerCode()), "客户与费用预算客户相同");
                checkCustomer(selectRespVos, vo);
            } else if (entity.getFeeBudgetType().equals(FeeBudgetTypeEnum.TERMINAL.getCode())) {
                setOrgProperty(entity, vo);
                vo.setCustomerCode(entity.getCustomerCode());
                vo.setCustomerCode(entity.getCustomerName());
                AssertUtils.isFalse(entity.getTerminalCode().equals(vo.getTerminalCode()), "门店与费用预算门店相同");
                checkTerminal(terminalVos, vo);
            } else {
                AssertUtils.isTrue(DictUtil.getDictValueMapsByCodes(TpmGlobalDictConstants.CHANNEL).containsKey(vo.getChannel()), "字典未找到相应渠道");
                checkOrg(mdmOrgRespVos, vo);
            }
            all = all.add(vo.getDistributionAmount());
            vo.setParentId(entity.getId());
        }
        AssertUtils.isTrue(all.compareTo(entity.getCanUseAmount()) <= 0, "总划拨金额大于可用余额");
    }

    private void setCheckNeedInfo(TpmFeeBudgetEntity entity, List<MdmOrgRespVo> mdmOrgRespVos, List<MdmCustomerMsgSelectRespVo> selectRespVos, List<MdmTerminalVo> terminalVos) {
        if (entity.getFeeBudgetType().equals(FeeBudgetTypeEnum.DEPARTMENT.getCode()) || entity.getFeeBudgetType().equals(FeeBudgetTypeEnum.CHANNEL.getCode())) {
            MdmOrgReqVo mdmOrgReqVo = new MdmOrgReqVo();
            mdmOrgReqVo.setOrgCode(entity.getOrgCode());
            mdmOrgRespVos.addAll(ApiResultUtil.objResult(orgFeign.findSingleChildrenOrgList(mdmOrgReqVo)));
        } else if (entity.getFeeBudgetType().equals(FeeBudgetTypeEnum.CUSTOMER.getCode())) {
            MdmCustomerMsgReqVo mdmCustomerMsgReqVo = new MdmCustomerMsgReqVo();
            mdmCustomerMsgReqVo.setOrgCodeList(Lists.newArrayList(entity.getOrgCode()));
            selectRespVos.addAll(ApiResultUtil.objResult(customerFeign.findCustomerSelectList(mdmCustomerMsgReqVo)));
        } else {
            MdmCustomerTerminalVo terminalVo = new MdmCustomerTerminalVo();
            terminalVo.setCustomerCodeList(Lists.newArrayList(entity.getCustomerCode()));
            terminalVos.addAll(ApiResultUtil.objResult(terminalFeign.customerTerminalList(terminalVo)));
        }
    }

    private void checkCustomer(List<MdmCustomerMsgSelectRespVo> selectRespVos, TpmFeeBudgetDistributionReqVo vo) {
        Optional<MdmCustomerMsgSelectRespVo> optional = selectRespVos.stream().filter(o -> o.getCustomerCode().equals(vo.getCustomerCode())).findAny();
        AssertUtils.isTrue(optional.isPresent(), "划拨门店不是费用预算客户的关联门店");
        vo.setCustomerName(optional.get().getCustomerName());
        vo.setCustomerOrgCode(optional.get().getCustomerOrgCode());
        vo.setCustomerOrgName(optional.get().getCustomerOrgName());
    }

    private void checkTerminal(List<MdmTerminalVo> terminalVos, TpmFeeBudgetDistributionReqVo vo) {
        Optional<MdmTerminalVo> optional = terminalVos.stream().filter(o -> o.getTerminalCode().equals(vo.getTerminalCode())).findAny();
        AssertUtils.isTrue(optional.isPresent(), "划拨门店不是费用预算客户的关联门店");
        vo.setTerminalName(optional.get().getTerminalName());
    }

    private void setOrgProperty(TpmFeeBudgetEntity entity, TpmFeeBudgetDistributionReqVo vo) {
        vo.setOrgCode(entity.getOrgCode());
        vo.setOrgName(entity.getOrgName());
        vo.setOrgType(entity.getOrgType());
    }

    private void checkOrg(List<MdmOrgRespVo> mdmOrgRespVos, TpmFeeBudgetDistributionReqVo vo) {
        Optional<MdmOrgRespVo> optional = mdmOrgRespVos.stream().filter(o -> o.getOrgCode().equals(vo.getOrgCode())).findAny();
        AssertUtils.isTrue(optional.isPresent(), "划拨组织不是费用预算组织的直接下级");
        vo.setOrgName(optional.get().getOrgName());
        vo.setOrgType(optional.get().getOrgType());
    }
}
