package com.biz.crm.feebudget.service.impl;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.mapper.BaseMapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.annotation.Klock;
import com.biz.crm.base.BusinessException;
import com.biz.crm.base.config.ThreadLocalUtil;
import com.biz.crm.budgetgeneraterule.mapper.TpmBudgetGenerateRuleMapper;
import com.biz.crm.budgetgeneraterule.model.TpmBudgetGenerateRuleEntity;
import com.biz.crm.budgetsubjects.mapper.TpmBudgetSubjectsMapper;
import com.biz.crm.budgetsubjects.model.TpmBudgetSubjectsEntity;
import com.biz.crm.common.GlobalParam;
import com.biz.crm.common.PageResult;
import com.biz.crm.crmlog.handle.util.CrmLogSendUtil;
import com.biz.crm.eunm.CrmEnableStatusEnum;
import com.biz.crm.eunm.YesNoEnum;
import com.biz.crm.eunm.tpm.BudgetSubjectsControlTypeEnum;
import com.biz.crm.eunm.tpm.FeeBudgetDetailTypeEnum;
import com.biz.crm.exception.CommonException;
import com.biz.crm.exception.tpm.FeeBudgetException;
import com.biz.crm.feebudget.mapper.TpmFeeBudgetControlMapper;
import com.biz.crm.feebudget.mapper.TpmFeeBudgetDetailsMapper;
import com.biz.crm.feebudget.mapper.TpmFeeBudgetMapper;
import com.biz.crm.feebudget.model.OperateBudgetControlReqVo;
import com.biz.crm.feebudget.model.TpmFeeBudgetControlEntity;
import com.biz.crm.feebudget.model.TpmFeeBudgetDetailsEntity;
import com.biz.crm.feebudget.model.TpmFeeBudgetEntity;
import com.biz.crm.feebudget.service.ITpmFeeBudgetControlService;
import com.biz.crm.feebudget.service.ITpmFeeBudgetService;
import com.biz.crm.nebular.tpm.feebudget.req.FeeBudgetControlOperateTypeEnum;
import com.biz.crm.nebular.tpm.feebudget.req.TpmFeeBudgetDistributionReqVo;
import com.biz.crm.nebular.tpm.feebudget.req.TpmFeeBudgetReqVo;
import com.biz.crm.nebular.tpm.feebudget.resp.TpmFeeBudgetDistributionRespVo;
import com.biz.crm.nebular.tpm.feebudget.resp.TpmFeeBudgetRespVo;
import com.biz.crm.util.*;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.math.BigDecimal;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.stream.Collectors;

/**
 * 费用预算 接口实现
 *
 * @author huanglong
 * @date 2020-09-14 17:11:59
 */
@Slf4j
@Service
@ConditionalOnMissingBean(name="TpmFeeBudgetServiceExpandImpl")
public class TpmFeeBudgetServiceImpl<M extends BaseMapper<T>,T> extends ServiceImpl<TpmFeeBudgetMapper, TpmFeeBudgetEntity> implements ITpmFeeBudgetService {

    @Resource
    private TpmFeeBudgetMapper tpmFeeBudgetMapper;
    @Autowired
    private FeeBudgetServiceHelper serviceHelper;
    @Resource
    private TpmFeeBudgetDetailsMapper detailsMapper;
    @Resource
    private TpmFeeBudgetControlMapper controlMapper;
    @Autowired
    private ITpmFeeBudgetControlService controlService;
    @Autowired
    private CrmLogSendUtil crmLogSendUtil;
    @Resource
    private TpmBudgetSubjectsMapper budgetSubjectsMapper;
    @Resource
    private TpmBudgetGenerateRuleMapper budgetGenerateRuleMapper;
    /**
     * 列表
     * @param reqVo
     * @return
     */
    @Override
    public PageResult<TpmFeeBudgetRespVo> findList(TpmFeeBudgetReqVo reqVo){
        Page<TpmFeeBudgetRespVo> page = PageUtil.buildPage(reqVo.getPageNum(), reqVo.getPageSize());
        reqVo.setSelectedCodeList(serviceHelper.dealSelectedCodeList(reqVo.getSelectedCode(),reqVo.getSelectedCodeList()));
        List<TpmFeeBudgetRespVo> list = tpmFeeBudgetMapper.findList(page, reqVo);
        serviceHelper.convertListDate(list);
        return PageResult.<TpmFeeBudgetRespVo>builder()
                .data(list)
                .count(page.getTotal())
                .build();
    }

    /**
     * 查询
     * @param id
     * @return tpmFeeBudgetRespVo
     */
    @Override
    public TpmFeeBudgetRespVo query(String id){
        AssertUtils.isNotEmpty(id,"id不能为空");
        TpmFeeBudgetReqVo reqVo = new TpmFeeBudgetReqVo();
        reqVo.setId(id);
        List<TpmFeeBudgetRespVo> list = this.findList(reqVo).getData();
        if(CollectionUtils.isEmpty(list)){
//            throw new BusinessException(FeeBudgetException.DATA_NOT_EXIST);
            return new TpmFeeBudgetRespVo();
        }
        list.get(0).setBudgetSubjectsControlType(list.get(0).getControlType());
        return list.get(0);
    }

    /**
     * 新增
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    @Klock(keys = {"fee","#reqVo.lockUserName"},waitTime = 0,leaseTime = 5)
    public void save(TpmFeeBudgetReqVo reqVo){
        if(reqVo.getIsIncomeGenerate()){
            //根据收入预算生成时,维度重复只会跳过保存,不抛错
            if(!serviceHelper.incomeGenerateCheck(reqVo)){
                return;
            }
        }else {
            serviceHelper.saveCheck(reqVo);
        }
        TpmFeeBudgetEntity entity = CrmBeanUtil.copy(reqVo,TpmFeeBudgetEntity.class);
        entity.setCanUseAmount(entity.getInitAmount());
        entity.setUsedAmount(BigDecimal.ZERO);
        entity.setAfterAdjustAmount(entity.getInitAmount());
        entity.setAdjustTotalAmount(BigDecimal.ZERO);
        //保存明细数据
        serviceHelper.createOrUpdateSaveDetail(entity,reqVo);
        //保存控制维度数据
        OperateBudgetControlReqVo controlReqVo = OperateBudgetControlReqVo.builder().entity(entity).reqVo(reqVo).typeEnum(FeeBudgetControlOperateTypeEnum.NEW).build();
        serviceHelper.saveFeeBudgetControlData(controlReqVo);
        this.save(entity);
        //日志
        Object menuCodeObj = ThreadLocalUtil.getObj(GlobalParam.MENU_CODE);
        TpmFeeBudgetRespVo newData = CrmBeanUtil.copy(entity, TpmFeeBudgetRespVo.class);
        crmLogSendUtil.sendForAdd(menuCodeObj.toString(),newData.getId(),newData.getFeeBudgetCode(),newData);

    }

    /**
     * 更新
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    @Klock(keys = {"fee","#reqVo.lockUserName"},waitTime = 0,leaseTime = 5)
    public void update(TpmFeeBudgetReqVo reqVo){
        if(StringUtils.isEmpty(reqVo.getId())){
            throw new BusinessException("数据主键不能为空");
        }
        TpmFeeBudgetEntity entity = this.getById(reqVo.getId());
        BigDecimal oldAmount = entity.getInitAmount();
        //日志老数据
        TpmFeeBudgetRespVo oldData = CrmBeanUtil.copy(entity, TpmFeeBudgetRespVo.class);
        AssertUtils.isNotNull(entity,FeeBudgetException.DATA_NOT_EXIST);
        //备注,金额,编码不能编辑,所以把原数据的三个值复制到vo
        reqVo.setFeeBudgetCode(entity.getFeeBudgetCode());
        reqVo.setControlId(entity.getControlId());
//        reqVo.setRemarks(entity.getRemarks());
        reqVo.setFeeBudgetCode(entity.getFeeBudgetCode());
        reqVo.setCanUseAmount(entity.getCanUseAmount());
        //查询费用预算明细,如果明细数量大于1,则不允许编辑
        Integer count = detailsMapper.getDetailCountByFeeBudgetCode(entity.getFeeBudgetCode());
        AssertUtils.isTrue(count == 0,FeeBudgetException.NOT_ALLOW_EDIT);
        serviceHelper.saveCheck(reqVo);
        CrmBeanUtil.copyProperties(reqVo,entity);
        entity.setCanUseAmount(entity.getInitAmount());
        entity.setUsedAmount(BigDecimal.ZERO);
        entity.setAfterAdjustAmount(entity.getInitAmount());
        entity.setAdjustTotalAmount(BigDecimal.ZERO);
        //保存明细数据
        serviceHelper.createOrUpdateSaveDetail(entity,reqVo);
        //保存控制维度数据
        OperateBudgetControlReqVo controlReqVo = OperateBudgetControlReqVo.builder().entity(entity).reqVo(reqVo).typeEnum(FeeBudgetControlOperateTypeEnum.UPDATE).build();
        serviceHelper.saveFeeBudgetControlData(controlReqVo);
        //修改操作
        this.updateById(entity);
        Object menuCodeObj = ThreadLocalUtil.getObj(GlobalParam.MENU_CODE);
        TpmFeeBudgetRespVo newData = CrmBeanUtil.copy(entity, TpmFeeBudgetRespVo.class);
        //发送修改日志
        crmLogSendUtil.sendForUpdate(menuCodeObj.toString(),newData.getId(),newData.getFeeBudgetCode(),oldData,newData);
    }

    /**
     * 删除
     * @param ids id集合
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteBatch(List<String> ids){
            AssertUtils.isNotEmpty(ids, CommonException.IDS_NULL);
            //如果明细里面只有一条数据,就可以删
            List<TpmFeeBudgetRespVo> list = tpmFeeBudgetMapper.countDetailsNum(ids);
            AssertUtils.isNotEmpty(list,"您选择的费用预算已被删除");
            Map<String,BigDecimal> map = Maps.newHashMap();
            //费用预算控制维度id对应的删除的费用预算编码集合的map
           Map<String,List<String>> controlAndFeeBudgetCodeMap = Maps.newHashMap();
            list.forEach(o->{
                AssertUtils.isTrue(o.getNum()<=1,"费用预算:"+o.getFeeBudgetCode()+",已产生明细数据,不能被删除");
                AssertUtils.isNotEmpty(o.getControlId(),"费用预算:"+o.getFeeBudgetCode()+",数据异常,未关联控制维度");
                BigDecimal bigDecimal = map.get(o.getControlId());
                if(Objects.isNull(bigDecimal)){
                    controlAndFeeBudgetCodeMap.put(o.getControlId(), Lists.newArrayList(o.getFeeBudgetCode()));
                    map.put(o.getControlId(),Optional.ofNullable(o.getCanUseAmount()).orElse(BigDecimal.ZERO));
                }else {
                    List<String> strings = controlAndFeeBudgetCodeMap.get(o.getControlId());
                    strings.add(o.getFeeBudgetCode());
                    controlAndFeeBudgetCodeMap.put(o.getControlId(),strings);
                    map.put(o.getControlId(),bigDecimal.add(Optional.ofNullable(o.getCanUseAmount()).orElse(BigDecimal.ZERO)));
                }
            });
            //查询费用预算维度
            //在删除费用预算的同时需要把控制维度的数据的可用余额减少
        List<TpmFeeBudgetControlEntity> controlEntities = controlMapper.selectList(new LambdaQueryWrapper<TpmFeeBudgetControlEntity>().in(TpmFeeBudgetControlEntity::getId, map.keySet()));
        if(CollectionUtils.isNotEmpty(controlEntities)){
            controlEntities.forEach(o->{
                BigDecimal bigDecimal = map.get(o.getId());
                List<String> feeBudgetCodes = controlAndFeeBudgetCodeMap.get(o.getId());
                feeBudgetCodes.forEach(p->{
                    String s = serviceHelper.replaceFeeBudgetCodes(o.getFeeBudgetCodes(), p);
                    o.setFeeBudgetCodes(s);
                });
                OperateBudgetControlReqVo controlReqVo = OperateBudgetControlReqVo.builder().controlEntity(o).typeEnum(FeeBudgetControlOperateTypeEnum.DELETE).controlUpdateAmount(bigDecimal).build();
                serviceHelper.saveFeeBudgetControlData(controlReqVo);
            });
        }
        //日志老数据
        PageResult<TpmFeeBudgetRespVo> oldDataList = this.findList(new TpmFeeBudgetReqVo().setIds(ids));
        List<TpmFeeBudgetRespVo> listData = oldDataList.getData();
        listData.forEach(respVo -> {
            LambdaQueryWrapper<TpmFeeBudgetEntity> wrapper = Wrappers.lambdaQuery(TpmFeeBudgetEntity.class)
                    .eq(TpmFeeBudgetEntity::getBudgetYear, respVo.getBudgetYear())
                    .eq(TpmFeeBudgetEntity::getBudgetMonth, respVo.getBudgetMonth())
                    .eq(TpmFeeBudgetEntity::getFeeBudgetType, respVo.getFeeBudgetType())
                    .notIn(TpmFeeBudgetEntity::getId, listData.stream().map(TpmFeeBudgetRespVo::getId).collect(Collectors.toList()));
            //如果该维度下已经没有其他的费用预算了，那就查找费用生成规则的数据，更新他的生成标记
            if(tpmFeeBudgetMapper.selectCount(wrapper) == 0){
                LambdaQueryWrapper<TpmBudgetGenerateRuleEntity> ruleWrapper = Wrappers.lambdaQuery(TpmBudgetGenerateRuleEntity.class)
                        .eq(TpmBudgetGenerateRuleEntity::getYear, respVo.getBudgetYear())
                        .eq(TpmBudgetGenerateRuleEntity::getMonth, respVo.getBudgetMonth())
                        .eq(TpmBudgetGenerateRuleEntity::getFeeBudgetType, respVo.getFeeBudgetType());
                TpmBudgetGenerateRuleEntity ruleEntity = budgetGenerateRuleMapper.selectOne(ruleWrapper);
                if(ruleEntity != null){
                    ruleEntity.setIsGenerate(YesNoEnum.yesNoEnum.ZERO.getValue());
                    budgetGenerateRuleMapper.updateById(ruleEntity);
                }
            }
        });
        if(CollectionUtils.isNotEmpty(listData)){
            List<String> codes = listData.stream().map(TpmFeeBudgetRespVo::getFeeBudgetCode).collect(Collectors.toList());
            //删除操作
            tpmFeeBudgetMapper.deleteBatchIds(ids);
            detailsMapper.delete(Wrappers.lambdaQuery(TpmFeeBudgetDetailsEntity.class).in(TpmFeeBudgetDetailsEntity::getFeeBudgetCode,codes));
            Object menuCodeObj = ThreadLocalUtil.getObj(GlobalParam.MENU_CODE);
            //发送删除日志
            for (TpmFeeBudgetRespVo oldData:oldDataList.getData()) {
                crmLogSendUtil.sendForDel(menuCodeObj.toString(),oldData.getId(),oldData.getFeeBudgetCode(),oldData);
            }
        }

    }

    /**
     * 启用
     * @param ids id集合
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void enableBatch(List<String> ids){
        //设置状态为启用
            AssertUtils.isNotEmpty(ids, CommonException.IDS_NULL);
            List<TpmFeeBudgetEntity> entities = tpmFeeBudgetMapper.selectBatchIds(ids);
        //日志老数据
        List<TpmFeeBudgetRespVo> oldDataList = CrmBeanUtil.copyList(entities, TpmFeeBudgetRespVo.class);
        if (CollectionUtils.isNotEmpty(entities)) {
                entities.forEach(o -> {
                    if(!StringUtils.equals(CrmEnableStatusEnum.ENABLE.getCode(),o.getEnableStatus())){
                        OperateBudgetControlReqVo controlReqVo = OperateBudgetControlReqVo.builder().entity(o).typeEnum(FeeBudgetControlOperateTypeEnum.ENABLE).build();
                        serviceHelper.saveFeeBudgetControlData(controlReqVo);
                    }
                    o.setEnableStatus(CrmEnableStatusEnum.ENABLE.getCode());

                });
            }
            //修改操作
            this.updateBatchById(entities);
            //获取菜单码
            Object menuCodeObj = ThreadLocalUtil.getObj(GlobalParam.MENU_CODE);
            List<TpmFeeBudgetRespVo> newDataList = CrmBeanUtil.copyList(entities, TpmFeeBudgetRespVo.class);
            for (int i = 0;i < newDataList.size();i++) {
                crmLogSendUtil.sendForUpdate(menuCodeObj.toString(),newDataList.get(i).getId(),newDataList.get(i).getBudgetSubjectsCode(),oldDataList.get(i),newDataList.get(i));
            }
    }

    /**
     * 禁用
     * @param ids id集合
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void disableBatch(List<String> ids){
        //设置状态为禁用
            AssertUtils.isNotEmpty(ids, CommonException.IDS_NULL);
            List<TpmFeeBudgetEntity> entities = tpmFeeBudgetMapper.selectBatchIds(ids);
            //日志老数据
            List<TpmFeeBudgetRespVo> oldDataList = CrmBeanUtil.copyList(entities, TpmFeeBudgetRespVo.class);
            if (CollectionUtils.isNotEmpty(entities)) {
                entities.forEach(o -> {
                    if(!StringUtils.equals(CrmEnableStatusEnum.DISABLE.getCode(),o.getEnableStatus())){
                        OperateBudgetControlReqVo controlReqVo = OperateBudgetControlReqVo.builder().entity(o).typeEnum(FeeBudgetControlOperateTypeEnum.DISABLE).build();
                        serviceHelper.saveFeeBudgetControlData(controlReqVo);
                    }
                    o.setEnableStatus(CrmEnableStatusEnum.DISABLE.getCode());
                });
            }
        this.updateBatchById(entities);
        List<TpmFeeBudgetRespVo> newDataList = CrmBeanUtil.copyList(entities, TpmFeeBudgetRespVo.class);
        //获取菜单码
        Object menuCodeObj = ThreadLocalUtil.getObj(GlobalParam.MENU_CODE);
        for (int i = 0;i < newDataList.size();i++) {
            crmLogSendUtil.sendForUpdate(menuCodeObj.toString(),newDataList.get(i).getId(),newDataList.get(i).getBudgetSubjectsCode(),oldDataList.get(i),newDataList.get(i));
        }
    }

    /**
     * 变更
     * @param reqVo
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    @Klock(keys = {"fee","#reqVo.lockUserName"},waitTime = 0,leaseTime = 5)
    public void change(TpmFeeBudgetReqVo reqVo) {
        //1、变更金额必须是正数；
        //2、追加时用可用余额+变更金额
        TpmFeeBudgetEntity budgetEntity = tpmFeeBudgetMapper.selectOne(new LambdaQueryWrapper<TpmFeeBudgetEntity>().eq(TpmFeeBudgetEntity::getId, reqVo.getId()));
        AssertUtils.isNotNull(budgetEntity,"预算数据不存在,请刷新重试");
        reqVo.setBudgetSubjectsCode(budgetEntity.getBudgetSubjectsCode());
        serviceHelper.changeCheck(reqVo);
        //日志老数据
        TpmFeeBudgetRespVo oldData = CrmBeanUtil.copy(budgetEntity,TpmFeeBudgetRespVo.class);
        BigDecimal decimal=BigDecimal.ZERO;
        if(StringUtils.equals(FeeBudgetDetailTypeEnum.CUT_OUT.getCode(),reqVo.getFeeBudgetDetailType())){
            decimal = budgetEntity.getCanUseAmount().subtract(reqVo.getChangeAmount());
            //检查变更后的金额不能小于0(如果预算科目的控制类型是不控制金额可以小于0)
            if(decimal.compareTo(BigDecimal.ZERO)<0 && !StringUtils.equals(BudgetSubjectsControlTypeEnum.NON.getCode(),reqVo.getBudgetSubjectsControlType())){
                throw new BusinessException(FeeBudgetException.AMOUNT_LT_ZERO);
            }
        }else {
            decimal = budgetEntity.getCanUseAmount().add(reqVo.getChangeAmount());
        }
        OperateBudgetControlReqVo controlReqVo = OperateBudgetControlReqVo.builder().entity(budgetEntity).typeEnum(FeeBudgetControlOperateTypeEnum.CHANGE).afterChangeCanUseAmount(decimal.subtract(budgetEntity.getCanUseAmount())).build();
        serviceHelper.saveFeeBudgetControlData(controlReqVo);
        budgetEntity.setCanUseAmount(decimal);
        serviceHelper.changeOrAdjustSaveDetail(budgetEntity,reqVo);
        //计算调整金额,调整后金额
        serviceHelper.computeAmount(budgetEntity,reqVo);
        //修改操作
        this.updateById(budgetEntity);
        //接入工作流 TODO
        Object menuCodeObj = ThreadLocalUtil.getObj(GlobalParam.MENU_CODE);
        TpmFeeBudgetRespVo newData = CrmBeanUtil.copy(budgetEntity, TpmFeeBudgetRespVo.class);
        crmLogSendUtil.sendForUpdate(menuCodeObj.toString(),newData.getId(),newData.getFeeBudgetCode(),oldData,newData);
    }

    /**
     * 调整
     * @param reqVo
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    @Klock(keys = {"fee","#reqVo.lockUserName"},waitTime = 0,leaseTime = 5)
    public void adjust(TpmFeeBudgetReqVo reqVo) {
        /*1、选择调入方时需要除开当前调出方；
        2、保存时校验：调出方的调整后余额不能为负（保存时计算一次，不能用页面上计算的数据）
        3、调出金额必须是正数*/
        serviceHelper.adjustCheck(reqVo);
        TpmFeeBudgetEntity adjustOutEntity = tpmFeeBudgetMapper.selectOne(new LambdaQueryWrapper<TpmFeeBudgetEntity>().eq(TpmFeeBudgetEntity::getId, reqVo.getId()));
        AssertUtils.isNotNull(adjustOutEntity,"调出方数据不存在");
        //查询预算科目
        TpmBudgetSubjectsEntity subjectsEntity = budgetSubjectsMapper.selectOne(Wrappers.lambdaQuery(TpmBudgetSubjectsEntity.class).eq(TpmBudgetSubjectsEntity::getBudgetSubjectsCode, adjustOutEntity.getBudgetSubjectsCode()));
        AssertUtils.isNotNull(subjectsEntity,"调出方对应的预算科目:"+adjustOutEntity.getBudgetSubjectsCode()+",不存在,无法执行预算调整操作");
        if (!BudgetSubjectsControlTypeEnum.NON.getCode().equals(subjectsEntity.getControlType())){
            AssertUtils.isTrue(adjustOutEntity.getCanUseAmount().compareTo(BigDecimal.ZERO)==1,"调出方可用余额必须大于0");
        }
        TpmFeeBudgetEntity adjustInEntity = tpmFeeBudgetMapper.selectOne(new LambdaQueryWrapper<TpmFeeBudgetEntity>().eq(TpmFeeBudgetEntity::getId, reqVo.getAdjustInFeeBudgetId()));
        AssertUtils.isNotNull(adjustInEntity,"调入方数据不存在");
        //日志老数据
        TpmFeeBudgetRespVo oldAdjustOut = CrmBeanUtil.copy(adjustOutEntity, TpmFeeBudgetRespVo.class);
        TpmFeeBudgetRespVo oldAdjustIn = CrmBeanUtil.copy(adjustInEntity, TpmFeeBudgetRespVo.class);
        //调出方可用余额减少
        BigDecimal outDecimal = adjustOutEntity.getCanUseAmount().subtract(reqVo.getChangeAmount());
        //检查变更后的金额不能小于0(如果预算科目的控制类型是不控制金额可以小于0)
        if(outDecimal.compareTo(BigDecimal.ZERO)<0&& !StringUtils.equals(BudgetSubjectsControlTypeEnum.NON.getCode(),subjectsEntity.getControlType())){
            throw new BusinessException(FeeBudgetException.AMOUNT_LT_ZERO);
        }
        //更新调出方控制维度数据
        OperateBudgetControlReqVo controlReqVo = OperateBudgetControlReqVo.builder().entity(adjustOutEntity).typeEnum(FeeBudgetControlOperateTypeEnum.ADJUST_OUT).afterChangeCanUseAmount(BigDecimal.ZERO.subtract(reqVo.getChangeAmount())).build();
        serviceHelper.saveFeeBudgetControlData(controlReqVo);
        adjustOutEntity.setCanUseAmount(outDecimal);
        //保存调出方明细
        reqVo.setAdjustInFeeBudgetCode(adjustInEntity.getFeeBudgetCode());
        reqVo.setFeeBudgetDetailType(FeeBudgetDetailTypeEnum.ADJUST_OUT.getCode());
        serviceHelper.changeOrAdjustSaveDetail(adjustOutEntity,reqVo);
        //计算调出方调整金额,调整后金额
        serviceHelper.computeAmount(adjustOutEntity,reqVo);
        this.updateById(adjustOutEntity);
        //调入方可用余额增加
        BigDecimal inDecimal = adjustInEntity.getCanUseAmount().add(reqVo.getChangeAmount());
        //更新调入方控制维度数据
        OperateBudgetControlReqVo controlReqVo1 = OperateBudgetControlReqVo.builder().entity(adjustInEntity).typeEnum(FeeBudgetControlOperateTypeEnum.ADJUST_IN).afterChangeCanUseAmount(reqVo.getChangeAmount()).build();
        serviceHelper.saveFeeBudgetControlData(controlReqVo1);
        adjustInEntity.setCanUseAmount(inDecimal);
        //保存调入方明细
        reqVo.setFeeBudgetDetailType(FeeBudgetDetailTypeEnum.ADJUST_IN.getCode());
        serviceHelper.changeOrAdjustSaveDetail(adjustInEntity,reqVo);
        //计算调入方调整金额,调整后金额
        serviceHelper.computeAmount(adjustInEntity,reqVo);
        this.updateById(adjustInEntity);
        //接入工作流 TODO
        //日志新数据
        TpmFeeBudgetRespVo newAdjustIn = CrmBeanUtil.copy(adjustInEntity, TpmFeeBudgetRespVo.class);
        TpmFeeBudgetRespVo newAdjustOut = CrmBeanUtil.copy(adjustOutEntity, TpmFeeBudgetRespVo.class);
        Object menuCodeObj =  ThreadLocalUtil.getObj(GlobalParam.MENU_CODE);
        crmLogSendUtil.sendForUpdate(menuCodeObj.toString(),newAdjustIn.getId(),newAdjustIn.getFeeBudgetCode(),oldAdjustIn,newAdjustIn);
        crmLogSendUtil.sendForUpdate(menuCodeObj.toString(),newAdjustOut.getId(),newAdjustOut.getFeeBudgetCode(),oldAdjustOut,newAdjustOut);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void saveDistribution(List<TpmFeeBudgetDistributionReqVo> reqVos) {
        if(CollectionUtil.listNotEmpty(reqVos)) {
            serviceHelper.checkDistribution(reqVos);
            BigDecimal all = BigDecimal.ZERO;
            TpmFeeBudgetEntity tpmFeeBudgetEntity = tpmFeeBudgetMapper.selectOne(Wrappers.lambdaQuery(TpmFeeBudgetEntity.class).eq(TpmFeeBudgetEntity::getFeeBudgetCode, reqVos.get(0).getFeeBudgetCode()));
            //新增费用预算
            for (TpmFeeBudgetDistributionReqVo e : reqVos) {
                TpmFeeBudgetReqVo reqVo = CrmBeanUtil.copy(tpmFeeBudgetEntity, TpmFeeBudgetReqVo.class);
                reqVo.setLockUserName(UserUtils.getUser().getUsername());
                reqVo.setFeeBudgetDetailType(FeeBudgetDetailTypeEnum.DISTRIBUTION.getCode());
                all = all.add(e.getDistributionAmount());
                CrmBeanUtil.copyProperties(e, reqVo);
                reqVo.setId(null);
                reqVo.setFeeBudgetCode(null);
                reqVo.setInitAmount(e.getDistributionAmount());
                save(reqVo);
            }
            tpmFeeBudgetEntity.setCanUseAmount(tpmFeeBudgetEntity.getCanUseAmount().subtract(all));
            tpmFeeBudgetMapper.updateById(tpmFeeBudgetEntity);
            TpmFeeBudgetDetailsEntity detailsEntity = new TpmFeeBudgetDetailsEntity();
            CrmBeanUtil.copyProperties(tpmFeeBudgetEntity, detailsEntity);
            detailsEntity.setId(null);
            detailsEntity.setBusinessCode(detailsEntity.getFeeBudgetCode());
            detailsEntity.setFeeBudgetDetailType(FeeBudgetDetailTypeEnum.DISTRIBUTION.getCode());
            detailsEntity.setFeeBudgetDetailTypeName(FeeBudgetDetailTypeEnum.DISTRIBUTION.getDes());
            detailsEntity.setMonth(tpmFeeBudgetEntity.getBudgetMonth());
            detailsEntity.setYear(tpmFeeBudgetEntity.getBudgetYear());
            detailsEntity.setAfterAmount(tpmFeeBudgetEntity.getCanUseAmount());
            detailsEntity.setFeeAmount(all);
            detailsEntity.setBeforAmount(tpmFeeBudgetEntity.getCanUseAmount().add(all));
            detailsMapper.insert(detailsEntity);
        }
    }
}
