package com.biz.crm.feewithholding.service.impl;

import com.baomidou.mybatisplus.core.mapper.BaseMapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.act.mapper.TpmActDetailMapper;
import com.biz.crm.base.BusinessException;
import com.biz.crm.base.config.ThreadLocalUtil;
import com.biz.crm.common.GlobalParam;
import com.biz.crm.common.PageResult;
import com.biz.crm.common.TpmGlobalDictConstants;
import com.biz.crm.crmlog.handle.util.CrmLogSendUtil;
import com.biz.crm.eunm.CodeRuleEnum;
import com.biz.crm.eunm.CrmEnableStatusEnum;
import com.biz.crm.eunm.tpm.SendStatusEnum;
import com.biz.crm.eunm.tpm.WithholdingTypeEnum;
import com.biz.crm.feewithholding.mapper.TpmFeeWithholdingMapper;
import com.biz.crm.feewithholding.model.TpmFeeWithholdingEntity;
import com.biz.crm.feewithholding.service.ITpmFeeWithholdingService;
import com.biz.crm.nebular.tpm.act.req.TpmFeeWithholdingActReqVo;
import com.biz.crm.nebular.tpm.act.resp.TpmFeeWithholdingActRespVo;
import com.biz.crm.nebular.tpm.feewithholding.req.TpmFeeWithholdingReqVo;
import com.biz.crm.nebular.tpm.feewithholding.resp.TpmFeeWithholdingRespVo;
import com.biz.crm.util.*;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 接口实现
 *
 * @author gavin
 * @date 2021-12-15 15:36:08
 */
@Slf4j
@Service
@ConditionalOnMissingBean(name = "TpmFeeWithholdingServiceExpandImpl")
public class TpmFeeWithholdingServiceImpl<M extends BaseMapper<T>, T> extends ServiceImpl<TpmFeeWithholdingMapper, TpmFeeWithholdingEntity> implements ITpmFeeWithholdingService {

    @Resource
    private TpmFeeWithholdingMapper tpmFeeWithholdingMapper;

    @Resource
    private TpmActDetailMapper tpmActDetailMapper;

    @Resource
    private CrmLogSendUtil crmLogSendUtil;

    public Map<String, Map<String, String>> getDicts(){
        List<String> dictCodes= Lists.newArrayList();
        dictCodes.add(TpmGlobalDictConstants.WITHHOLDING_TYPE);
        dictCodes.add(TpmGlobalDictConstants.SEND_STATUS);
        //查询字典数据
        return DictUtil.getDictValueMapsByCodes(dictCodes);
    }

    /**
     * 列表
     *
     * @param reqVo
     * @return
     */
    @Override
    public PageResult<TpmFeeWithholdingRespVo> findList(TpmFeeWithholdingReqVo reqVo) {
        Page<TpmFeeWithholdingRespVo> page = PageUtil.buildPage(reqVo.getPageNum(), reqVo.getPageSize());
        List<TpmFeeWithholdingRespVo> list = tpmFeeWithholdingMapper.findList(page, reqVo);
        convertListDate(list);
        return PageResult.<TpmFeeWithholdingRespVo>builder()
                .data(list)
                .count(page.getTotal())
                .build();
    }

    public void convertListDate(List<TpmFeeWithholdingRespVo> list) {
        if(CollectionUtils.isNotEmpty(list)){
            //获取需要的字典集合
            Map<String, Map<String, String>> map = this.getDicts();
            list.forEach(o->{
                //启用禁用状态
                if(StringUtils.isNotEmpty(o.getEnableStatus())){
                    o.setEnableStatusName(Optional.ofNullable(map.get(TpmGlobalDictConstants.ENABLE_STATUS)).orElse(Maps.newHashMap()).get(o.getEnableStatus()));
                }
                //预提类型
                if(StringUtils.isNotEmpty(o.getWithholdingType())){
                    o.setWithholdingTypeName(Optional.ofNullable(map.get(TpmGlobalDictConstants.WITHHOLDING_TYPE)).orElse(Maps.newHashMap()).get(o.getWithholdingType()));
                }
                //发送状态
                if(StringUtils.isNotEmpty(o.getSendStatus())){
                    o.setSendStatusName(Optional.ofNullable(map.get(TpmGlobalDictConstants.SEND_STATUS)).orElse(Maps.newHashMap()).get(o.getSendStatus()));
                }
            });
        }
    }

    /**
     * 查询
     *
     * @param id
     * @return tpmFeeWithholdingRespVo
     */
    @Override
    public TpmFeeWithholdingRespVo query(String id) {
        TpmFeeWithholdingRespVo respVo = CrmBeanUtil.copy(tpmFeeWithholdingMapper.findById(id), TpmFeeWithholdingRespVo.class);
        convertListDate(Lists.newArrayList(respVo));
        return respVo;
    }

    /**
     * 新增
     *
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void auto(TpmFeeWithholdingReqVo reqVo) {
        Date date;
        try {
            date = DateUtil.yyyy_MM.parse(reqVo.getYearAndMonth());
        }catch (Exception e){
            throw new BusinessException("时间格式错误");
        }
        AssertUtils.isFalse(DateUtil.monthNotBeforeNow(date), "只能预提当前月以前的活动");
        List<TpmFeeWithholdingActRespVo> respVos = tpmActDetailMapper
                .forFeeWithholding(TpmFeeWithholdingActReqVo.builder().feeDateStr(reqVo.getYearAndMonth()).build());
        //更新的预提
        List<TpmFeeWithholdingEntity> updateData = new ArrayList<>();
        //新增的预提
        List<TpmFeeWithholdingEntity> addData = new ArrayList<>();
        //查询活动明细编码存在的费用预提
        Map<String, TpmFeeWithholdingEntity> entityMap = CollectionUtil.listNotEmpty(respVos) ? searchExist(respVos.stream().map(TpmFeeWithholdingActRespVo::getActDetailCode).collect(Collectors.toList())) : new HashMap<>();
        respVos.forEach(o -> {
            TpmFeeWithholdingEntity entity;
            if ((entity = entityMap.get(o.getActDetailCode())) != null) {
                if (SendStatusEnum.NOT_SEND.getCode().equals(entity.getSendStatus())) {
                    TpmFeeWithholdingRespVo oldData = CrmBeanUtil.copy(entity, TpmFeeWithholdingRespVo.class);
                    entity.setWithholdingAmount(o.getApplyAmount().subtract(o.getAccountAmount()));
                    TpmFeeWithholdingRespVo newData = CrmBeanUtil.copy(entity, TpmFeeWithholdingRespVo.class);
                    crmLogSendUtil.sendForUpdate(ThreadLocalUtil.getObj(GlobalParam.MENU_CODE).toString(), entity.getId(), entity.getId(), oldData, newData);
                    updateData.add(entity);
                }
            } else {
                entity = new TpmFeeWithholdingEntity();
                entity.setWithholdingAmount(o.getApplyAmount().subtract(o.getAccountAmount()));
                entity.setYearAndMonth(reqVo.getYearAndMonth());
                entity.setActDetailCode(o.getActDetailCode());
                entity.setWithholdingType(WithholdingTypeEnum.AUTO.getCode());
                addData.add(entity);
            }
        });
        saveOrUpdate(addData, updateData);
    }

    private void saveOrUpdate(List<TpmFeeWithholdingEntity> addData, List<TpmFeeWithholdingEntity> updateData){
        if(updateData.size() == 0 && addData.size() == 0){
            throw new BusinessException("无可新增或更新预提数据");
        }else {
            if(addData.size() > 0) {
                List<String> codes = CodeUtil.generateCodeList(CodeRuleEnum.FEE_WITHHOLDING.getCode(), addData.size());
                for (int i = 0; i < addData.size(); i++) {
                    TpmFeeWithholdingEntity newData = addData.get(i);
                    newData.setSendStatus(SendStatusEnum.NOT_SEND.getCode());
                    newData.setWithholdingCode(codes.get(i));
                    crmLogSendUtil.sendForAdd(ThreadLocalUtil.getObj(GlobalParam.MENU_CODE).toString(), newData.getId(), newData.getId(), CrmBeanUtil.copy(newData, TpmFeeWithholdingRespVo.class));
                }
                saveBatch(addData);
            }
            if(updateData.size() > 0){
                updateBatchById(updateData);
            }
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void handle(List<TpmFeeWithholdingReqVo> reqVos) {
        List<String> actDetailCodeList = reqVos.stream().map(TpmFeeWithholdingReqVo::getActDetailCode).collect(Collectors.toList());
        //查询活动明细数据
        Map<String, TpmFeeWithholdingActRespVo> actRespVoMap = tpmActDetailMapper
                .forFeeWithholding(TpmFeeWithholdingActReqVo.builder().actCodeDetailList(actDetailCodeList).build())
                .stream().collect(Collectors.toMap(TpmFeeWithholdingActRespVo::getActDetailCode, respVo -> respVo));
        //查询活动明细编码存在的费用预提
        Map<String, TpmFeeWithholdingEntity> entityMap = searchExist(actDetailCodeList);
        StringBuilder msg = new StringBuilder();
        //更新的预提
        List<TpmFeeWithholdingEntity> updateData = new ArrayList<>();
        //新增的预提
        List<TpmFeeWithholdingEntity> addData = new ArrayList<>();
        reqVos.forEach(reqVo -> {
            TpmFeeWithholdingEntity entity = CrmBeanUtil.copy(reqVo, TpmFeeWithholdingEntity.class);
            TpmFeeWithholdingActRespVo respVo = actRespVoMap.get(reqVo.getActDetailCode());
            if (respVo == null) {
                msg.append("未找到活动明细:").append(reqVo.getActDetailCode()).append(";");
            } else {
                if (respVo.getApplyAmount().subtract(respVo.getAccountAmount()).compareTo(reqVo.getWithholdingAmount()) < 0) {
                    msg.append("活动:").append(reqVo.getActDetailCode()).append("可预提金额不足;");
                }
                try {
                    if(DateUtil.monthNotBeforeNow(DateUtil.yyyy_MM.parse(respVo.getFeeDateStr()))){
                        msg.append("活动:").append(reqVo.getActDetailCode()).append("不是当前月以前的活动;");
                    }
                }catch (Exception e){
                    msg.append("活动:").append(reqVo.getActDetailCode()).append("费用所属年月时间格式错误;");
                }
            }
            TpmFeeWithholdingEntity checkEntity;
            if ((checkEntity = entityMap.get(reqVo.getActDetailCode())) != null) {
                if (SendStatusEnum.SEND.getCode().equals(checkEntity.getSendStatus())) {
                    return;
                }
                entity.setId(checkEntity.getId());
                updateData.add(entity);
            } else {
                if(respVo != null) {
                    entity.setYearAndMonth(respVo.getFeeDateStr());
                }
                entity.setWithholdingType(WithholdingTypeEnum.HANDLE.getCode());
                entity.setActDetailCode(reqVo.getActDetailCode());
                addData.add(entity);
            }
        });
        if (StringUtils.isEmpty(msg.toString())) {
            saveOrUpdate(addData, updateData);
        } else {
            throw new BusinessException(msg.toString());
        }
    }

    private Map<String, TpmFeeWithholdingEntity> searchExist(List<String> actDetailCodeList) {
        //查询活动明细编码存在的费用预提
        Map<String, TpmFeeWithholdingEntity> entityMap = tpmFeeWithholdingMapper
                .selectList(Wrappers.lambdaQuery(TpmFeeWithholdingEntity.class).in(TpmFeeWithholdingEntity::getActDetailCode, actDetailCodeList))
                .stream()
                .collect(Collectors.toMap(TpmFeeWithholdingEntity::getActDetailCode, o -> o));
        return entityMap;
    }

    /**
     * 更新
     *
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void update(TpmFeeWithholdingReqVo reqVo) {
        TpmFeeWithholdingEntity entity = this.getById(reqVo.getId());
        if(SendStatusEnum.SEND.getCode().equals(entity.getSendStatus())){
            throw new BusinessException("费用预提已推送");
        }
        List<TpmFeeWithholdingActRespVo> list = tpmActDetailMapper
                .forFeeWithholding(TpmFeeWithholdingActReqVo.builder().actCodeDetailList(Lists.newArrayList(entity.getActDetailCode())).build());
        if(list.size() == 0){
            throw new BusinessException("对应活动明细不存在");
        }
        TpmFeeWithholdingActRespVo actRespVo = list.get(0);
        if(reqVo.getWithholdingAmount().compareTo(actRespVo.getApplyAmount().subtract(actRespVo.getAccountAmount())) > 0){
            throw new BusinessException("活动可预提金额不足");
        }
        entity.setWithholdingAmount(reqVo.getWithholdingAmount());
        this.updateById(entity);
    }

    /**
     * 删除
     *
     * @param ids
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteBatch(List<String> ids) {
        if (CollectionUtils.isNotEmpty(ids)) {
           tpmFeeWithholdingMapper.deleteBatchIds(ids);
        }
    }

    /**
     * 启用
     *
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void enableBatch(TpmFeeWithholdingReqVo reqVo) {
        //设置状态为启用
        List<TpmFeeWithholdingEntity> tpmFeeWithholdingEntities = tpmFeeWithholdingMapper.selectBatchIds(reqVo.getIds());
        if (CollectionUtils.isNotEmpty(tpmFeeWithholdingEntities)) {
            tpmFeeWithholdingEntities.forEach(o -> {
                o.setEnableStatus(CrmEnableStatusEnum.ENABLE.getCode());
            });
        }
        this.updateBatchById(tpmFeeWithholdingEntities);
    }

    /**
     * 禁用
     *
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void disableBatch(TpmFeeWithholdingReqVo reqVo) {
        //设置状态为禁用
        List<TpmFeeWithholdingEntity> tpmFeeWithholdingEntities = tpmFeeWithholdingMapper.selectBatchIds(reqVo.getIds());
        if (CollectionUtils.isNotEmpty(tpmFeeWithholdingEntities)) {
            tpmFeeWithholdingEntities.forEach(o -> {
                o.setEnableStatus(CrmEnableStatusEnum.DISABLE.getCode());
            });
        }
        this.updateBatchById(tpmFeeWithholdingEntities);
    }

    @Override
    public List<TpmFeeWithholdingActRespVo> findActDetailByFeeWithholding(List<String> actCodeList) {
        return tpmActDetailMapper.forFeeWithholding(TpmFeeWithholdingActReqVo.builder().actCodeList(actCodeList).feeDateStrEnd(DateUtil.getFormatDateStr(new Date(), DateUtil.DEFAULT_MONTH_DAY_PATTERN_)).build());
    }
}
