package com.biz.crm.fiscalyear.service.impl;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.biz.crm.base.BusinessException;
import com.biz.crm.common.TpmGlobalDictConstants;
import com.biz.crm.eunm.tpm.FiscalHalfYearEnum;
import com.biz.crm.exception.tpm.FiscalYearSettingException;
import com.biz.crm.fiscalyear.mapper.TpmFiscalYearSettingMapper;
import com.biz.crm.fiscalyear.model.TpmFiscalYearSettingEntity;
import com.biz.crm.nebular.tpm.fiscalyear.req.TpmFiscalYearSettingReqVo;
import com.biz.crm.nebular.tpm.fiscalyear.resp.TpmFiscalYearSettingRespVo;
import com.biz.crm.util.AssertUtils;
import com.biz.crm.util.DateUtil;
import com.biz.crm.util.DictUtil;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.util.*;

/**
 * @Project crm
 * @PackageName com.biz.crm.fiscalyear.service.impl
 * @ClassName FiscalYearSettingServiceHelper
 * @Author HuangLong
 * @Date 2020/9/8 14:02
 * @Description 财年设置工具类
 */
@Slf4j
@Service
public class FiscalYearSettingServiceHelper {

    @Resource
    private TpmFiscalYearSettingMapper mapper;

    public Map<String, Map<String, String>> getDicts(){
        List<String> dictCodes= Lists.newArrayList();
        dictCodes.add(TpmGlobalDictConstants.ENABLE_STATUS);
        dictCodes.add(TpmGlobalDictConstants.FISCAL_MONTH);
        dictCodes.add(TpmGlobalDictConstants.FISCAL_QUATER);
        dictCodes.add(TpmGlobalDictConstants.FISCAL_YEAR);
        dictCodes.add(TpmGlobalDictConstants.FISCAL_HALF_YEAR);
        //查询字典数据
        return DictUtil.getDictValueMapsByCodes(dictCodes);
    }

    /**
     * 转换列表数据
     * @param list
     */
    public void convertListDate(List<TpmFiscalYearSettingRespVo> list) {
        if(CollectionUtils.isNotEmpty(list)){
            //获取需要的字典集合
            Map<String, Map<String, String>> map = this.getDicts();
            list.forEach(o->{
                //启用禁用状态
                if(StringUtils.isNotEmpty(o.getEnableStatus())){
                    o.setEnableStatusName(Optional.ofNullable(map.get(TpmGlobalDictConstants.ENABLE_STATUS)).orElse(Maps.newHashMap()).get(o.getEnableStatus()));
                }
                if(StringUtils.isNotEmpty(o.getYear())){
                    o.setYearName(Optional.ofNullable(map.get(TpmGlobalDictConstants.FISCAL_YEAR)).orElse(Maps.newHashMap()).get(o.getYear()));
                }
                if(StringUtils.isNotEmpty(o.getMonth())){
                    o.setMonthName(Optional.ofNullable(map.get(TpmGlobalDictConstants.FISCAL_MONTH)).orElse(Maps.newHashMap()).get(o.getMonth()));
                }
                if(StringUtils.isNotEmpty(o.getHalfYear())){
                    o.setHalfYearName(Optional.ofNullable(map.get(TpmGlobalDictConstants.FISCAL_HALF_YEAR)).orElse(Maps.newHashMap()).get(o.getHalfYear()));
                }
            });
        }
    }
    /**
     * 通过查询条件获取wrapper
     * @return
     */
    public LambdaQueryWrapper<TpmFiscalYearSettingEntity> getQueryWrapper(TpmFiscalYearSettingReqVo reqVo){
        LambdaQueryWrapper<TpmFiscalYearSettingEntity> wrapper = new LambdaQueryWrapper<>();
        //预算财年
        wrapper.eq(StringUtils.isNotEmpty(reqVo.getYear()),TpmFiscalYearSettingEntity::getYear,reqVo.getYear());
        //季度
        wrapper.eq(StringUtils.isNotEmpty(reqVo.getQuater()),TpmFiscalYearSettingEntity::getQuater,reqVo.getQuater());
        //月度
        wrapper.eq(StringUtils.isNotEmpty(reqVo.getMonth()),TpmFiscalYearSettingEntity::getMonth,reqVo.getMonth());
        //半年度
        wrapper.eq(StringUtils.isNotEmpty(reqVo.getHalfYear()),TpmFiscalYearSettingEntity::getHalfYear,reqVo.getHalfYear());
        //启用禁用
        wrapper.eq(StringUtils.isNotEmpty(reqVo.getEnableStatus()),TpmFiscalYearSettingEntity::getEnableStatus,reqVo.getEnableStatus());
        return wrapper;
    }
    /**
     * 修改财年数据(启用,禁用)
     * @param reqVo
     * @param entity
     */
    public void updateBySelectAll(TpmFiscalYearSettingReqVo reqVo,TpmFiscalYearSettingEntity entity){
        LambdaQueryWrapper<TpmFiscalYearSettingEntity> productQueryWrapper = this.getQueryWrapper(reqVo);
        //TODO  这里必须加上数据权限的限制!!!!
        mapper.update(entity, productQueryWrapper);
    }
    /**
     * 新增,编辑,校验
     * @param reqVo
     */
    public void saveCheck(TpmFiscalYearSettingReqVo reqVo){
        //校验数据不为空
        AssertUtils.isNotEmpty(reqVo.getMonth(),"月份不能为空");
        AssertUtils.isNotEmpty(reqVo.getYear(),"预算财年不能为空");
        AssertUtils.isNotEmpty(reqVo.getQuater(),"季度不能为空");
        AssertUtils.isNotEmpty(reqVo.getBeginDate(),"开始时间不能为空");
        AssertUtils.isNotEmpty(reqVo.getEndDate(),"结束时间不能为空");
        if(StringUtils.isEmpty(reqVo.getEndDateSecond())){
            reqVo.setEndDateSecond(DateUtil.DAY_LATEST_TIME);
        }
        if(StringUtils.isEmpty(reqVo.getBeginDateSecond())){
            reqVo.setBeginDateSecond(DateUtil.DAY_EARLIEST_TIME);
        }
        //根据月份来给半年度属性赋值（Month>=7 为下半年 否则为上半年）
        if (Integer.valueOf(reqVo.getMonth()) >= 7){
            reqVo.setHalfYear(FiscalHalfYearEnum.DOWN_YEAR.getCode());
        }else {
            reqVo.setHalfYear(FiscalHalfYearEnum.UP_YEAR.getCode());
        }
        //开始时间不能大于结束时间
        AssertUtils.isTrue(StringUtils.compare(reqVo.getBeginDate()+reqVo.getBeginDateSecond(),reqVo.getEndDate()+reqVo.getEndDateSecond())<0,"开始时间不能大于结束时间");
        this.checkDataExist(reqVo);
    }
    /**
     * 校验数据是否已经存在
     * @param reqVo
     * @return
     */
    public void checkDataExist(TpmFiscalYearSettingReqVo reqVo ) {
        LambdaQueryWrapper<TpmFiscalYearSettingEntity> wrapper = new LambdaQueryWrapper<TpmFiscalYearSettingEntity>().eq(TpmFiscalYearSettingEntity::getYear,reqVo.getYear())
                .eq(TpmFiscalYearSettingEntity::getQuater,reqVo.getQuater()).eq(TpmFiscalYearSettingEntity::getMonth,reqVo.getMonth()).eq(TpmFiscalYearSettingEntity::getHalfYear,reqVo.getHalfYear());
        if(StringUtils.isNotEmpty(reqVo.getId())){
            wrapper.ne(TpmFiscalYearSettingEntity::getId,reqVo.getId());
        }
        List<TpmFiscalYearSettingEntity> list = mapper.selectList(wrapper);
        AssertUtils.isTrue(CollectionUtils.isEmpty(list),"预算财年+月度数据已存在，请重新填写");
        //取出数据库所有的数据(财年数据每年只会有几十条数据,这里全部取出来)
        List<TpmFiscalYearSettingEntity> allList = mapper.selectList(new LambdaQueryWrapper<TpmFiscalYearSettingEntity>().ne(StringUtils.isNotEmpty(reqVo.getId()),TpmFiscalYearSettingEntity::getId,reqVo.getId()));
        if(CollectionUtils.isNotEmpty(allList)){
            //如果时间存在交集,则报错
            String nowBeginDateStr=reqVo.getBeginDate()+reqVo.getBeginDateSecond();
            String nowEndDateStr=reqVo.getEndDate()+reqVo.getEndDateSecond();
            for (TpmFiscalYearSettingEntity o : allList){

                String oldBeginDateStr=o.getBeginDate()+o.getBeginDateSecond();
                String oldEndDateStr=o.getEndDate()+o.getEndDateSecond();
                    //分三种情况: 1:新数据的开始时间在老数据的范围内,
                    // 2,新数据的结束时间在老数据的范围内,
                    // 3新数据的开始小于老数据的开始时间,并且新数据的结束时间大于老数据的结束时间
                Boolean flag=false;
                if(StringUtils.compare(nowBeginDateStr,oldBeginDateStr)>=0&&StringUtils.compare(nowBeginDateStr,oldEndDateStr)<=0){
                    flag=true;

                }
                if(StringUtils.compare(nowEndDateStr,oldBeginDateStr)>=0&&StringUtils.compare(nowEndDateStr,oldEndDateStr)<=0){
                    flag=true;

                }
                if(StringUtils.compare(nowBeginDateStr,oldBeginDateStr)<0&&StringUtils.compare(nowEndDateStr,oldEndDateStr)>0){
                    flag=true;

                }
                if(flag){
                    throw new BusinessException("开始时间+结束时间已存在预算财年数据，请重新填写");
                }
            }
        }
    }

}
