package com.biz.crm.invoicepool.service.impl;

import com.baomidou.mybatisplus.core.mapper.BaseMapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.base.BusinessException;
import com.biz.crm.base.config.ThreadLocalUtil;
import com.biz.crm.common.GlobalParam;
import com.biz.crm.common.PageResult;
import com.biz.crm.common.TpmGlobalDictConstants;
import com.biz.crm.crmlog.handle.util.CrmLogSendUtil;
import com.biz.crm.eunm.CrmDelFlagEnum;
import com.biz.crm.eunm.CrmEnableStatusEnum;
import com.biz.crm.invoicepool.mapper.TpmInvoicePoolMapper;
import com.biz.crm.invoicepool.model.TpmInvoicePoolEntity;
import com.biz.crm.invoicepool.service.ITpmInvoicePoolService;
import com.biz.crm.nebular.tpm.invoicepool.req.TpmInvoicePoolReqVo;
import com.biz.crm.nebular.tpm.invoicepool.resp.TpmInvoicePoolRespVo;
import com.biz.crm.util.*;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.math.BigDecimal;
import java.util.List;
import java.util.Map;
import java.util.Optional;

/**
 * 接口实现
 *
 * @author gavin
 * @date 2021-12-16 16:29:22
 */
@Slf4j
@Service
@ConditionalOnMissingBean(name="TpmInvoicePoolServiceExpandImpl")
public class TpmInvoicePoolServiceImpl<M extends BaseMapper<T>,T> extends ServiceImpl<TpmInvoicePoolMapper, TpmInvoicePoolEntity> implements ITpmInvoicePoolService {

    @Resource
    private TpmInvoicePoolMapper tpmInvoicePoolMapper;

    @Resource
    private CrmLogSendUtil crmLogSendUtil;

    @Autowired
    private TpmInvoiceHelper invoiceHelper;

    /**
     * 列表
     * @param reqVo
     * @return
     */
    @Override
    public PageResult<TpmInvoicePoolRespVo> findList(TpmInvoicePoolReqVo reqVo){
        Page<TpmInvoicePoolRespVo> page = PageUtil.buildPage(reqVo.getPageNum(), reqVo.getPageSize());
        List<TpmInvoicePoolRespVo> list = tpmInvoicePoolMapper.findList(page, reqVo);
        convertData(list);
        return PageResult.<TpmInvoicePoolRespVo>builder()
                .data(list)
                .count(page.getTotal())
                .build();
    }

    private void convertData(List<TpmInvoicePoolRespVo> list){
        if(CollectionUtils.isNotEmpty(list)){
            //获取需要的字典集合
            Map<String, Map<String, String>> map = DictUtil.getDictValueMapsByCodes(Lists.newArrayList(TpmGlobalDictConstants.ENABLE_STATUS, TpmGlobalDictConstants.INVOICE_TYPE));
            list.forEach(o->{
                //启用禁用状态
                if(StringUtils.isNotEmpty(o.getEnableStatus())){
                    o.setEnableStatusName(Optional.ofNullable(map.get(TpmGlobalDictConstants.ENABLE_STATUS)).orElse(Maps.newHashMap()).get(o.getEnableStatus()));
                }
                //预提类型
                if(StringUtils.isNotEmpty(o.getInvoiceType())){
                    o.setInvoiceTypeName(Optional.ofNullable(map.get(TpmGlobalDictConstants.INVOICE_TYPE)).orElse(Maps.newHashMap()).get(o.getInvoiceType()));
                }
            });
        }
    }

    /**
     * 查询
     * @param id
     * @return tpmInvoicePoolRespVo
     */
    @Override
    public TpmInvoicePoolRespVo query(String id, String invoiceNumber){
        if(id != null){
            return CrmBeanUtil.copy(tpmInvoicePoolMapper.selectById(id), TpmInvoicePoolRespVo.class);
        }
        if(invoiceNumber != null){
            return CrmBeanUtil.copy(tpmInvoicePoolMapper.selectOne(Wrappers.lambdaQuery(TpmInvoicePoolEntity.class).eq(TpmInvoicePoolEntity::getInvoiceNumber, invoiceNumber)), TpmInvoicePoolRespVo.class);
        }
        return null;
    }

    /**
     * 新增
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void save(TpmInvoicePoolReqVo reqVo){
        invoiceHelper.checkInvoicePool(reqVo);
        TpmInvoicePoolEntity entity = CrmBeanUtil.copy(reqVo,TpmInvoicePoolEntity.class);
        entity.setUsedAmount(entity.getTotalAmount());
        entity.setAvailableAmount(BigDecimal.ZERO);
        this.save(entity);
        TpmInvoicePoolRespVo newData = CrmBeanUtil.copy(entity, TpmInvoicePoolRespVo.class);
        crmLogSendUtil.sendForAdd(ThreadLocalUtil.getObj(GlobalParam.MENU_CODE).toString(), newData.getId(), newData.getId(), newData);
    }

    /**
     * 更新
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void update(TpmInvoicePoolReqVo reqVo){
        invoiceHelper.checkInvoice(reqVo);
        TpmInvoicePoolEntity entity = this.getById(reqVo.getId());
        TpmInvoicePoolRespVo oldData = CrmBeanUtil.copy(entity, TpmInvoicePoolRespVo.class);
        CrmBeanUtil.copyProperties(reqVo, entity);
        NumberUtil.checkNumber(reqVo.getUsedAmount(), "使用金额", BigDecimal.ZERO, null);
        NumberUtil.checkNumber(reqVo.getAvailableAmount(), "可用金额", BigDecimal.ZERO, null);
        AssertUtils.isTrue(reqVo.getUsedAmount().add(reqVo.getAvailableAmount()).compareTo(reqVo.getTotalAmount()) == 0, "税价合计应等于使用金额加上可用余额");
        TpmInvoicePoolRespVo newData = CrmBeanUtil.copy(entity, TpmInvoicePoolRespVo.class);
        this.updateById(entity);
        crmLogSendUtil.sendForUpdate(ThreadLocalUtil.getObj(GlobalParam.MENU_CODE).toString(), newData.getId(), newData.getId(), oldData, newData);
    }

    /**
     * 删除
     * @param ids
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteBatch(List<String> ids){
        if(CollectionUtils.isNotEmpty(ids)){
           tpmInvoicePoolMapper.deleteBatchIds(ids);
        }
    }

    /**
     * 启用
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void enableBatch(TpmInvoicePoolReqVo reqVo){
        //设置状态为启用
        List<TpmInvoicePoolEntity> tpmInvoicePoolEntities = tpmInvoicePoolMapper.selectBatchIds(reqVo.getIds());
        if(CollectionUtils.isNotEmpty(tpmInvoicePoolEntities)){
            tpmInvoicePoolEntities.forEach(o -> {
                o.setEnableStatus(CrmEnableStatusEnum.ENABLE.getCode());
            });
        }
        this.updateBatchById(tpmInvoicePoolEntities);
    }

    /**
     * 禁用
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void disableBatch(TpmInvoicePoolReqVo reqVo){
        //设置状态为禁用
        List<TpmInvoicePoolEntity> tpmInvoicePoolEntities = tpmInvoicePoolMapper.selectBatchIds(reqVo.getIds());
        if(CollectionUtils.isNotEmpty(tpmInvoicePoolEntities)){
                tpmInvoicePoolEntities.forEach(o -> {
                o.setEnableStatus(CrmEnableStatusEnum.DISABLE.getCode());
            });
        }
        this.updateBatchById(tpmInvoicePoolEntities);
    }
}
