package com.biz.crm.liqueuract.service.impl.actbuilder;

import com.biz.crm.CrmCodeRuleConstants;
import com.biz.crm.act.service.impl.ActServiceHelper;
import com.biz.crm.base.BusinessException;
import com.biz.crm.common.TpmGlobalDictConstants;
import com.biz.crm.costtypecategories.model.TpmCostTypeCategoriesEntity;
import com.biz.crm.eunm.GlobalWhetherEnum;
import com.biz.crm.eunm.tpm.*;
import com.biz.crm.liqueuract.model.*;
import com.biz.crm.liqueuract.service.impl.TpmLiqueurActRegisterServiceHelper;
import com.biz.crm.nebular.tpm.liqueuract.req.TpmLiqueurActRegisterDetailBudgetReqVo;
import com.biz.crm.nebular.tpm.liqueuract.req.TpmLiqueurActRegisterDetailReqVo;
import com.biz.crm.nebular.tpm.liqueuract.req.TpmLiqueurActRegisterReqVo;
import com.biz.crm.util.AssertUtils;
import com.biz.crm.util.CodeUtil;
import com.biz.crm.util.CrmBeanUtil;
import com.biz.crm.util.DateUtil;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.collections4.MapUtils;
import org.apache.commons.lang3.StringUtils;

import java.math.BigDecimal;
import java.util.*;
import java.util.concurrent.atomic.AtomicReference;
import java.util.stream.Collectors;

/**
 * @author maoshen
 * @date 2021/3/9.
 */
public class ActRegisterBuilder implements ActRegisterBaseBuilder {

    private TpmLiqueurActRegisterServiceHelper helper;

    private TpmLiqueurActRegisterReqVo reqVo;

    private TpmLiqueurActRegisterEntity actRegisterEntity=new TpmLiqueurActRegisterEntity();

    private TpmLiqueurActEntity actEntity;

    private Set<String> controllerIds ;

    private List<TpmLiqueurActRegisterDetailBudgetEntity> registerDetailBudgetEntities=Lists.newArrayList();

    private List<TpmLiqueurActRegisterDetailEntity> registerDetailEntities=Lists.newArrayList();

    private List<TpmLiqueurActRegisterDetailAttachEntity> registerDetailAttachEntities=Lists.newArrayList();

    private List<TpmLiqueurActBudgetEntity> actBudgetEntities=Lists.newArrayList();

    private List<TpmLiqueurActBudgetTransactionEntity> actBudgetTransactionEntities=Lists.newArrayList();

    private Map<String,List<TpmLiqueurActBudgetEntity>> actBudgetEntitiesMap;

    private Map<String, TpmCostTypeCategoriesEntity> categoriesEntityMap;


    public ActRegisterBuilder(TpmLiqueurActRegisterServiceHelper helper, TpmLiqueurActRegisterReqVo reqVo) {
        this.helper = helper;
        this.reqVo = reqVo;
    }

    public static ActRegisterBuilder builder(TpmLiqueurActRegisterServiceHelper helper, TpmLiqueurActRegisterReqVo reqVo) {
        return new ActRegisterBuilder(helper, reqVo);
    }

    /**
     * 组装数据
     * @return
     */
    @Override
    public ActRegisterBaseBuilder init() {
        AssertUtils.isNotEmpty(reqVo.getActCode(),"活动编码不能为空");
        TpmLiqueurActEntity actEntity = helper.findActEntityByCode(reqVo.getActCode());
        AssertUtils.isNotNull(actEntity,"当前活动申请数据不存在");
        this.actEntity =actEntity;
        //通过活动申请选定的活动类型查询关联的所有投入类型
        this.categoriesEntityMap=helper.findCategoriesMapByFineCode(actEntity.getFineCode());
        //通过活动编码查询活动申请选定的预算数据
        List<TpmLiqueurActBudgetEntity> actBudgetEntities = helper.findActBudgetByActCode(actEntity.getActCode());
        this.actBudgetEntities=actBudgetEntities;
        AssertUtils.isNotEmpty(actBudgetEntities,"活动申请的预算数据异常");
        this.actBudgetEntitiesMap=actBudgetEntities.stream().collect(Collectors.groupingBy(TpmLiqueurActBudgetEntity::getControlId));
        //获取支付方式类型map
        Map<String,String> payTypeTypeMap = helper.getPayTypeTypeMap(TpmGlobalDictConstants.PAY_TYPE,TpmGlobalDictConstants.PAY_TYPE_TYPE);
        if (CollectionUtils.isNotEmpty(reqVo.getDetailVos())){
            reqVo.getDetailVos().stream().forEach(detailVo->{
                if (CollectionUtils.isNotEmpty(detailVo.getAttachVos())){
                    detailVo.getAttachVos().stream().forEach(attachVo->{
                        if(StringUtils.isNotEmpty(attachVo.getPayType())){
                            attachVo.setPayTypeType(payTypeTypeMap.get(attachVo.getPayType()));
                        }
                    });
                }
            });
        }
        return this;
    }

    /**
     * 校验数据
     * @return
     */
    @Override
    public ActRegisterBaseBuilder check() {
        AssertUtils.isNotEmpty(reqVo.getSaveType(), "操作类型不能为空");
        AssertUtils.isNotEmpty(reqVo.getDetailVos(),"请登记活动明细");
        if(ActSaveTypeEnum.getSendApproveTypes().contains(reqVo.getSaveType())){
            AssertUtils.isNotEmpty(reqVo.getWorkFlowKey(),"请选择流程");
        }
        //检验活动申请的状态
        if(!StringUtils.equals(ActApproveStatusEnum.APPROVED.getCode(),this.actEntity.getApproveStatus())){
            throw new BusinessException("活动申请的状态不为审批通过,不允许进行活动登记");
        }
        AssertUtils.isTrue(Optional.ofNullable(this.actEntity.getCanUseAmount()).orElse(BigDecimal.ZERO).compareTo(BigDecimal.ZERO)==1,"当前活动申请可用余额不足,不能进行活动登记");
        Date actEndDate = DateUtil.str2Date(this.actEntity.getEndDate() + " " + this.actEntity.getEndDateSecond(), DateUtil.datetimeFormat);
        AssertUtils.isTrue(        DateUtil.equalIsDateMoreThanAnother(actEndDate,new Date()),"当前活动已结束,不能进行活动登记");
        if (StringUtils.isEmpty(reqVo.getActRegisterCode())) {
            String registerCode = CodeUtil.createOneCode(CrmCodeRuleConstants.ACT_REGISTER);
            reqVo.setActRegisterCode(registerCode);
        }
        reqVo.setTotalAmount(BigDecimal.ZERO);
        reqVo.setActCode(actEntity.getActCode());
        reqVo.setActName(actEntity.getActName());
        reqVo.setFineCode(actEntity.getFineCode());
        reqVo.setFineName(actEntity.getFineName());
        reqVo.getDetailVos().forEach(o->{
            AssertUtils.isNotEmpty(o.getActDetailName(),"请填写活动明细名称");
//            AssertUtils.isNotEmpty(o.getBeginDate(),"请选择开始时间");
//            AssertUtils.isNotEmpty(o.getEndDate(),"请选择结束时间");
//            AssertUtils.isTrue(DateUtil.Date2TimeStamp(o.getBeginDate(),DateUtil.DEFAULT_DATE_FORMAT)<=DateUtil.Date2TimeStamp(o.getEndDate(),DateUtil.DEFAULT_DATE_FORMAT),"开始时间不能大于结束时间");
            AssertUtils.isNotNull(o.getApplyAmount(),"请填写登记金额");
            AssertUtils.isTrue(o.getApplyAmount().compareTo(BigDecimal.ZERO)==1,"活动明细登记金额必须大于0");
            AssertUtils.isNotEmpty(o.getDetailBudgetVos(),"请选择关联预算");
            if (StringUtils.isEmpty(o.getActDetailCode())) {
                String actDetailCode = CodeUtil.createOneCode(CrmCodeRuleConstants.ACT_REGISTER_DETAIL);
                o.setActDetailCode(actDetailCode);
            }
            o.setActCode(reqVo.getActCode());
            o.setActRegisterCode(reqVo.getActRegisterCode());
            reqVo.setTotalAmount(reqVo.getTotalAmount().add(o.getApplyAmount()));
            //校验预算信息
            this.checkDetailBudgets(o);
            //校验附加信息
            if(CollectionUtils.isNotEmpty(o.getAttachVos())){
                this.checkDetailAttaches(o);
            }
        });
        return this;
    }

    /**
     * 校验附加信息
     * @param detailReqVo
     */
    public void checkDetailAttaches(TpmLiqueurActRegisterDetailReqVo detailReqVo) {
        if(CollectionUtils.isNotEmpty(detailReqVo.getAttachVos())){
            //校验附加信息的投入类型是否跟活动明细的预算科目是否有关联,并且投入类型必须是启用的 // TODO: 2021/3/26
            AtomicReference<BigDecimal> attachTotalAmount = new AtomicReference<BigDecimal>(BigDecimal.ZERO);
            detailReqVo.getAttachVos().forEach(o->{
//                AssertUtils.isNotNull(o.getApplyAmount(),"活动明细附加信息申请金额不能为空");
                if(Objects.nonNull(o.getPrice())&&Objects.nonNull(o.getApplyNumber())){
                    //如果单价和数量都有值,但是单价和数量的乘积不等于申请金额,抛错
//                    AssertUtils.isTrue(o.getPrice().multiply(BigDecimal.valueOf(o.getApplyNumber())).compareTo(o.getApplyAmount())==0,"附加信息申请金额必须等于单价和数量的乘积");
                }
                o.setActRegisterCode(reqVo.getActRegisterCode());
                o.setAttachCode(CodeUtil.createOneCode(CrmCodeRuleConstants.ACT_REGISTER_DETAIL_ATTACH));
                o.setActDetailCode(detailReqVo.getActDetailCode());
                attachTotalAmount.getAndAccumulate(Optional.ofNullable(o.getApplyAmount()).orElse(BigDecimal.ZERO),BigDecimal::add);
            });
            if(attachTotalAmount.get().compareTo(detailReqVo.getApplyAmount())==1){
                //附加信息的总申请金额不能大于活动明细的申请金额
                throw new BusinessException("附加信息的总申请金额不能大于活动明细的申请金额");
            }
        }
    }

    /**
     * 校验预算信息
     */
    public void checkDetailBudgets(TpmLiqueurActRegisterDetailReqVo detailReqVo) {
        List<TpmLiqueurActRegisterDetailBudgetReqVo> budgetReqVos = detailReqVo.getDetailBudgetVos();
        AssertUtils.isNotEmpty(budgetReqVos,"活动明细:"+detailReqVo.getActDetailName()+",未选择预算");
        Map<String,Integer> map= Maps.newHashMap();
        Integer order = 0;
        for (TpmLiqueurActRegisterDetailBudgetReqVo o : budgetReqVos) {
            if(StringUtils.isNotEmpty(o.getControlId())){
                if(!map.containsKey(o.getControlId())){
                    map.put(o.getControlId(),order);
                    order++;
                }
            }
        }
        List<Integer> orderList = map.values().stream().sorted().collect(Collectors.toList());
        Map<Integer, String> stringMap = MapUtils.invertMap(map);
        List<String> orderControlIds=Lists.newArrayList();
        orderList.forEach(o->{
            orderControlIds.add(stringMap.get(o));
        });
        AssertUtils.isNotEmpty(orderControlIds,"活动明细选择的费用预算控制维度Id不能为空");
        BigDecimal applyAmount = detailReqVo.getApplyAmount();
        int reduceOrder=1;
        for (String o : orderControlIds) {
            //按照用户选择的预算的顺序依次扣减,如果有没有用到的预算,使用金额设置为0
            List<TpmLiqueurActBudgetEntity> tpmLiqueurActBudgetEntities = actBudgetEntitiesMap.get(o);
            AssertUtils.isNotEmpty(tpmLiqueurActBudgetEntities,"活动明细:"+detailReqVo.getActDetailName()+",对应的费用预算数据异常,请重新选择");
            //把预算按照月份排正序
            List<TpmLiqueurActBudgetEntity> budgetSortedEntities = tpmLiqueurActBudgetEntities.stream().sorted(Comparator.comparing(TpmLiqueurActBudgetEntity::getBudgetMonth)).collect(Collectors.toList());
            AssertUtils.isNotEmpty(tpmLiqueurActBudgetEntities,"活动预算数据异常");
            for(int i= 0;i<budgetSortedEntities.size();i++){
                TpmLiqueurActBudgetEntity p = budgetSortedEntities.get(i);
                TpmLiqueurActRegisterDetailBudgetEntity detailBudgetEntity = new TpmLiqueurActRegisterDetailBudgetEntity();
                detailBudgetEntity.setActBudgetCode(p.getActBudgetCode());
                detailBudgetEntity.setActDetailCode(detailReqVo.getActDetailCode());
                detailBudgetEntity.setActRegisterCode(reqVo.getActRegisterCode());
                detailBudgetEntity.setControlId(p.getControlId());
                detailBudgetEntity.setFeeBudgetCode(p.getFeeBudgetCode());
                detailBudgetEntity.setActCode(p.getActCode());
                detailBudgetEntity.setUsedAmount(BigDecimal.ZERO);
                //提交审批的时候才算占活动预算的钱
                if (ActSaveTypeEnum.getCheckBudgetTypes().contains(reqVo.getSaveType())) {
                    if(p.getCanUseAmount().compareTo(BigDecimal.ZERO)==1){
                        if(applyAmount.compareTo(p.getCanUseAmount())>=0){
                            //申请金额大于预算的可用余额
                            detailBudgetEntity.setUsedAmount(p.getCanUseAmount());
                            applyAmount=applyAmount.subtract(p.getCanUseAmount());
                            p.setCanUseAmount(BigDecimal.ZERO);
                        }else {
                            //申请金额小于预算的可用余额
                            detailBudgetEntity.setUsedAmount(applyAmount);
                            p.setCanUseAmount(p.getCanUseAmount().subtract(applyAmount));
                            applyAmount=BigDecimal.ZERO;
                        }
                    }
                    TpmLiqueurActBudgetTransactionEntity transactionEntity = new TpmLiqueurActBudgetTransactionEntity();
                    transactionEntity.setBusinessCode(reqVo.getActRegisterCode());
                    transactionEntity.setBusinessLineCode(detailReqVo.getActDetailCode());
                    transactionEntity.setControlId(p.getControlId());
                    transactionEntity.setBusinessLineName(detailReqVo.getActDetailName());
                    transactionEntity.setBusinessName(this.actEntity.getActName());
                    transactionEntity.setFeeAmount(BigDecimal.ZERO.subtract(detailBudgetEntity.getUsedAmount()));
                    transactionEntity.setActBudgetCode(p.getActBudgetCode());
                    transactionEntity.setBusinessType(LiqueurActBudgetBusinessTypeEnum.ACT_REGISTER.getCode());
                    transactionEntity.setTransactionType(FeeBudgetDetailTypeEnum.USE.getCode());
                    transactionEntity.setTransactionTypeName(FeeBudgetDetailTypeEnum.USE.getDes());
                    transactionEntity.setBusinessRemarks(LiqueurActBudgetRemarkEnum.ACT_OCCUPIED_BUDGET.getDes());
                    transactionEntity.setFeeBudgetCode(p.getFeeBudgetCode());
                    transactionEntity.setReduceOrder(reduceOrder);
                    reduceOrder++;
                    this.actBudgetTransactionEntities.add(transactionEntity);
                }
                this.registerDetailBudgetEntities.add(detailBudgetEntity);
            }
        }
        if (ActSaveTypeEnum.getCheckBudgetTypes().contains(reqVo.getSaveType())) {
            if(applyAmount.compareTo(BigDecimal.ZERO)>0){
                //如果扣减完之后余额还大于零,则报预算余额不足
                throw new BusinessException("活动明细:"+detailReqVo.getActDetailName()+",超过活动申请可用余额"+",超额部分="+applyAmount.setScale(2,BigDecimal.ROUND_HALF_UP));
            }
        }
    }

    @Override
    public ActRegisterBaseBuilder convert() {
        //转换登记主表
        CrmBeanUtil.copyProperties(reqVo,actRegisterEntity);
        actRegisterEntity.setCostFormCode(actEntity.getCostFormCode());
        actRegisterEntity.setAuditFormCode(actEntity.getAuditFormCode());
        //转换明细数据
        reqVo.getDetailVos().forEach(o->{
            TpmLiqueurActRegisterDetailEntity detailEntity = CrmBeanUtil.copy(o, TpmLiqueurActRegisterDetailEntity.class);
            this.registerDetailEntities.add(detailEntity);
            detailEntity.setAuditAmount(BigDecimal.ZERO);
            detailEntity.setIsAudit(GlobalWhetherEnum.NO.getCode());
            detailEntity.setIsAllAudit(GlobalWhetherEnum.NO.getCode());
            //转换附加信息
            if(CollectionUtils.isNotEmpty(o.getAttachVos())){
                List<TpmLiqueurActRegisterDetailAttachEntity> detailAttachEntities = CrmBeanUtil.copyList(o.getAttachVos(), TpmLiqueurActRegisterDetailAttachEntity.class);
                this.registerDetailAttachEntities.addAll(detailAttachEntities);
            }
        });
        return this;
    }

    @Override
    public TpmLiqueurActRegisterEntity save() {
        helper.saveOrUpdateActRegister(actRegisterEntity, reqVo);
        helper.saveActRegisterDetails(registerDetailEntities,reqVo);
        helper.saveActRegisterAttaches(registerDetailAttachEntities,reqVo);
        helper.saveActRegisterBudgets(registerDetailBudgetEntities,reqVo);
        helper.saveActFeeBudgets(actBudgetEntities,reqVo);
        helper.saveActBudgetTransactions(actBudgetTransactionEntities,reqVo);
        helper.updateActEntity(actEntity,reqVo);
//        int i=1/0;
        return actRegisterEntity;
    }

    @Override
    public TpmLiqueurActRegisterEntity update() {
//        int i=1/0;
        return this.save();
    }
}
