package com.biz.crm.sale.service.impl;

import com.biz.crm.base.config.ThreadLocalUtil;
import com.biz.crm.common.GlobalParam;
import com.biz.crm.crmlog.handle.util.CrmLogSendUtil;
import com.biz.crm.exception.tpm.FeeBudgetException;
import com.biz.crm.nebular.tpm.sale.req.TpmSalesDataReqVo;
import com.biz.crm.nebular.tpm.sale.resp.TpmSalesDataRespVo;
import com.biz.crm.nebular.tpm.sale.resp.TpmSalesPlansRespVo;
import com.biz.crm.sale.mapper.TpmSalesDataMapper;
import com.biz.crm.sale.model.TpmSalesDataEntity;
import com.biz.crm.sale.service.ITpmSalesDataService;
import com.biz.crm.util.AssertUtils;
import com.biz.crm.util.PageUtil;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.stereotype.Service;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.common.PageResult;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.util.CrmBeanUtil;
import org.apache.commons.collections4.CollectionUtils;
import java.util.List;
import lombok.extern.slf4j.Slf4j;
import org.springframework.transaction.annotation.Transactional;
import com.biz.crm.eunm.CrmDelFlagEnum;
import com.biz.crm.eunm.CrmEnableStatusEnum;
import com.baomidou.mybatisplus.core.mapper.BaseMapper;
import javax.annotation.Resource;

/**
 * 实际销售数据 接口实现
 *
 * @author jerry7
 * @date 2021-04-28 15:32:39
 */
@Slf4j
@Service
@ConditionalOnMissingBean(name="TpmSalesDataServiceExpandImpl")
public class TpmSalesDataServiceImpl<M extends BaseMapper<T>,T> extends ServiceImpl<TpmSalesDataMapper, TpmSalesDataEntity> implements ITpmSalesDataService {

    @Resource
    private TpmSalesDataMapper tpmSalesDataMapper;

    @Resource
    private CrmLogSendUtil crmLogSendUtil;

    /**
     * 列表
     * @param reqVo
     * @return
     */
    @Override
    public PageResult<TpmSalesDataRespVo> findList(TpmSalesDataReqVo reqVo){
        Page<TpmSalesDataRespVo> page = PageUtil.buildPage(reqVo.getPageNum(), reqVo.getPageSize());
        List<TpmSalesDataRespVo> list = tpmSalesDataMapper.findList(page, reqVo);
        return PageResult.<TpmSalesDataRespVo>builder()
                .data(list)
                .count(page.getTotal())
                .build();
    }

    /**
     * 查询
     * @param id
     * @return tpmSalesDataRespVo
     */
    @Override
    public TpmSalesDataRespVo query(String id){
        TpmSalesDataEntity tpmSalesDataEntity  = this.getById(id);
        return CrmBeanUtil.copy(tpmSalesDataEntity,TpmSalesDataRespVo.class);
    }

    /**
     * 新增
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void save(TpmSalesDataReqVo reqVo){
        TpmSalesDataEntity entity = CrmBeanUtil.copy(reqVo,TpmSalesDataEntity.class);
        this.save(entity); //日志
        Object menuCodeObj = ThreadLocalUtil.getObj(GlobalParam.MENU_CODE);
        TpmSalesDataRespVo newData = CrmBeanUtil.copy(entity, TpmSalesDataRespVo.class);
        crmLogSendUtil.sendForAdd(menuCodeObj.toString(), newData.getId(), newData.getId(), newData);

    }

    /**
     * 更新
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void update(TpmSalesDataReqVo reqVo){
        TpmSalesDataEntity entity = this.getById(reqVo.getId());
        //日志老数据
        TpmSalesDataRespVo oldData = CrmBeanUtil.copy(entity,  TpmSalesDataRespVo.class);
        AssertUtils.isNotNull(entity, FeeBudgetException.DATA_NOT_EXIST);
        Object menuCodeObj = ThreadLocalUtil.getObj(GlobalParam.MENU_CODE);
        TpmSalesDataRespVo newData = CrmBeanUtil.copy(reqVo,  TpmSalesDataRespVo.class);
        //发送修改日志
        crmLogSendUtil.sendForUpdate(menuCodeObj.toString(), newData.getId(), newData.getId(), oldData, newData);
        this.updateById(entity);
    }

    /**
     * 删除
     * @param ids
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteBatch(List<String> ids){
        List<TpmSalesDataEntity> tpmSalesDataEntities = tpmSalesDataMapper.selectBatchIds(ids);
        if(CollectionUtils.isNotEmpty(tpmSalesDataEntities)){
            tpmSalesDataEntities.forEach(o -> {
                 o.setDelFlag(CrmDelFlagEnum.DELETE.getCode());
            });
        }
        this.updateBatchById(tpmSalesDataEntities);
    }

    /**
     * 启用
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void enableBatch(TpmSalesDataReqVo reqVo){
        //设置状态为启用
        List<TpmSalesDataEntity> tpmSalesDataEntities = tpmSalesDataMapper.selectBatchIds(reqVo.getIds());
        if(CollectionUtils.isNotEmpty(tpmSalesDataEntities)){
            tpmSalesDataEntities.forEach(o -> {
                o.setEnableStatus(CrmEnableStatusEnum.ENABLE.getCode());
            });
        }
        this.updateBatchById(tpmSalesDataEntities);
    }

    /**
     * 禁用
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void disableBatch(TpmSalesDataReqVo reqVo){
        //设置状态为禁用
        List<TpmSalesDataEntity> tpmSalesDataEntities = tpmSalesDataMapper.selectBatchIds(reqVo.getIds());
        if(CollectionUtils.isNotEmpty(tpmSalesDataEntities)){
                tpmSalesDataEntities.forEach(o -> {
                o.setEnableStatus(CrmEnableStatusEnum.DISABLE.getCode());
            });
        }
        this.updateBatchById(tpmSalesDataEntities);
    }
}
