package com.biz.crm.sale.service.impl;

import com.baomidou.mybatisplus.core.mapper.BaseMapper;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.CrmCodeRuleConstants;
import com.biz.crm.base.config.ThreadLocalUtil;
import com.biz.crm.common.GlobalParam;
import com.biz.crm.common.PageResult;
import com.biz.crm.crmlog.handle.util.CrmLogSendUtil;
import com.biz.crm.eunm.CrmDelFlagEnum;
import com.biz.crm.eunm.CrmEnableStatusEnum;
import com.biz.crm.exception.tpm.FeeBudgetException;
import com.biz.crm.nebular.tpm.liqueuraudit.resp.TpmLiqueurAuditRespVo;
import com.biz.crm.nebular.tpm.sale.req.TpmSalesPlansReqVo;
import com.biz.crm.nebular.tpm.sale.resp.TpmSalesPlansRespVo;
import com.biz.crm.sale.mapper.TpmSalesPlansMapper;
import com.biz.crm.sale.model.TpmSalesPlansEntity;
import com.biz.crm.sale.service.ITpmSalesPlansService;
import com.biz.crm.util.AssertUtils;
import com.biz.crm.util.CodeUtil;
import com.biz.crm.util.CrmBeanUtil;
import com.biz.crm.util.PageUtil;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.math.BigDecimal;
import java.util.List;

/**
 * 销售计划 接口实现
 *
 * @author jerry7
 * @date 2021-04-28 15:32:39
 */
@Slf4j
@Service
@ConditionalOnMissingBean(name = "TpmSalesPlansServiceExpandImpl")
public class TpmSalesPlansServiceImpl<M extends BaseMapper<T>, T> extends ServiceImpl<TpmSalesPlansMapper, TpmSalesPlansEntity> implements ITpmSalesPlansService {

    @Resource
    private TpmSalesPlansMapper tpmSalesPlansMapper;

    @Resource
    private CrmLogSendUtil crmLogSendUtil;

    /**
     * 列表
     *
     * @param reqVo
     * @return
     */
    @Override
    public PageResult<TpmSalesPlansRespVo> findList(TpmSalesPlansReqVo reqVo) {
        Page<TpmSalesPlansRespVo> page = PageUtil.buildPage(reqVo.getPageNum(), reqVo.getPageSize());
        List<TpmSalesPlansRespVo> list = tpmSalesPlansMapper.findList(page, reqVo);
        return PageResult.<TpmSalesPlansRespVo>builder()
                .data(list)
                .count(page.getTotal())
                .build();
    }

    /**
     * 查询
     *
     * @param id
     * @return tpmSalesPlansRespVo
     */
    @Override
    public TpmSalesPlansRespVo query(String id) {
        TpmSalesPlansEntity entity = this.getById(id);
        return CrmBeanUtil.copy(entity, TpmSalesPlansRespVo.class);
    }

    /**
     * 新增
     *
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void save(TpmSalesPlansReqVo reqVo) {
        SaveCheck(reqVo);
        TpmSalesPlansEntity entity = CrmBeanUtil.copy(reqVo, TpmSalesPlansEntity.class);
        entity.setSalePlansId(CodeUtil.createOneCode(CrmCodeRuleConstants.SCALES_PLAN));
        this.save(entity);
        //日志
        Object menuCodeObj = ThreadLocalUtil.getObj(GlobalParam.MENU_CODE);
        TpmSalesPlansRespVo newData = CrmBeanUtil.copy(entity, TpmSalesPlansRespVo.class);
        crmLogSendUtil.sendForAdd(menuCodeObj.toString(), newData.getId(), newData.getSalePlansId(), newData);
    }

    /**
     * 更新
     *
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void update(TpmSalesPlansReqVo reqVo) {
        SaveCheck(reqVo);
        TpmSalesPlansEntity entity = this.getById(reqVo.getId());
        //日志老数据
        TpmSalesPlansRespVo oldData = CrmBeanUtil.copy(entity, TpmSalesPlansRespVo.class);
        AssertUtils.isNotNull(entity, FeeBudgetException.DATA_NOT_EXIST);
        this.updateById(CrmBeanUtil.copy(reqVo, TpmSalesPlansEntity.class));
        Object menuCodeObj = ThreadLocalUtil.getObj(GlobalParam.MENU_CODE);
        TpmSalesPlansRespVo newData = CrmBeanUtil.copy(reqVo, TpmSalesPlansRespVo.class);
        //发送修改日志
        crmLogSendUtil.sendForUpdate(menuCodeObj.toString(), newData.getId(), newData.getSalePlansId(), oldData, newData);
    }

    private void SaveCheck(TpmSalesPlansReqVo reqVo) {
        AssertUtils.isNotEmpty(reqVo.getSpYear(), "年份不能为空");
        AssertUtils.isNotEmpty(reqVo.getSpMonth(), "月份不能为空");
        AssertUtils.isNotEmpty(reqVo.getSpType(), "销售计划类型不能为空");
        AssertUtils.isNotEmpty(reqVo.getChannelCode(), "销售渠道编码不能为空");
        AssertUtils.isNotEmpty(reqVo.getChannelName(), "销售渠道名称不能为空");
        AssertUtils.isNotEmpty(reqVo.getProductLevelCode(), "产品层级编码不能为空");
        AssertUtils.isNotEmpty(reqVo.getProductLevelName(), "产品层级名称不能为空");
        AssertUtils.isNotNull(reqVo.getAmount(), "金额不能为空");
        AssertUtils.isTrue(reqVo.getAmount().compareTo(BigDecimal.ZERO) > 0, "金额不能小于0");
    }

    /**
     * 删除
     *
     * @param ids
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteBatch(List<String> ids) {
        List<TpmSalesPlansEntity> tpmSalesPlansEntities = tpmSalesPlansMapper.selectBatchIds(ids);
        if (CollectionUtils.isNotEmpty(tpmSalesPlansEntities)) {
            tpmSalesPlansEntities.forEach(o -> {
                o.setDelFlag(CrmDelFlagEnum.DELETE.getCode());
            });
        }
        this.updateBatchById(tpmSalesPlansEntities);
    }

    /**
     * 启用
     *
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void enableBatch(TpmSalesPlansReqVo reqVo) {
        //设置状态为启用
        List<TpmSalesPlansEntity> tpmSalesPlansEntities = tpmSalesPlansMapper.selectBatchIds(reqVo.getIds());
        if (CollectionUtils.isNotEmpty(tpmSalesPlansEntities)) {
            tpmSalesPlansEntities.forEach(o -> {
                o.setEnableStatus(CrmEnableStatusEnum.ENABLE.getCode());
            });
        }
        this.updateBatchById(tpmSalesPlansEntities);
    }

    /**
     * 禁用
     *
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void disableBatch(TpmSalesPlansReqVo reqVo) {
        //设置状态为禁用
        List<TpmSalesPlansEntity> tpmSalesPlansEntities = tpmSalesPlansMapper.selectBatchIds(reqVo.getIds());
        if (CollectionUtils.isNotEmpty(tpmSalesPlansEntities)) {
            tpmSalesPlansEntities.forEach(o -> {
                o.setEnableStatus(CrmEnableStatusEnum.DISABLE.getCode());
            });
        }
        this.updateBatchById(tpmSalesPlansEntities);
    }
}
