package com.biz.crm.admin.service.impl;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.admin.core.cron.CronExpression;
import com.biz.crm.admin.core.model.XxlJobGroup;
import com.biz.crm.admin.core.model.XxlJobInfo;
import com.biz.crm.admin.core.model.XxlJobLogReport;
import com.biz.crm.admin.core.route.ExecutorRouteStrategyEnum;
import com.biz.crm.admin.core.scheduler.MisfireStrategyEnum;
import com.biz.crm.admin.core.scheduler.ScheduleTypeEnum;
import com.biz.crm.admin.core.thread.JobScheduleHelper;
import com.biz.crm.admin.core.util.I18nUtil;
import com.biz.crm.admin.mapper.*;
import com.biz.crm.admin.service.XxlJobService;
import com.biz.crm.base.BusinessException;
import com.biz.crm.common.PageResult;
import com.biz.crm.core.biz.model.ReturnT;
import com.biz.crm.core.enums.ExecutorBlockStrategyEnum;
import com.biz.crm.core.glue.GlueTypeEnum;
import com.biz.crm.core.util.DateUtil;
import com.biz.crm.eunm.job.JobColumn;
import com.biz.crm.eunm.job.JobColumnStr;
import com.biz.crm.mdm.position.MdmPositionFeign;
import com.biz.crm.nebular.job.req.JobChartInfoReqVo;
import com.biz.crm.nebular.job.req.XxlJobInfoReqVo;
import com.biz.crm.nebular.job.resp.JobChartInfoInfoRespVo;
import com.biz.crm.nebular.job.resp.JobStatisticsInfoRespVo;
import com.biz.crm.util.AssertUtils;
import com.biz.crm.util.PageUtil;
import com.biz.crm.util.StringUtils;
import com.biz.crm.util.UUIDGenerator;
import com.google.common.collect.Lists;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.text.MessageFormat;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

/**
 * core job action for xxl-job
 *
 * @author xuxueli 2016-5-28 15:30:33
 */
@Service
public class XxlJobServiceImpl implements XxlJobService {
    private static Logger logger = LoggerFactory.getLogger(XxlJobServiceImpl.class);

    @Resource
    private XxlJobGroupDao xxlJobGroupDao;
    @Resource
    private XxlJobInfoDao xxlJobInfoDao;
    @Resource
    public XxlJobLogDao xxlJobLogDao;
    @Resource
    private XxlJobLogGlueDao xxlJobLogGlueDao;
    @Resource
    private XxlJobLogReportDao xxlJobLogReportDao;
    @Resource
    private MdmPositionFeign mdmPositionFeign;

    @Override
    public PageResult<XxlJobInfo> pageList(XxlJobInfoReqVo xxlJobInfoReqVo) {
        Page<XxlJobInfo> page = PageUtil.buildPage(xxlJobInfoReqVo.getPageNum(), xxlJobInfoReqVo.getPageSize());
        // page list
        List<XxlJobInfo> list = xxlJobInfoDao.pageList(page, xxlJobInfoReqVo);
        list.forEach(xxlJobInfo -> xxlJobInfo.setTriggerStatusValue(JobColumn.TRIGGER_STATUS.getLabelByCode(xxlJobInfo.getTriggerStatus())));
        // package result
        return PageResult.<XxlJobInfo>builder()
                .data(list)
                .count(page.getTotal())
                .build();
    }

    @Override
    public XxlJobInfo query(String id) {
        return this.xxlJobInfoDao.loadById(id);
    }

    @Override
    public XxlJobInfo add(XxlJobInfo jobInfo) {

        // valid base
        XxlJobGroup group = xxlJobGroupDao.load(jobInfo.getJobGroup());
        AssertUtils.isNotNull(group, "请选择执行器！");
        AssertUtils.isNotEmpty(jobInfo.getJobDesc(), "请输入任务描述！");

        // valid trigger
        ScheduleTypeEnum scheduleTypeEnum = ScheduleTypeEnum.match(jobInfo.getScheduleType(), null);
        AssertUtils.isNotNull(scheduleTypeEnum, "调度类型不能为空！");
        if (scheduleTypeEnum == ScheduleTypeEnum.CRON) {
            if (jobInfo.getScheduleConf() == null || !CronExpression.isValidExpression(jobInfo.getScheduleConf())) {
                throw new BusinessException("Cron非法！");
            }
        } else if (scheduleTypeEnum == ScheduleTypeEnum.FIX_RATE) {
            AssertUtils.isNotEmpty(jobInfo.getScheduleConf(), "调度规则不能为空");
            try {
                int fixSecond = Integer.parseInt(jobInfo.getScheduleConf());
                if (fixSecond < 1) {
                    throw new BusinessException("调度类型非法");
                }
            } catch (Exception e) {
                throw new BusinessException("调度类型非法");
            }
        }

        // valid job
        AssertUtils.isNotNull(GlueTypeEnum.match(jobInfo.getGlueType()), "运行模式非法");
        AssertUtils.isTrue(!(GlueTypeEnum.BEAN == GlueTypeEnum.match(jobInfo.getGlueType()) && (jobInfo.getExecutorHandler() == null || jobInfo.getExecutorHandler().trim().length() == 0)), "请输入JobHandler");
        // 》fix "\r" in shell
        if (GlueTypeEnum.GLUE_SHELL == GlueTypeEnum.match(jobInfo.getGlueType()) && jobInfo.getGlueSource() != null) {
            jobInfo.setGlueSource(jobInfo.getGlueSource().replaceAll("\r", ""));
        }

        // valid advanced
        AssertUtils.isNotNull(ExecutorRouteStrategyEnum.match(jobInfo.getExecutorRouteStrategy(), null), "路由策略非法");
        AssertUtils.isNotNull(MisfireStrategyEnum.match(jobInfo.getMisfireStrategy(), null), "调度过期策略非法");
        AssertUtils.isNotNull(ExecutorBlockStrategyEnum.match(jobInfo.getExecutorBlockStrategy(), null), "阻塞处理策略非法");

        // 校验子任务ID
        if (jobInfo.getChildJobId() != null && jobInfo.getChildJobId().trim().length() > 0) {
            String[] childJobIds = jobInfo.getChildJobId().split(",");
            for (String childJobIdItem : childJobIds) {
                if (childJobIdItem != null && childJobIdItem.trim().length() > 0) {
                    XxlJobInfo childJobInfo = xxlJobInfoDao.loadById(childJobIdItem);
                    AssertUtils.isNotNull(childJobInfo, ("子任务ID" + "({0})" + "不存在"));
                } else {
                    throw new BusinessException("子任务ID" + "({0})" + "非法");
                }
            }

            // join , avoid "xxx,,"
            StringBuilder temp = new StringBuilder();
            for (String item : childJobIds) {
                temp.append(item).append(",");
            }
            temp = new StringBuilder(temp.substring(0, temp.length() - 1));

            jobInfo.setChildJobId(temp.toString());
        }

        // add in db
        jobInfo.setAddTime(new Date());
        jobInfo.setUpdateTime(new Date());
        jobInfo.setGlueUpdatetime(new Date());
        jobInfo.setGlueSource(JobColumnStr.GLUE_SOURCE.getLabelByCode(jobInfo.getGlueType()));
        jobInfo.setId(UUIDGenerator.generate());
        //xxlJobInfoDao.save(jobInfo);
        xxlJobInfoDao.insert(jobInfo);
        if (StringUtils.isEmpty(jobInfo.getId())) {
            throw new BusinessException(I18nUtil.getString("新增") + I18nUtil.getString("失败"));
        }
        return jobInfo;
    }

    private boolean isNumeric(String str) {
        try {
            int result = Integer.valueOf(str);
            return true;
        } catch (NumberFormatException e) {
            return false;
        }
    }

    @Override
    public ReturnT<String> update(XxlJobInfo jobInfo) {

        // valid base
        if (jobInfo.getJobDesc() == null || jobInfo.getJobDesc().trim().length() == 0) {
            return new ReturnT<String>(ReturnT.FAIL_CODE, (I18nUtil.getString("请输入") + I18nUtil.getString("任务描述")));
        }
        if (jobInfo.getAuthor() == null || jobInfo.getAuthor().trim().length() == 0) {
            return new ReturnT<String>(ReturnT.FAIL_CODE, (I18nUtil.getString("请输入") + I18nUtil.getString("负责人")));
        }

        // valid trigger
        ScheduleTypeEnum scheduleTypeEnum = ScheduleTypeEnum.match(jobInfo.getScheduleType(), null);
        if (scheduleTypeEnum == null) {
            return new ReturnT<String>(ReturnT.FAIL_CODE, (I18nUtil.getString("调度类型") + I18nUtil.getString("非法")));
        }
        if (scheduleTypeEnum == ScheduleTypeEnum.CRON) {
            if (jobInfo.getScheduleConf() == null || !CronExpression.isValidExpression(jobInfo.getScheduleConf())) {
                return new ReturnT<String>(ReturnT.FAIL_CODE, "Cron" + I18nUtil.getString("非法"));
            }
        } else if (scheduleTypeEnum == ScheduleTypeEnum.FIX_RATE /*|| scheduleTypeEnum == ScheduleTypeEnum.FIX_DELAY*/) {
            if (jobInfo.getScheduleConf() == null) {
                return new ReturnT<String>(ReturnT.FAIL_CODE, (I18nUtil.getString("调度类型") + I18nUtil.getString("非法")));
            }
            try {
                int fixSecond = Integer.valueOf(jobInfo.getScheduleConf());
                if (fixSecond < 1) {
                    return new ReturnT<String>(ReturnT.FAIL_CODE, (I18nUtil.getString("调度类型") + I18nUtil.getString("非法")));
                }
            } catch (Exception e) {
                return new ReturnT<String>(ReturnT.FAIL_CODE, (I18nUtil.getString("调度类型") + I18nUtil.getString("非法")));
            }
        }

        // valid advanced
        if (ExecutorRouteStrategyEnum.match(jobInfo.getExecutorRouteStrategy(), null) == null) {
            return new ReturnT<String>(ReturnT.FAIL_CODE, (I18nUtil.getString("路由策略") + I18nUtil.getString("非法")));
        }
        if (MisfireStrategyEnum.match(jobInfo.getMisfireStrategy(), null) == null) {
            return new ReturnT<String>(ReturnT.FAIL_CODE, (I18nUtil.getString("调度过期策略") + I18nUtil.getString("非法")));
        }
        if (ExecutorBlockStrategyEnum.match(jobInfo.getExecutorBlockStrategy(), null) == null) {
            return new ReturnT<String>(ReturnT.FAIL_CODE, (I18nUtil.getString("阻塞处理策略") + I18nUtil.getString("非法")));
        }

        // 》ChildJobId valid
        if (jobInfo.getChildJobId() != null && jobInfo.getChildJobId().trim().length() > 0) {
            String[] childJobIds = jobInfo.getChildJobId().split(",");
            for (String childJobIdItem : childJobIds) {
                if (childJobIdItem != null && childJobIdItem.trim().length() > 0 ) {
                    XxlJobInfo childJobInfo = xxlJobInfoDao.loadById(childJobIdItem);
                    if (childJobInfo == null) {
                        return new ReturnT<String>(ReturnT.FAIL_CODE,
                                MessageFormat.format((I18nUtil.getString("子任务ID") + "({0})" + I18nUtil.getString("不存在")), childJobIdItem));
                    }
                } else {
                    return new ReturnT<String>(ReturnT.FAIL_CODE,
                            MessageFormat.format((I18nUtil.getString("子任务ID") + "({0})" + I18nUtil.getString("非法")), childJobIdItem));
                }
            }

            // join , avoid "xxx,,"
            String temp = "";
            for (String item : childJobIds) {
                temp += item + ",";
            }
            temp = temp.substring(0, temp.length() - 1);

            jobInfo.setChildJobId(temp);
        }

        // group valid
        XxlJobGroup jobGroup = xxlJobGroupDao.load(jobInfo.getJobGroup());
        if (jobGroup == null) {
            return new ReturnT<String>(ReturnT.FAIL_CODE, (I18nUtil.getString("执行器") + I18nUtil.getString("非法")));
        }

        // stage job info
        XxlJobInfo exists_jobInfo = xxlJobInfoDao.loadById(jobInfo.getId());
        if (exists_jobInfo == null) {
            return new ReturnT<String>(ReturnT.FAIL_CODE, (I18nUtil.getString("jobinfo_field_id") + I18nUtil.getString("不存在")));
        }

        // next trigger time (5s后生效，避开预读周期)
        long nextTriggerTime = exists_jobInfo.getTriggerNextTime();
        boolean scheduleDataNotChanged = jobInfo.getScheduleType().equals(exists_jobInfo.getScheduleType()) && jobInfo.getScheduleConf().equals(exists_jobInfo.getScheduleConf());
        if (exists_jobInfo.getTriggerStatus() == 1 && !scheduleDataNotChanged) {
            try {
                Date nextValidTime = JobScheduleHelper.generateNextValidTime(jobInfo, new Date(System.currentTimeMillis() + JobScheduleHelper.PRE_READ_MS));
                if (nextValidTime == null) {
                    return new ReturnT<String>(ReturnT.FAIL_CODE, (I18nUtil.getString("调度类型") + I18nUtil.getString("非法")));
                }
                nextTriggerTime = nextValidTime.getTime();
            } catch (Exception e) {
                logger.error(e.getMessage(), e);
                return new ReturnT<String>(ReturnT.FAIL_CODE, (I18nUtil.getString("调度类型") + I18nUtil.getString("非法")));
            }
        }

        exists_jobInfo.setJobGroup(jobInfo.getJobGroup());
        exists_jobInfo.setJobDesc(jobInfo.getJobDesc());
        exists_jobInfo.setAuthor(jobInfo.getAuthor());
        exists_jobInfo.setAlarmEmail(jobInfo.getAlarmEmail());
        exists_jobInfo.setScheduleType(jobInfo.getScheduleType());
        exists_jobInfo.setScheduleConf(jobInfo.getScheduleConf());
        exists_jobInfo.setMisfireStrategy(jobInfo.getMisfireStrategy());
        exists_jobInfo.setExecutorRouteStrategy(jobInfo.getExecutorRouteStrategy());
        exists_jobInfo.setExecutorHandler(jobInfo.getExecutorHandler());
        exists_jobInfo.setExecutorParam(jobInfo.getExecutorParam());
        exists_jobInfo.setExecutorBlockStrategy(jobInfo.getExecutorBlockStrategy());
        exists_jobInfo.setExecutorTimeout(jobInfo.getExecutorTimeout());
        exists_jobInfo.setExecutorFailRetryCount(jobInfo.getExecutorFailRetryCount());
        exists_jobInfo.setChildJobId(jobInfo.getChildJobId());
        exists_jobInfo.setTriggerNextTime(nextTriggerTime);

        exists_jobInfo.setUpdateTime(new Date());
        xxlJobInfoDao.updateById(exists_jobInfo);


        return ReturnT.SUCCESS;
    }

    @Override
    @Transactional
    public void remove(List<String> ids) {
        ids.forEach(id -> {
            XxlJobInfo xxlJobInfo = xxlJobInfoDao.loadById(id);
            if (xxlJobInfo != null) {
                xxlJobInfoDao.delete(id);
                xxlJobLogDao.delete(id);
                xxlJobLogGlueDao.deleteByJobId(id);
            }
        });
    }

    @Override
    public ReturnT<String> start(String id) {
        XxlJobInfo xxlJobInfo = xxlJobInfoDao.loadById(id);

        // valid
        ScheduleTypeEnum scheduleTypeEnum = ScheduleTypeEnum.match(xxlJobInfo.getScheduleType(), ScheduleTypeEnum.NONE);
        if (ScheduleTypeEnum.NONE == scheduleTypeEnum) {
            return new ReturnT<String>(ReturnT.FAIL_CODE, (I18nUtil.getString("调度类型_none_limit_start")));
        }

        // next trigger time (5s后生效，避开预读周期)
        long nextTriggerTime = 0;
        try {
            Date nextValidTime = JobScheduleHelper.generateNextValidTime(xxlJobInfo, new Date(System.currentTimeMillis() + JobScheduleHelper.PRE_READ_MS));
            if (nextValidTime == null) {
                return new ReturnT<String>(ReturnT.FAIL_CODE, (I18nUtil.getString("调度类型") + I18nUtil.getString("非法")));
            }
            nextTriggerTime = nextValidTime.getTime();
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
            return new ReturnT<String>(ReturnT.FAIL_CODE, (I18nUtil.getString("调度类型") + I18nUtil.getString("非法")));
        }

        xxlJobInfo.setTriggerStatus(1);
        xxlJobInfo.setTriggerLastTime(0);
        xxlJobInfo.setTriggerNextTime(nextTriggerTime);

        xxlJobInfo.setUpdateTime(new Date());
        xxlJobInfoDao.updateById(xxlJobInfo);
        return ReturnT.SUCCESS;
    }

    @Override
    public ReturnT<String> stop(String id) {
        XxlJobInfo xxlJobInfo = xxlJobInfoDao.loadById(id);

        xxlJobInfo.setTriggerStatus(0);
        xxlJobInfo.setTriggerLastTime(0);
        xxlJobInfo.setTriggerNextTime(0);

        xxlJobInfo.setUpdateTime(new Date());
        xxlJobInfoDao.updateById(xxlJobInfo);
        return ReturnT.SUCCESS;
    }

    /**
     * 获取任务统计数据
     *
     * @return 统计数据返回vo
     */
    @Override
    public JobStatisticsInfoRespVo statisticsInfo() {
        int jobLogCount = 0;
        XxlJobLogReport xxlJobLogReport = xxlJobLogReportDao.queryLogReportTotal();
        if (xxlJobLogReport != null) {
            jobLogCount = xxlJobLogReport.getRunningCount() + xxlJobLogReport.getSucCount() + xxlJobLogReport.getFailCount();
        }

        // 执行器数量
        Set<String> executorAddressSet = new HashSet<>();
        List<XxlJobGroup> groupList = xxlJobGroupDao.findAll();

        if (groupList != null && !groupList.isEmpty()) {
            for (XxlJobGroup group : groupList) {
                if (group.getRegistryList() != null && !group.getRegistryList().isEmpty()) {
                    executorAddressSet.addAll(group.getRegistryList());
                }
            }
        }
        return new JobStatisticsInfoRespVo().setJobInfoCount(xxlJobInfoDao.findAllCount())
                .setJobLogCount(jobLogCount)
                .setExecutorCount(executorAddressSet.size());
    }

    /**
     * 获取任务统计图表数据
     * @param reqVo 获取图表请求数据
     * @return 统计数据
     */
    @Override
    public JobChartInfoInfoRespVo chartInfo(JobChartInfoReqVo reqVo) {

        // process
        List<String> triggerDayList = Lists.newArrayList();
        List<Integer> triggerDayCountRunningList = Lists.newArrayList();
        List<Integer> triggerDayCountSucList = Lists.newArrayList();
        List<Integer> triggerDayCountFailList = Lists.newArrayList();
        int triggerCountRunningTotal = 0;
        int triggerCountSucTotal = 0;
        int triggerCountFailTotal = 0;

        //根据时间获取调度日志
        List<XxlJobLogReport> logReportList = xxlJobLogReportDao.queryLogReport(reqVo.getStartTime(), reqVo.getEndTime());

        if (logReportList != null && logReportList.size() > 0) {
            for (XxlJobLogReport item : logReportList) {
                String day = DateUtil.formatDate(item.getTriggerDay());
                int triggerDayCountRunning = item.getRunningCount();
                int triggerDayCountSuc = item.getSucCount();
                int triggerDayCountFail = item.getFailCount();

                triggerDayList.add(day);
                triggerDayCountRunningList.add(triggerDayCountRunning);
                triggerDayCountSucList.add(triggerDayCountSuc);
                triggerDayCountFailList.add(triggerDayCountFail);

                triggerCountRunningTotal += triggerDayCountRunning;
                triggerCountSucTotal += triggerDayCountSuc;
                triggerCountFailTotal += triggerDayCountFail;
            }
        } else {
            for (int i = -6; i <= 0; i++) {
                triggerDayList.add(DateUtil.formatDate(DateUtil.addDays(new Date(), i)));
                triggerDayCountRunningList.add(0);
                triggerDayCountSucList.add(0);
                triggerDayCountFailList.add(0);
            }
        }
        return new JobChartInfoInfoRespVo().setTriggerDayList(triggerDayList)
                .setTriggerDayCountRunningList(triggerDayCountRunningList)
                .setTriggerDayCountSucList(triggerDayCountSucList)
                .setTriggerDayCountFailList(triggerDayCountFailList)
                .setTriggerCountRunningTotal(triggerCountRunningTotal)
                .setTriggerCountSucTotal(triggerCountSucTotal)
                .setTriggerCountFailTotal(triggerCountFailTotal);
    }

    @Override
    public  void test(){
         mdmPositionFeign.detail("","ZW00012087");
    }
}
