package com.biz.crm.admin.controller;

import com.biz.crm.admin.core.exception.XxlJobException;
import com.biz.crm.admin.core.model.XxlJobGroup;
import com.biz.crm.admin.core.model.XxlJobInfo;
import com.biz.crm.admin.core.model.XxlJobUser;
import com.biz.crm.admin.core.route.ExecutorRouteStrategyEnum;
import com.biz.crm.admin.core.scheduler.MisfireStrategyEnum;
import com.biz.crm.admin.core.scheduler.ScheduleTypeEnum;
import com.biz.crm.admin.core.thread.JobScheduleHelper;
import com.biz.crm.admin.core.thread.JobTriggerPoolHelper;
import com.biz.crm.admin.core.trigger.TriggerTypeEnum;
import com.biz.crm.admin.core.util.I18nUtil;
import com.biz.crm.admin.mapper.XxlJobGroupDao;
import com.biz.crm.admin.service.LoginService;
import com.biz.crm.admin.service.XxlJobService;
import com.biz.crm.common.PageResult;
import com.biz.crm.core.biz.model.ReturnT;
import com.biz.crm.core.enums.ExecutorBlockStrategyEnum;
import com.biz.crm.core.glue.GlueTypeEnum;
import com.biz.crm.core.util.DateUtil;
import com.biz.crm.nebular.job.req.XxlJobInfoReqVo;
import com.biz.crm.util.Result;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Controller;
import org.springframework.ui.Model;
import org.springframework.util.StringUtils;
import org.springframework.web.bind.annotation.*;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletRequest;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.List;

/**
 * index controller
 *
 * @author xuxueli 2015-12-19 16:13:16
 */
@Controller
@RequestMapping("/jobinfo")
@Api(tags = "任务管理表;")
public class JobInfoController {
    private static Logger logger = LoggerFactory.getLogger(JobInfoController.class);

    @Resource
    private XxlJobGroupDao xxlJobGroupDao;
    @Resource
    private XxlJobService xxlJobService;

    public String index(HttpServletRequest request, Model model, @RequestParam(required = false, defaultValue = "-1") int jobGroup) {

        // 枚举-字典
        model.addAttribute("ExecutorRouteStrategyEnum", ExecutorRouteStrategyEnum.values());        // 路由策略-列表
        model.addAttribute("GlueTypeEnum", GlueTypeEnum.values());                                // Glue类型-字典
        model.addAttribute("ExecutorBlockStrategyEnum", ExecutorBlockStrategyEnum.values());        // 阻塞处理策略-字典
        model.addAttribute("ScheduleTypeEnum", ScheduleTypeEnum.values());                        // 调度类型
        model.addAttribute("MisfireStrategyEnum", MisfireStrategyEnum.values());                    // 调度过期策略

        // 执行器列表
        List<XxlJobGroup> jobGroupList_all = xxlJobGroupDao.findAll();

        // filter group
        List<XxlJobGroup> jobGroupList = filterJobGroupByRole(request, jobGroupList_all);
        if (jobGroupList == null || jobGroupList.size() == 0) {
            throw new XxlJobException(I18nUtil.getString("jobgroup_empty"));
        }

        model.addAttribute("JobGroupList", jobGroupList);
        model.addAttribute("jobGroup", jobGroup);

        return "jobinfo/jobinfo.index";
    }

    public static List<XxlJobGroup> filterJobGroupByRole(HttpServletRequest request, List<XxlJobGroup> jobGroupList_all) {
        List<XxlJobGroup> jobGroupList = new ArrayList<>();
        if (jobGroupList_all != null && jobGroupList_all.size() > 0) {
            XxlJobUser loginUser = (XxlJobUser) request.getAttribute(LoginService.LOGIN_IDENTITY_KEY);
            if (loginUser.getRole() == 1) {
                jobGroupList = jobGroupList_all;
            } else {
                List<String> groupIdStrs = new ArrayList<>();
                if (loginUser.getPermission() != null && loginUser.getPermission().trim().length() > 0) {
                    groupIdStrs = Arrays.asList(loginUser.getPermission().trim().split(","));
                }
                for (XxlJobGroup groupItem : jobGroupList_all) {
                    if (groupIdStrs.contains(String.valueOf(groupItem.getId()))) {
                        jobGroupList.add(groupItem);
                    }
                }
            }
        }
        return jobGroupList;
    }

    public static void validPermission(HttpServletRequest request, String jobGroup) {
        XxlJobUser loginUser = (XxlJobUser) request.getAttribute(LoginService.LOGIN_IDENTITY_KEY);
        if (!loginUser.validPermission(jobGroup)) {
            throw new RuntimeException(I18nUtil.getString("system_permission_limit") + "[username=" + loginUser.getUsername() + "]");
        }
    }

    @PostMapping("/pageList")
    @ResponseBody
    @ApiOperation("获取任务列表")
    public Result<PageResult<XxlJobInfo>> pageList(@RequestBody XxlJobInfoReqVo xxlJobInfoReqVo) {
        return Result.ok(xxlJobService.pageList(xxlJobInfoReqVo));
    }

    /**
     * 查询
     */
    @ApiOperation(value = "查询")
    @GetMapping("/query")
    @ResponseBody
    public Result<XxlJobInfo> query(@RequestParam(value = "id", required = false) String id) {
        XxlJobInfo xxlJobInfo=xxlJobService.query(id);
        return Result.ok(xxlJobInfo);
    }

    @PostMapping("/add")
    @ResponseBody
    @ApiOperation("新增任务")
    public Result add(@RequestBody XxlJobInfo jobInfo) {
        xxlJobService.add(jobInfo);
        return Result.ok();
    }

    @PostMapping("/update")
    @ResponseBody
    @ApiOperation("更新任务")
    public Result update(@RequestBody XxlJobInfo jobInfo) {
        return Result.ok(xxlJobService.update(jobInfo).getMsg());
    }

    @PostMapping("/remove")
    @ResponseBody
    @ApiOperation("删除任务")
    public Result remove(@RequestBody List<String> ids) {
        xxlJobService.remove(ids);
        return Result.ok();
    }

    @PostMapping("/stop")
    @ResponseBody
    @ApiOperation("停止任务")
    public Result pause(@RequestBody List<String> ids) {
        ids.forEach(id -> xxlJobService.stop(id));
        return Result.ok();
    }

    @PostMapping("/start")
    @ResponseBody
    @ApiOperation("开启任务")
    public Result start(@RequestBody List<String> ids) {
        ids.forEach(id -> xxlJobService.start(id));
        return Result.ok();
    }

    @PostMapping("/trigger")
    @ResponseBody
    @ApiOperation("触发一次任务")
    public Result<Object> triggerJob(@RequestBody XxlJobInfoReqVo vo) {
        // force cover job param
        if (StringUtils.isEmpty(vo.getExecutorParam())) {
            XxlJobInfo jobInfo = xxlJobService.query(vo.getId());
            vo.setExecutorParam(jobInfo.getExecutorParam());
        }

        JobTriggerPoolHelper.trigger(vo.getId(), TriggerTypeEnum.MANUAL, -1, null, vo.getExecutorParam(), vo.getAddressList());
        return Result.ok();
    }

    @PostMapping("/nextTriggerTime")
    @ResponseBody
    @ApiOperation("下次执行时间")
    public Result nextTriggerTime(@RequestBody XxlJobInfoReqVo xxlJobInfoReqVo) {

        XxlJobInfo paramXxlJobInfo = new XxlJobInfo();
        paramXxlJobInfo.setScheduleType(xxlJobInfoReqVo.getScheduleType());
        paramXxlJobInfo.setScheduleConf(xxlJobInfoReqVo.getScheduleConf());

        List<String> result = new ArrayList<>();
        try {
            Date lastTime = new Date();
            for (int i = 0; i < 5; i++) {
                lastTime = JobScheduleHelper.generateNextValidTime(paramXxlJobInfo, lastTime);
                if (lastTime != null) {
                    result.add(DateUtil.formatDateTime(lastTime));
                } else {
                    break;
                }
            }
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
            return Result.error(ReturnT.FAIL_CODE, (I18nUtil.getString("schedule_type") + I18nUtil.getString("system_unvalid")) + e.getMessage());
        }
        return Result.ok(result);

    }

}
