package com.biz.crm.common.captcha.local.service.internal;

import com.alibaba.fastjson.JSON;
import com.biz.crm.common.captcha.sdk.common.constant.RedisKeys;
import com.biz.crm.common.captcha.sdk.common.enums.CaptchaBaseMapEnum;
import com.biz.crm.common.captcha.sdk.config.CaptchaProperties;
import com.biz.crm.common.captcha.sdk.dto.CaptchaDto;
import com.biz.crm.common.captcha.sdk.dto.PointDto;
import com.biz.crm.common.captcha.sdk.service.CaptchaCacheService;
import com.biz.crm.common.captcha.sdk.service.CaptchaService;
import com.biz.crm.common.captcha.sdk.service.FrequencyLimitService;
import com.bizunited.nebula.common.util.Aes128Utils;
import org.apache.commons.lang3.Validate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.core.io.Resource;
import org.springframework.core.io.support.PathMatchingResourcePatternResolver;
import org.springframework.core.io.support.ResourcePatternResolver;
import org.springframework.util.Base64Utils;
import org.springframework.util.FileCopyUtils;

import javax.annotation.PostConstruct;
import javax.imageio.ImageIO;
import java.awt.*;
import java.awt.image.BufferedImage;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.util.*;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;

/**
 * 验证码服务类
 *
 * @author zwb
 * @date 2021-09-27
 */
public class BlockPuzzleCaptchaServiceImpl implements CaptchaService {

  /**
   * 日志
   */
  private static final Logger LOGGER = LoggerFactory.getLogger(BlockPuzzleCaptchaServiceImpl.class);

  //加密秘钥
  @Value("${security.aes128.key:1234123412ABCDEF}")
  private String encryptKey;

  //滑块底图
  private static Map<String, String> originalCacheMap = new ConcurrentHashMap();

  //滑块
  private static Map<String, String> slidingBlockCacheMap = new ConcurrentHashMap();

  //文件map
  private static Map<String, String[]> fileNameMap = new ConcurrentHashMap<>();

  //缓存实体类
  @Autowired
  private CaptchaCacheService cacheService;

  //配置文件中的属性
  @Autowired
  private CaptchaProperties prop;

  //接口限制类
  @Autowired
  private FrequencyLimitService limitHandler;

  //图片类型
  private static final String IMAGE_TYPE_PNG = "png";

  //check校验坐标 过期时间
  private static Long EXPIRESIN_SECONDS = 2 * 60L;

  //后台二次校验坐标过期时间
  private static Long EXPIRESIN_THREE = 3 * 60L;


  /**
   * 初始化水印，字体，接口限制请求数量.增加@PostConstruct注解初始化属性
   */
  @PostConstruct
  public void init() {
    //读取配置文件中默认的配置并将其赋值给属性类，此类可用于后续初始化实体类的配置
    LOGGER.info("自定义配置项：{}", prop.toString());
    Validate.notNull(prop, "配置文件为空");
    Validate.notNull(prop.getInterferenceOptions(), "滑块干扰项配置不能为空");
    Validate.notBlank(prop.getCaptchaOriginal(), "滑动拼图底图路径不能为空");
    Validate.notBlank(prop.getCaptchaSlidingblock(), "滑动拼图滑块图片路径不能为空");
    Validate.notNull(prop.getSlipOffset(), "校验滑动拼图允许误差偏移量不能为空");
    Validate.notNull(prop.getReqGetMinuteLimit(), "get接口一分钟内限制访问数不能为空");
    Validate.notNull(prop.getReqCheckMinuteLimit(), "check接口一分钟内限制访问数不能为空");
    Validate.notNull(prop.getReqCheckFailLimit(), "check接口 一分钟内失败次数限制不能为空");
    Validate.notNull(prop.getReqCheckFailLockSeconds(), "check接口验证失败后，接口锁定时间不能为空");
    //resources目录下初始化滑块底图和点击文字验证底图
    LOGGER.info("--->>>初始化文字验证码底图<<<---");
    this.initializeBaseMap(prop.getCaptchaOriginal(),prop.getCaptchaSlidingblock());

  }

  /**
   * 获取一个图片底图和滑块图片
   */
  @Override
  public CaptchaDto getCaptcha(CaptchaDto captchaDto) {
    Validate.notBlank(captchaDto.getClientUniqueId(), "用户标识不能为空");
    //获取验证码接口限流
    limitHandler.validateGetImg(captchaDto);
    //原生图片
    BufferedImage originalImage = getOriginal();
    Validate.notNull(originalImage, "滑动底图未初始化成功，请检查路径");

    //抠图图片
    String jigsawImageBase64 = getslidingBlock();
    BufferedImage jigsawImage = getBase64StrToImage(jigsawImageBase64);
    Validate.notNull(jigsawImage, "滑块图片未转换成功，请检查路径");
    CaptchaDto captcha = pictureTemplatesCut(originalImage, jigsawImage, jigsawImageBase64, captchaDto.getClientUniqueId());
    Validate.notNull(captcha, "获取验证码失败,请联系管理员！");
    Validate.notBlank(captcha.getJigsawImageBase64(), "获取验证码失败,滑块图片为空，请联系管理员！");
    Validate.notBlank(captcha.getOriginalImageBase64(), "获取验证码失败,原图为空，请联系管理员！");
    return captcha;
  }

  /**
   * 验证滑块图片验证是否正确
   *
   * @param captchaDto
   */
  @Override
  public CaptchaDto checkCaptcha(CaptchaDto captchaDto) {
    Validate.notBlank(captchaDto.getClientUniqueId(), "用户标识不能为空");
    //验证接口限流
    limitHandler.validateCheckImg(captchaDto);
    //取坐标信息
    String codeKey = String.format(RedisKeys.CAPTCHA_CHECK_TOKEN_KEY, captchaDto.getToken());
    Validate.isTrue(cacheService.existsMCode(codeKey, captchaDto.getClientUniqueId()), "验证码已失效，请重新获取！");
    String s = cacheService.getMCode(codeKey, captchaDto.getClientUniqueId());
    //验证码只用一次，即刻失效
    cacheService.deleteMCode(codeKey, captchaDto.getClientUniqueId());
    PointDto point = null;
    PointDto point1 = null;
    String pointJson = null;
    try {
      point = JSON.parseObject(s, PointDto.class);
      //aes解密
      pointJson = Aes128Utils.decrypt(captchaDto.getPointJson(), encryptKey, Aes128Utils.EncodeType.CBC, Aes128Utils.Padding.PKCS_7_PADDING);
      point1 = JSON.parseObject(pointJson, PointDto.class);
    } catch (Exception e) {
      LOGGER.error("验证码坐标解析失败", e);
      afterValidateFail(captchaDto);
      throw new IllegalStateException("验证码坐标解析失败");
    }
    if (point.x - prop.getSlipOffset() > point1.x
        || point1.x > point.x + prop.getSlipOffset()
        || point.y != point1.y) {
      afterValidateFail(captchaDto);
      throw new IllegalStateException("验证失败！");
    }
    //校验成功，将信息存入缓存
    String value = null;
    try {
      value = Aes128Utils.encrypt(captchaDto.getToken().concat("---").concat(pointJson), encryptKey, Aes128Utils.EncodeType.CBC, Aes128Utils.Padding.PKCS_7_PADDING);
    } catch (Exception e) {
      LOGGER.error("AES加密失败", e);
      afterValidateFail(captchaDto);
      throw new IllegalStateException("AES加密失败");
    }
    String secondKey = String.format(RedisKeys.CAPTCHA_REDIS_SECOND_CHECK_KEY, value);
    cacheService.setMCode(secondKey, captchaDto.getToken(), EXPIRESIN_THREE, captchaDto.getClientUniqueId());
    captchaDto.setResult(true);
    captchaDto.resetClientFlag();
    return captchaDto;
  }

  /**
   * 二次校验验证码接口是否正常
   *
   * @param captchaVerification 二次校验码，前端传递
   * @param clientUniqueId      用户唯一标识
   */
  @Override
  public void verificationCaptcha(String captchaVerification, String clientUniqueId) {
    try {
      Validate.notBlank(clientUniqueId, "用户标识不能为空");
      Validate.notBlank(captchaVerification, "二次校验码不能为空");
      String codeKey = String.format(RedisKeys.CAPTCHA_REDIS_SECOND_CHECK_KEY, captchaVerification);
      Validate.isTrue(cacheService.existsMCode(codeKey, clientUniqueId), "二次验证失败");
      //二次校验取值后，即刻失效
      cacheService.deleteMCode(codeKey, clientUniqueId);
    } catch (Exception e) {
      LOGGER.error("验证码坐标解析失败", e);
      throw new IllegalStateException("验证码坐标解析失败");
    }
  }


  /**
   * 根据模板切图
   *
   * @param originalImage     原图
   * @param jigsawImage       滑块图片
   * @param jigsawImageBase64 滑块图片base64编码
   * @param clientUniqueId    用户唯一标识
   * @throws Exception
   */
  public CaptchaDto pictureTemplatesCut(BufferedImage originalImage, BufferedImage jigsawImage, String jigsawImageBase64, String clientUniqueId) {
    try {
      Validate.notBlank(clientUniqueId, "用户标识不能为空");
      CaptchaDto dataVO = new CaptchaDto();

      int originalWidth = originalImage.getWidth();
      int originalHeight = originalImage.getHeight();
      int jigsawWidth = jigsawImage.getWidth();
      int jigsawHeight = jigsawImage.getHeight();

      //随机生成拼图坐标
      PointDto point = generateJigsawPoint(originalWidth, originalHeight, jigsawWidth, jigsawHeight);
      int x = point.getX();
      int y = point.getY();

      //生成新的拼图图像
      BufferedImage newJigsawImage = new BufferedImage(jigsawWidth, jigsawHeight, jigsawImage.getType());
      Graphics2D graphics = newJigsawImage.createGraphics();

      int bold = 5;
      //如果需要生成RGB格式，需要做如下配置,Transparency 设置透明
      newJigsawImage = graphics.getDeviceConfiguration().createCompatibleImage(jigsawWidth, jigsawHeight, Transparency.TRANSLUCENT);
      // 新建的图像根据模板颜色赋值,源图生成遮罩
      cutByTemplate(originalImage, jigsawImage, newJigsawImage, x, 0);
      if (prop.getInterferenceOptions() > 0) {
        int position = 0;
        if (originalWidth - x - 5 > jigsawWidth * 2) {
          //在原扣图右边插入干扰图
          position = getRandomInt(x + jigsawWidth + 5, originalWidth - jigsawWidth);
        } else {
          //在原扣图左边插入干扰图
          position = getRandomInt(100, x - jigsawWidth - 5);
        }
        while (true) {
          String s = getslidingBlock();
          if (!jigsawImageBase64.equals(s)) {
            interferenceByTemplate(originalImage, Validate.notNull(getBase64StrToImage(s)), position, 0);
            break;
          }
        }
      }
      if (prop.getInterferenceOptions() > 1) {
        while (true) {
          String s = getslidingBlock();
          if (!jigsawImageBase64.equals(s)) {
            Integer randomInt = getRandomInt(jigsawWidth, 100 - jigsawWidth);
            interferenceByTemplate(originalImage, Objects.requireNonNull(getBase64StrToImage(s)),
                randomInt, 0);
            break;
          }
        }
      }


      // 设置“抗锯齿”的属性
      graphics.setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON);
      graphics.setStroke(new BasicStroke(bold, BasicStroke.CAP_BUTT, BasicStroke.JOIN_BEVEL));
      graphics.drawImage(newJigsawImage, 0, 0, null);
      graphics.dispose();

      ByteArrayOutputStream os = new ByteArrayOutputStream();//新建流。
      ImageIO.write(newJigsawImage, IMAGE_TYPE_PNG, os);//利用ImageIO类提供的write方法，将bi以png图片的数据模式写入流。
      byte[] jigsawImages = os.toByteArray();

      ByteArrayOutputStream oriImagesOs = new ByteArrayOutputStream();//新建流。
      ImageIO.write(originalImage, IMAGE_TYPE_PNG, oriImagesOs);//利用ImageIO类提供的write方法，将bi以jpg图片的数据模式写入流。
      byte[] oriCopyImages = oriImagesOs.toByteArray();
      //设置原生图片base64码
      dataVO.setOriginalImageBase64(Base64Utils.encodeToString(oriCopyImages).replaceAll("\r|\n", ""));
      //设置滑块图片base64码
      dataVO.setJigsawImageBase64(Base64Utils.encodeToString(jigsawImages).replaceAll("\r|\n", ""));
      dataVO.setToken(getUUID());

      //将坐标信息存入redis中
      String codeKey = String.format(RedisKeys.CAPTCHA_CHECK_TOKEN_KEY, dataVO.getToken());
      cacheService.setMCode(codeKey, JSON.toJSONString(point), EXPIRESIN_SECONDS, clientUniqueId);
      LOGGER.debug("token：{},point:{}", dataVO.getToken(), JSON.toJSONString(point));
      return dataVO;
    } catch (Exception e) {
      LOGGER.error("切图失败", e);
      throw new IllegalStateException("切图失败", e);
    }
  }

  /**
   * 随机生成拼图坐标
   *
   * @param originalWidth  原图宽度
   * @param originalHeight 原图高度
   * @param jigsawWidth    滑块图宽度
   * @param jigsawHeight   滑块图高度
   * @return
   */
  public PointDto generateJigsawPoint(int originalWidth, int originalHeight, int jigsawWidth, int jigsawHeight) {
    Random random = new Random();
    int widthDifference = originalWidth - jigsawWidth;
    int heightDifference = originalHeight - jigsawHeight;
    int x, y = 0;
    if (widthDifference <= 0) {
      x = 5;
    } else {
      x = random.nextInt(originalWidth - jigsawWidth - 100) + 100;
    }
    if (heightDifference <= 0) {
      y = 5;
    } else {
      y = random.nextInt(originalHeight - jigsawHeight) + 5;
    }
    return new PointDto(x, y);
  }

  /**
   * 新建的图像根据模板颜色赋值,源图生成遮罩
   *
   * @param oriImage      原图
   * @param templateImage 模板图
   * @param newImage      新抠出的小图
   * @param x             随机扣取坐标X
   * @param y             随机扣取坐标y
   * @throws Exception
   */
  public void cutByTemplate(BufferedImage oriImage, BufferedImage templateImage, BufferedImage newImage, int x, int y) {
    //临时数组遍历用于高斯模糊存周边像素值
    int[][] martrix = new int[3][3];
    int[] values = new int[9];

    int xLength = templateImage.getWidth();
    int yLength = templateImage.getHeight();
    // 模板图像宽度
    for (int i = 0; i < xLength; i++) {
      // 模板图片高度
      for (int j = 0; j < yLength; j++) {
        // 如果模板图像当前像素点不是透明色 copy源文件信息到目标图片中
        int rgb = templateImage.getRGB(i, j);
        if (rgb < 0) {
          newImage.setRGB(i, j, oriImage.getRGB(x + i, y + j));

          //抠图区域高斯模糊
          readPixel(oriImage, x + i, y + j, values);
          fillMatrix(martrix, values);
          oriImage.setRGB(x + i, y + j, avgMatrix(martrix));
        }

        //防止数组越界判断
        if (i == (xLength - 1) || j == (yLength - 1)) {
          continue;
        }
        int rightRgb = templateImage.getRGB(i + 1, j);
        int downRgb = templateImage.getRGB(i, j + 1);
        //描边处理，,取带像素和无像素的界点，判断该点是不是临界轮廓点,如果是设置该坐标像素是白色
        if ((rgb >= 0 && rightRgb < 0) || (rgb < 0 && rightRgb >= 0) || (rgb >= 0 && downRgb < 0) || (rgb < 0 && downRgb >= 0)) {
          newImage.setRGB(i, j, Color.white.getRGB());
          oriImage.setRGB(x + i, y + j, Color.white.getRGB());
        }
      }
    }

  }


  /**
   * 干扰抠图处理
   *
   * @param oriImage      原图
   * @param templateImage 模板图
   * @param x             随机扣取坐标X
   * @param y             随机扣取坐标y
   * @throws Exception
   */
  public void interferenceByTemplate(BufferedImage oriImage, BufferedImage templateImage, int x, int y) {
    //临时数组遍历用于高斯模糊存周边像素值
    int[][] martrix = new int[3][3];
    int[] values = new int[9];

    int xLength = templateImage.getWidth();
    int yLength = templateImage.getHeight();
    // 模板图像宽度
    for (int i = 0; i < xLength; i++) {
      // 模板图片高度
      for (int j = 0; j < yLength; j++) {
        // 如果模板图像当前像素点不是透明色 copy源文件信息到目标图片中
        int rgb = templateImage.getRGB(i, j);
        if (rgb < 0) {
          //抠图区域高斯模糊
          readPixel(oriImage, x + i, y + j, values);
          fillMatrix(martrix, values);
          oriImage.setRGB(x + i, y + j, avgMatrix(martrix));
        }
        //防止数组越界判断
        if (i == (xLength - 1) || j == (yLength - 1)) {
          continue;
        }
        int rightRgb = templateImage.getRGB(i + 1, j);
        int downRgb = templateImage.getRGB(i, j + 1);
        //描边处理，,取带像素和无像素的界点，判断该点是不是临界轮廓点,如果是设置该坐标像素是白色
        if ((rgb >= 0 && rightRgb < 0) || (rgb < 0 && rightRgb >= 0) || (rgb >= 0 && downRgb < 0) || (rgb < 0 && downRgb >= 0)) {
          oriImage.setRGB(x + i, y + j, Color.white.getRGB());
        }
      }
    }

  }

  /**
   * 用于抠图区域高斯模糊
   *
   * @param img
   * @param x
   * @param y
   * @param pixels
   */
  public void readPixel(BufferedImage img, int x, int y, int[] pixels) {
    int xStart = x - 1;
    int yStart = y - 1;
    int current = 0;
    for (int i = xStart; i < 3 + xStart; i++) {
      for (int j = yStart; j < 3 + yStart; j++) {
        int tx = i;
        if (tx < 0) {
          tx = -tx;

        } else if (tx >= img.getWidth()) {
          tx = x;
        }
        int ty = j;
        if (ty < 0) {
          ty = -ty;
        } else if (ty >= img.getHeight()) {
          ty = y;
        }
        pixels[current++] = img.getRGB(tx, ty);

      }
    }
  }

  /**
   * 用于抠图区域高斯模糊
   *
   * @param matrix
   * @param values
   */
  public void fillMatrix(int[][] matrix, int[] values) {
    int filled = 0;
    for (int i = 0; i < matrix.length; i++) {
      int[] x = matrix[i];
      for (int j = 0; j < x.length; j++) {
        x[j] = values[filled++];
      }
    }
  }

  /**
   * 用于抠图区域高斯模糊
   *
   * @param matrix
   * @return
   */
  public int avgMatrix(int[][] matrix) {
    int r = 0;
    int g = 0;
    int b = 0;
    for (int i = 0; i < matrix.length; i++) {
      int[] x = matrix[i];
      for (int j = 0; j < x.length; j++) {
        if (j == 1) {
          continue;
        }
        Color c = new Color(x[j]);
        r += c.getRed();
        g += c.getGreen();
        b += c.getBlue();
      }
    }
    return new Color(r / 8, g / 8, b / 8).getRGB();
  }

  /**
   * 初始化滑块底图，滑块图片
   *
   * @param original 底图路径
   * @param slidingblock 滑块图片路径
   */
  public void initializeBaseMap(String original,String slidingblock) {
    cacheBootImage(getResourcesImagesFile(original + "/*.png"),
        getResourcesImagesFile(slidingblock + "/*.png"));
  }

  /**
   * 初始化底图及滑块图片
   *
   * @param originalMap     原图路径map
   * @param slidingBlockMap 滑块图片文件路径map
   */
  public void cacheBootImage(Map<String, String> originalMap, Map<String, String> slidingBlockMap) {
    originalCacheMap.putAll(originalMap);
    slidingBlockCacheMap.putAll(slidingBlockMap);
    fileNameMap.put(CaptchaBaseMapEnum.ORIGINAL.getCodeValue(), originalCacheMap.keySet().toArray(new String[0]));
    fileNameMap.put(CaptchaBaseMapEnum.SLIDING_BLOCK.getCodeValue(), slidingBlockCacheMap.keySet().toArray(new String[0]));
    LOGGER.info("自定义resource底图:{}", JSON.toJSONString(fileNameMap));
  }

  /**
   * 获取某个路径下的所有文件
   *
   * @param path 文件路径
   * @return
   */
  public Map<String, String> getResourcesImagesFile(String path) {
    Map<String, String> imgMap = new HashMap<>();
    ResourcePatternResolver resolver = new PathMatchingResourcePatternResolver();
    try {
      Resource[] resources = resolver.getResources(path);
      for (Resource resource : resources) {
        byte[] bytes = FileCopyUtils.copyToByteArray(resource.getInputStream());
        String string = Base64Utils.encodeToString(bytes);
        String filename = resource.getFilename();
        imgMap.put(filename, string);
      }
    } catch (Exception e) {
      LOGGER.error("获取路径下的所有文件失败", e);
      throw new IllegalStateException("获取路径下的所有文件失败", e);
    }
    return imgMap;
  }


  /**
   * 验证失败记录失败次数用做接口次数验证
   *
   * @param data
   */
  public void afterValidateFail(CaptchaDto data) {
    // 验证失败 分钟内计数
    String fails = String.format(RedisKeys.CAPTCHA_LIMIT_KEY, "FAIL", data.getClientUniqueId());
    cacheService.getAndIncrement(fails, 1, 60, TimeUnit.SECONDS);
  }


  /**
   * 获取滑块原图的底图
   *
   * @return
   */
  public BufferedImage getOriginal() {
    String[] strings = fileNameMap.get(CaptchaBaseMapEnum.ORIGINAL.getCodeValue());
    if (null == strings || strings.length == 0) {
      return null;
    }
    Integer randomInt = getRandomInt(0, strings.length);
    String s = originalCacheMap.get(strings[randomInt]);
    return getBase64StrToImage(s);
  }

  /**
   * 获取滑块的抠图图片
   *
   * @return
   */
  public String getslidingBlock() {
    String[] strings = fileNameMap.get(CaptchaBaseMapEnum.SLIDING_BLOCK.getCodeValue());
    if (null == strings || strings.length == 0) {
      return null;
    }
    Integer randomInt = getRandomInt(0, strings.length);
    String s = slidingBlockCacheMap.get(strings[randomInt]);
    return s;
  }


  /**
   * base64 字符串转图片
   *
   * @param base64String base64字符串
   * @return
   */
  public BufferedImage getBase64StrToImage(String base64String) {
    byte[] bytes = Base64Utils.decodeFromString(base64String);
    try (ByteArrayInputStream inputStream = new ByteArrayInputStream(bytes)) {
      return ImageIO.read(inputStream);
    } catch (IOException e) {
      LOGGER.error("字符串转图片失败", e);
      throw new IllegalStateException("字符串转图片失败", e);
    }
  }

  /**
   * 生成UUID
   *
   * @return
   */
  public String getUUID() {
    String uuid = UUID.randomUUID().toString();
    uuid = uuid.replace("-", "");
    return uuid;
  }

  /**
   * 随机范围内数字
   *
   * @param startNum 最小数
   * @param endNum   最大数
   * @return
   */
  public Integer getRandomInt(int startNum, int endNum) {
    return ThreadLocalRandom.current().nextInt(endNum - startNum) + startNum;
  }
}
