package com.biz.crm.common.captcha.local.service.internal;

import com.biz.crm.common.captcha.sdk.common.constant.RedisKeys;
import com.biz.crm.common.captcha.sdk.service.CaptchaCacheService;
import com.bizunited.nebula.common.service.redis.RedisMutexService;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;

import java.util.concurrent.TimeUnit;

/**
 * 基于redis的缓存实现类
 *
 * @author zwb
 * @date 2021-09-27
 */
public class CaptchaCacheServiceRedisImpl implements CaptchaCacheService {

  @Autowired
  private RedisMutexService redisMutexService;

  /**
   * 设置缓存值
   *
   * @param key              键
   * @param value            需要存储的值
   * @param expiresInSeconds 过期时间 秒
   * @param clientUniqueId 客户端唯一标识
   */
  @Override
  public void setMCode(String key, String value, long expiresInSeconds, String clientUniqueId) {
    Validate.notBlank(clientUniqueId, "用户标识不能为空");
    String mapName = String.format(RedisKeys.CAPTCHA_MAP_NAME_KEY, clientUniqueId);
    redisMutexService.setMCode(mapName, key, value, expiresInSeconds * 1000);
  }

  /**
   * 判断是否存在这个key
   *
   * @param key
   * @param clientUniqueId 客户端唯一标识
   * @return true存在 false 不存在
   */
  @Override
  public boolean existsMCode(String key, String clientUniqueId) {
    Validate.notBlank(clientUniqueId, "用户标识不能为空");
    String mapName = String.format(RedisKeys.CAPTCHA_MAP_NAME_KEY, clientUniqueId);
    String mCode;
    mCode = redisMutexService.getMCode(mapName, key);
    if (mCode == null) {
      return false;
    } else {
      return true;
    }
  }

  /**
   * 删除某个key
   *
   * @param key
   * @param clientUniqueId 客户端唯一标识
   * @return
   */
  @Override
  public void deleteMCode(String key, String clientUniqueId) {
    Validate.notBlank(clientUniqueId, "用户标识不能为空");
    String mapName = String.format(RedisKeys.CAPTCHA_MAP_NAME_KEY, clientUniqueId);
    redisMutexService.removeMCode(mapName, key);
  }

  /**
   * 获取key对应值
   * @param key
   * @param clientUniqueId 客户端唯一标识
   * @return
   * 有两种情况会返回null：mapName对应的虚拟map结构中，并没有设定指定的key值；另一种是之前设定了，但是已经到了有效期
   */
  @Override
  public String getMCode(String key, String clientUniqueId) {
    Validate.notBlank(clientUniqueId, "用户标识不能为空");
    String mapName = String.format(RedisKeys.CAPTCHA_MAP_NAME_KEY, clientUniqueId);
    return redisMutexService.getMCode(mapName, key);
  }

  /**
   * 根据编码从redis获取一个自增长的数字，并且设置一个过期时间
   * @param code 键
   * @param min 最小值 如果获取的值小于最小值就直接返回最小值
   * @param expire 过期时间
   * @param unit 过期单位
   * @return
   */
  @Override
  public Long getAndIncrement(String code, long min, long expire, TimeUnit unit) {
    return redisMutexService.getAndIncrement(code, min, expire, unit);
  }

  /**
   * 获取当前自增的数据,如果没有则返回null
   *
   * @param code 键
   * @return
   */
  @Override
  public Long getIncrement(String code) {
    return redisMutexService.getIncrement(code);
  }

  /**
   * 使用Redis客户端，在服务器完成一次原子性质的获取和增加操作
   *
   * @param code key键值
   * @param min 最小值 如果获取的值小于最小值就直接返回最小值
   * @return
   */
  @Override
  public long getAndIncrement(String code, long min) {
    return redisMutexService.getAndIncrement(code, min);
  }
}
