package com.biz.crm.common.captcha.local.service.internal;

import com.alibaba.fastjson.JSON;
import com.biz.crm.common.captcha.sdk.common.constant.RedisKeys;
import com.biz.crm.common.captcha.sdk.common.enums.CaptchaBaseMapEnum;
import com.biz.crm.common.captcha.sdk.config.CaptchaProperties;
import com.biz.crm.common.captcha.sdk.dto.CaptchaDto;
import com.biz.crm.common.captcha.sdk.dto.PointDto;
import com.biz.crm.common.captcha.sdk.service.CaptchaCacheService;
import com.biz.crm.common.captcha.sdk.service.CaptchaService;
import com.biz.crm.common.captcha.sdk.service.FrequencyLimitService;
import com.bizunited.nebula.common.util.Aes128Utils;
import org.apache.commons.lang3.Validate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.core.io.Resource;
import org.springframework.core.io.support.PathMatchingResourcePatternResolver;
import org.springframework.core.io.support.ResourcePatternResolver;
import org.springframework.util.Base64Utils;
import org.springframework.util.FileCopyUtils;

import javax.annotation.PostConstruct;
import javax.imageio.ImageIO;
import java.awt.*;
import java.awt.geom.AffineTransform;
import java.awt.image.BufferedImage;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.util.List;
import java.util.*;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;

/**
 * 点选文字验证码
 *
 * @author zwb on 2021/11/09
 */
public class ClickWordCaptchaServiceImpl implements CaptchaService {

  /**
   * 日志
   */
  private static final Logger LOGGER = LoggerFactory.getLogger(ClickWordCaptchaServiceImpl.class);

  //接口限制类
  @Autowired
  private FrequencyLimitService limitHandler;

  //配置文件中的属性
  @Autowired
  private CaptchaProperties prop;

  //缓存实体类
  @Autowired
  private CaptchaCacheService cacheService;

  //加密秘钥
  @Value("${security.aes128.key:1234123412ABCDEF}")
  private String encryptKey;

  //过期两分钟
  private static Long EXPIRESIN_SECONDS = 2 * 60L;

  //过期三分钟
  private static Long EXPIRESIN_THREE = 3 * 60L;

  //底图
  private static Map<String, String> originalCacheMap = new ConcurrentHashMap();

  //点选文字
  private static Map<String, String> picClickCacheMap = new ConcurrentHashMap();

  //文件map
  private static Map<String, String[]> fileNameMap = new ConcurrentHashMap<>();

  //点选文字中可能出现的所有汉字 unicode编码
  private static String HAN_ZI = "\u7684\u4e00\u4e86\u662f\u6211\u4e0d\u5728\u4eba\u4eec\u6709\u6765\u4ed6\u8fd9\u4e0a\u7740\u4e2a\u5730\u5230\u5927\u91cc\u8bf4\u5c31\u53bb\u5b50\u5f97\u4e5f\u548c\u90a3\u8981\u4e0b\u770b\u5929\u65f6\u8fc7\u51fa\u5c0f\u4e48\u8d77\u4f60\u90fd\u628a\u597d\u8fd8\u591a\u6ca1\u4e3a\u53c8\u53ef\u5bb6\u5b66\u53ea\u4ee5\u4e3b\u4f1a\u6837\u5e74\u60f3\u751f\u540c\u8001\u4e2d\u5341\u4ece\u81ea\u9762\u524d\u5934\u9053\u5b83\u540e\u7136\u8d70\u5f88\u50cf\u89c1\u4e24\u7528\u5979\u56fd\u52a8\u8fdb\u6210\u56de\u4ec0\u8fb9\u4f5c\u5bf9\u5f00\u800c\u5df1\u4e9b\u73b0\u5c71\u6c11\u5019\u7ecf\u53d1\u5de5\u5411\u4e8b\u547d\u7ed9\u957f\u6c34\u51e0\u4e49\u4e09\u58f0\u4e8e\u9ad8\u624b\u77e5\u7406\u773c\u5fd7\u70b9\u5fc3\u6218\u4e8c\u95ee\u4f46\u8eab\u65b9\u5b9e\u5403\u505a\u53eb\u5f53\u4f4f\u542c\u9769\u6253\u5462\u771f\u5168\u624d\u56db\u5df2\u6240\u654c\u4e4b\u6700\u5149\u4ea7\u60c5\u8def\u5206\u603b\u6761\u767d\u8bdd\u4e1c\u5e2d\u6b21\u4eb2\u5982\u88ab\u82b1\u53e3\u653e\u513f\u5e38\u6c14\u4e94\u7b2c\u4f7f\u5199\u519b\u5427\u6587\u8fd0\u518d\u679c\u600e\u5b9a\u8bb8\u5feb\u660e\u884c\u56e0\u522b\u98de\u5916\u6811\u7269\u6d3b\u90e8\u95e8\u65e0\u5f80\u8239\u671b\u65b0\u5e26\u961f\u5148\u529b\u5b8c\u5374\u7ad9\u4ee3\u5458\u673a\u66f4\u4e5d\u60a8\u6bcf\u98ce\u7ea7\u8ddf\u7b11\u554a\u5b69\u4e07\u5c11\u76f4\u610f\u591c\u6bd4\u9636\u8fde\u8f66\u91cd\u4fbf\u6597\u9a6c\u54ea\u5316\u592a\u6307\u53d8\u793e\u4f3c\u58eb\u8005\u5e72\u77f3\u6ee1\u65e5\u51b3\u767e\u539f\u62ff\u7fa4\u7a76\u5404\u516d\u672c\u601d\u89e3\u7acb\u6cb3\u6751\u516b\u96be\u65e9\u8bba\u5417\u6839\u5171\u8ba9\u76f8\u7814\u4eca\u5176\u4e66\u5750\u63a5\u5e94\u5173\u4fe1\u89c9\u6b65\u53cd\u5904\u8bb0\u5c06\u5343\u627e\u4e89\u9886\u6216\u5e08\u7ed3\u5757\u8dd1\u8c01\u8349\u8d8a\u5b57\u52a0\u811a\u7d27\u7231\u7b49\u4e60\u9635\u6015\u6708\u9752\u534a\u706b\u6cd5\u9898\u5efa\u8d76\u4f4d\u5531\u6d77\u4e03\u5973\u4efb\u4ef6\u611f\u51c6\u5f20\u56e2\u5c4b\u79bb\u8272\u8138\u7247\u79d1\u5012\u775b\u5229\u4e16\u521a\u4e14\u7531\u9001\u5207\u661f\u5bfc\u665a\u8868\u591f\u6574\u8ba4\u54cd\u96ea\u6d41\u672a\u573a\u8be5\u5e76\u5e95\u6df1\u523b\u5e73\u4f1f\u5fd9\u63d0\u786e\u8fd1\u4eae\u8f7b\u8bb2\u519c\u53e4\u9ed1\u544a\u754c\u62c9\u540d\u5440\u571f\u6e05\u9633\u7167\u529e\u53f2\u6539\u5386\u8f6c\u753b\u9020\u5634\u6b64\u6cbb\u5317\u5fc5\u670d\u96e8\u7a7f\u5185\u8bc6\u9a8c\u4f20\u4e1a\u83dc\u722c\u7761\u5174\u5f62\u91cf\u54b1\u89c2\u82e6\u4f53\u4f17\u901a\u51b2\u5408\u7834\u53cb\u5ea6\u672f\u996d\u516c\u65c1\u623f\u6781\u5357\u67aa\u8bfb\u6c99\u5c81\u7ebf\u91ce\u575a\u7a7a\u6536\u7b97\u81f3\u653f\u57ce\u52b3\u843d\u94b1\u7279\u56f4\u5f1f\u80dc\u6559\u70ed\u5c55\u5305\u6b4c\u7c7b\u6e10\u5f3a\u6570\u4e61\u547c\u6027\u97f3\u7b54\u54e5\u9645\u65e7\u795e\u5ea7\u7ae0\u5e2e\u5566\u53d7\u7cfb\u4ee4\u8df3\u975e\u4f55\u725b\u53d6\u5165\u5cb8\u6562\u6389\u5ffd\u79cd\u88c5\u9876\u6025\u6797\u505c\u606f\u53e5\u533a\u8863\u822c\u62a5\u53f6\u538b\u6162\u53d4\u80cc\u7ec6";

  //点选文字字体格式
  private static String clickWordFontStr = "NotoSerif-Light.ttf";

  //点选文字字体对象
  private static Font clickWordFont;

  //点选文字大小 方便计算文字坐标
  private static int HAN_ZI_SIZE;

  //点选文字一半大小 方便计算文字坐标
  private static int HAN_ZI_SIZE_HALF;


  @PostConstruct
  public void init() {
    //读取配置文件中默认的配置并将其赋值给属性类，此类可用于后续初始化实体类的配置
    LOGGER.info("自定义配置项：{}", prop.toString());
    Validate.notNull(prop, "配置文件为空");
    Validate.notBlank(prop.getCaptchaOriginal(), "文字点选底图路径不能为空");
    Validate.notNull(prop.getFontType(), "点选文字验证码的文字字体不能为空");
    Validate.notNull(prop.getFontSize(), "点选文字验证码的文字大小不能为空");
    Validate.notNull(prop.getFontStyle(), "点选文字验证码的文字样式不能为空");
    Validate.notNull(prop.getSlipOffset(), "点选文字验证码的允许误差偏移量不能为空");
    Validate.notNull(prop.isFontColorRandom(), "点选文字验证码的文字颜色是否随机不能为空");
    Validate.notNull(prop.getReqGetMinuteLimit(), "get接口一分钟内限制访问数不能为空");
    Validate.notNull(prop.getReqCheckMinuteLimit(), "check接口一分钟内限制访问数不能为空");
    Validate.notNull(prop.getReqCheckFailLimit(), "check接口 一分钟内失败次数限制不能为空");
    Validate.notNull(prop.getReqCheckFailLockSeconds(), "check接口验证失败后，接口锁定时间不能为空");
    LOGGER.info("--->>>初始化验证码底图<<<---");
    //resources目录下初始化点击文字验证底图
    this.initializeBaseMap(prop.getCaptchaOriginal());

    //新建一个字体
    clickWordFontStr = prop.getFontType();
    try {
      int size = prop.getFontSize();
      HAN_ZI_SIZE=size;
      HAN_ZI_SIZE_HALF=size/2;
      int style = prop.getFontStyle();
      this.clickWordFont = new Font(clickWordFontStr, style, size);
    } catch (Exception ex) {
      //加载时就出现异常
      LOGGER.error("load font error:{}", ex);
      throw new IllegalStateException("load font error", ex);
    }
  }

  @Override
  public CaptchaDto getCaptcha(CaptchaDto captchaDto) {
    //获取验证码接口限流
    limitHandler.validateGetImg(captchaDto);
    //获取点选底图
    BufferedImage bufferedImage = this.getPicClick();
    Validate.notNull(bufferedImage, "滑动底图未初始化成功，请检查路径");
    //图片转base64 和获取点选文字
    CaptchaDto imageData = getImageData(bufferedImage, captchaDto.getClientUniqueId());
    Validate.notNull(imageData, "获取验证码失败,请联系管理员！");
    Validate.notBlank(imageData.getOriginalImageBase64(), "获取验证码失败,底图为空，请联系管理员！");
    return imageData;
  }

  @Override
  public CaptchaDto checkCaptcha(CaptchaDto captchaDto) {
    Validate.notBlank(captchaDto.getClientUniqueId(), "用户标识不能为空");
    //验证接口限流
    limitHandler.validateCheckImg(captchaDto);
    //取坐标信息
    String codeKey = String.format(RedisKeys.CAPTCHA_CHECK_TOKEN_KEY, captchaDto.getToken());
    Validate.isTrue(cacheService.existsMCode(codeKey, captchaDto.getClientUniqueId()), "验证码已失效，请重新获取！");

    String s = cacheService.getMCode(codeKey, captchaDto.getClientUniqueId());
    //验证码只用一次，即刻失效
    cacheService.deleteMCode(codeKey, captchaDto.getClientUniqueId());
    List<PointDto> point = null;
    List<PointDto> point1 = null;
    String pointJson = null;

    try {
      point = JSON.parseArray(s, PointDto.class);
      //aes解密
      pointJson = Aes128Utils.decrypt(captchaDto.getPointJson(), encryptKey, Aes128Utils.EncodeType.CBC, Aes128Utils.Padding.PKCS_7_PADDING);
      point1 = JSON.parseArray(pointJson, PointDto.class);
    } catch (Exception e) {
      LOGGER.error("验证码坐标解析失败", e);
      afterValidateFail(captchaDto);
      throw new IllegalStateException("验证码坐标解析失败");
    }
    for (int i = 0; i < point.size(); i++) {
      if (point.get(i).x - HAN_ZI_SIZE > point1.get(i).x
          || point1.get(i).x > point.get(i).x + HAN_ZI_SIZE
          || point.get(i).y - HAN_ZI_SIZE > point1.get(i).y
          || point1.get(i).y > point.get(i).y + HAN_ZI_SIZE) {
        afterValidateFail(captchaDto);
        throw new IllegalStateException("验证失败！");
      }
    }
    //校验成功，将信息存入缓存
    String value = null;
    try {
      value = Aes128Utils.encrypt(captchaDto.getToken().concat("---").concat(pointJson), encryptKey, Aes128Utils.EncodeType.CBC, Aes128Utils.Padding.PKCS_7_PADDING);
    } catch (Exception e) {
      LOGGER.error("AES加密失败", e);
      afterValidateFail(captchaDto);
      throw new IllegalStateException("AES加密失败");
    }
    String secondKey = String.format(RedisKeys.CAPTCHA_REDIS_SECOND_CHECK_KEY, value);
    cacheService.setMCode(secondKey, captchaDto.getToken(), EXPIRESIN_THREE, captchaDto.getClientUniqueId());
    captchaDto.setResult(true);
    captchaDto.resetClientFlag();
    return captchaDto;
  }

  @Override
  public void verificationCaptcha(String captchaVerification, String clientUniqueId) {
    try {
      Validate.notBlank(clientUniqueId, "用户标识不能为空");
      Validate.notBlank(captchaVerification, "二次校验码不能为空");
      String codeKey = String.format(RedisKeys.CAPTCHA_REDIS_SECOND_CHECK_KEY, captchaVerification);
      Validate.isTrue(cacheService.existsMCode(codeKey, clientUniqueId), "二次验证失败");
      //二次校验取值后，即刻失效
      cacheService.deleteMCode(codeKey, clientUniqueId);
    } catch (Exception e) {
      LOGGER.error("验证码坐标解析失败", e);
      throw new IllegalStateException("验证码坐标解析失败");
    }
  }


  /**
   * 根据背景图获取图片及点选文字
   * @param backgroundImage
   * @param clientUniqueId
   * @return
   */
  private CaptchaDto getImageData(BufferedImage backgroundImage, String clientUniqueId) {
    CaptchaDto dataVO = new CaptchaDto();
    List<String> wordList = new ArrayList<String>();
    List<PointDto> pointList = new ArrayList();

    Graphics backgroundGraphics = backgroundImage.getGraphics();
    int width = backgroundImage.getWidth();
    int height = backgroundImage.getHeight();

    int wordCount = prop.getWordCount();
    //定义随机1到arr.length某一个字不参与校验
    int num = this.getRandomInt(1, wordCount);
    Set<String> currentWords = getRandomWords(wordCount);
    int i = 0;
    for (String word : currentWords) {
      //随机字体坐标
      PointDto point = randomWordPoint(width, height, i, wordCount);
      //随机字体颜色
      if (prop.isFontColorRandom()) {
        backgroundGraphics.setColor(new Color(this.getRandomInt(1, 255),
            this.getRandomInt(1, 255), this.getRandomInt(1, 255)));
      } else {
        backgroundGraphics.setColor(Color.BLACK);
      }
      //设置角度
      AffineTransform affineTransform = new AffineTransform();
      affineTransform.rotate(Math.toRadians(this.getRandomInt(-45, 45)), 0, 0);
      Font rotatedFont = clickWordFont.deriveFont(affineTransform);
      backgroundGraphics.setFont(rotatedFont);
      backgroundGraphics.drawString(word, point.getX(), point.getY());

      if ((num - 1) != i) {
        wordList.add(word);
        pointList.add(point);
      }
      i++;
    }


    //创建合并图片
    BufferedImage combinedImage = new BufferedImage(width, height, BufferedImage.TYPE_INT_RGB);
    Graphics combinedGraphics = combinedImage.getGraphics();
    combinedGraphics.drawImage(backgroundImage, 0, 0, null);

    dataVO.setOriginalImageBase64(this.getImageToBase64Str(backgroundImage).replaceAll("\r|\n", ""));
    //pointList信息不传到前端，只做后端check校验
    dataVO.setWordList(wordList);
    dataVO.setToken(this.getUUID());
    //将坐标信息存入redis中
    String codeKey = String.format(RedisKeys.CAPTCHA_CHECK_TOKEN_KEY, dataVO.getToken());
    cacheService.setMCode(codeKey, JSON.toJSONString(pointList), EXPIRESIN_SECONDS, clientUniqueId);
    return dataVO;
  }

  /**
   * 从所有汉字中随机获取指定个数的汉字
   * @param wordCount 需要获取的汉字个数
   * @return
   */
  private Set<String> getRandomWords(int wordCount) {
    Set<String> words = new HashSet<>();
    int size = HAN_ZI.length();
    for (; ; ) {
      String t = HAN_ZI.charAt(this.getRandomInt(size)) + "";
      words.add(t);
      if (words.size() >= wordCount) {
        break;
      }
    }
    return words;
  }

  /**
   * 随机字体循环排序下标
   *
   * @param imageWidth    图片宽度
   * @param imageHeight   图片高度
   * @param wordSortIndex 字体循环排序下标(i)
   * @param wordCount     字数量
   * @return
   */
  private PointDto randomWordPoint(int imageWidth, int imageHeight, int wordSortIndex, int wordCount) {
    int avgWidth = imageWidth / (wordCount + 1);
    int x, y;
    if (avgWidth < HAN_ZI_SIZE_HALF) {
      x = this.getRandomInt(1 + HAN_ZI_SIZE_HALF, imageWidth);
    } else {
      if (wordSortIndex == 0) {
        x = this.getRandomInt(1 + HAN_ZI_SIZE_HALF, avgWidth * (wordSortIndex + 1) - HAN_ZI_SIZE_HALF);
      } else {
        x = this.getRandomInt(avgWidth * wordSortIndex + HAN_ZI_SIZE_HALF, avgWidth * (wordSortIndex + 1) - HAN_ZI_SIZE_HALF);
      }
    }
    y = this.getRandomInt(HAN_ZI_SIZE, imageHeight - HAN_ZI_SIZE);
    return new PointDto(x, y);
  }

  /**
   * 随机范围内数字
   *
   * @param startNum 最小数
   * @param endNum   最大数
   * @return
   */
  public Integer getRandomInt(int startNum, int endNum) {
    return ThreadLocalRandom.current().nextInt(endNum - startNum) + startNum;
  }

  /**
   * 获取[0,bound) 之间的数字
   * @param bound
   * @return
   */
  public static int getRandomInt(int bound) {
    return ThreadLocalRandom.current().nextInt(bound);
  }

  /**
   * 图片转base64 字符串
   *
   * @param templateImage
   * @return
   */
  public static String getImageToBase64Str(BufferedImage templateImage) {

    byte[] bytes=null;
    try (ByteArrayOutputStream baos = new ByteArrayOutputStream();){
      ImageIO.write(templateImage, "png", baos);
      bytes= baos.toByteArray();
    } catch (IOException e) {
      LOGGER.error("图片转base64失败", e);
      throw new IllegalStateException("图片转base64失败", e);
    }
    return Base64Utils.encodeToString(bytes).trim();
  }


  /**
   * 获取一张随机的背景图片
   * @return
   */
  public BufferedImage getPicClick() {
    String[] strings = fileNameMap.get(CaptchaBaseMapEnum.PIC_CLICK.getCodeValue());
    if (null == strings || strings.length == 0) {
      return null;
    }
    Integer randomInt = this.getRandomInt(0, strings.length);
    String s = picClickCacheMap.get(strings[randomInt]);
    return getBase64StrToImage(s);
  }

  /**
   * base64 字符串转图片
   *
   * @param base64String base64字符串
   * @return
   */
  public BufferedImage getBase64StrToImage(String base64String) {
    byte[] bytes = Base64Utils.decodeFromString(base64String);
    try (ByteArrayInputStream inputStream = new ByteArrayInputStream(bytes)) {
      return ImageIO.read(inputStream);
    } catch (IOException e) {
      LOGGER.error("字符串转图片失败", e);
      throw new IllegalStateException("字符串转图片失败", e);
    }
  }

  /**
   * 生成UUID
   *
   * @return
   */
  public String getUUID() {
    String uuid = UUID.randomUUID().toString();
    uuid = uuid.replace("-", "");
    return uuid;
  }

  /**
   * 初始化文字点选底图
   *
   * @param original 底图路径
   */
  public void initializeBaseMap(String original) {
    cacheBootImage(getResourcesImagesFile(original + "/*.png"));
  }

  /**
   * 初始化点选文字底图
   *
   * @param picClickMap 点选文字路径map
   */
  public void cacheBootImage( Map<String, String> picClickMap) {
    picClickCacheMap.putAll(picClickMap);
    fileNameMap.put(CaptchaBaseMapEnum.PIC_CLICK.getCodeValue(), picClickCacheMap.keySet().toArray(new String[0]));
    LOGGER.info("自定义resource底图:{}", JSON.toJSONString(fileNameMap));
  }

  /**
   * 获取某个路径下的所有文件
   *
   * @param path 文件路径
   * @return
   */
  public Map<String, String> getResourcesImagesFile(String path) {
    Map<String, String> imgMap = new HashMap<>();
    ResourcePatternResolver resolver = new PathMatchingResourcePatternResolver();
    try {
      Resource[] resources = resolver.getResources(path);
      for (Resource resource : resources) {
        byte[] bytes = FileCopyUtils.copyToByteArray(resource.getInputStream());
        String string = Base64Utils.encodeToString(bytes);
        String filename = resource.getFilename();
        imgMap.put(filename, string);
      }
    } catch (Exception e) {
      LOGGER.error("获取路径下的所有文件失败", e);
      throw new IllegalStateException("获取路径下的所有文件失败", e);
    }
    return imgMap;
  }


  /**
   * 验证失败记录失败次数用做接口次数验证
   *
   * @param data
   */
  public void afterValidateFail(CaptchaDto data) {
    // 验证失败 分钟内计数
    String fails = String.format(RedisKeys.CAPTCHA_LIMIT_KEY, "FAIL", data.getClientUniqueId());
    cacheService.getAndIncrement(fails, 1, 60, TimeUnit.SECONDS);
  }

}
