package com.biz.crm.common.sequese.local.algorithm.db.service;

import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import org.mybatis.spring.MyBatisSystemException;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.dao.DuplicateKeyException;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;
import com.biz.crm.common.sequese.local.algorithm.db.dto.CommonSequenceNextvalDto;
import com.biz.crm.common.sequese.local.algorithm.db.entity.CommonSequenceEntity;
import com.biz.crm.common.sequese.local.algorithm.db.mapper.CommonSequenceMapper;
import com.biz.crm.common.sequese.sdk.generator.service.aigorithm.CrmSequeseGeneratorByDb;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import lombok.extern.slf4j.Slf4j;

/**
 * 仿Oracle序列号生成器，依赖数据库的吞吐量
 * 
 * @author Ken.xu
 * @version 1.0 Copyright 2023年4月23日 下午4:39:37
 */
@Slf4j
@Component
public class CrmSequeseGeneratorByDbImpl implements CrmSequeseGeneratorByDb {
  @Autowired
  private CommonSequenceMapper commonSequenceMapper;

  @Override
  public void initAlgorithm() {
    log.info("初始化仿Oracle序列号生成器");
  }

  @Override
  @Transactional(propagation = Propagation.REQUIRES_NEW)
  public Integer nextVal(String subSystem, String bizCode) {
    String key = this.getKey(subSystem, bizCode);
    return this.nextvalExe(key, 1);
  }

  @Override
  public Integer[] nextValArray(String subSystem, String bizCode, int seqNum) {
    // 数据纠正
    if (seqNum < 1) {
      seqNum = 1;
    }
    String key = this.getKey(subSystem, bizCode);
    Integer lastSeqVal = this.nextvalExe(key, seqNum);
    // 批量赋值
    Integer[] batchNextVal = new Integer[seqNum];
    lastSeqVal++;
    for (int idx = 0; idx < batchNextVal.length; idx++) {
      batchNextVal[idx] = lastSeqVal - seqNum + idx; // 对获取到的序列值进行还原序列
    }
    return batchNextVal;
  }

  @Override
  public Integer currVal(String subSystem, String bizCode) {
    String key = this.getKey(subSystem, bizCode);
    return this.currvalExe(key);
  }

  @Override
  public String getKey(String subSystem, String bizCode) {
    // String tenantCode = getTenantCode();
    return String.format("%s:%s", subSystem, bizCode);
  }

  /**
   * 当前算法规则
   * @return
   */
  @Override
  public String getSequeseRuleCode() {
    return "数据库算法";
  }

  // ===================

  /**
   * 获取当前序列值
   * 
   * @param seqName
   * @return
   */
  private Integer currvalExe(String seqName) {
    String tenantCode = TenantUtils.getTenantCode();
    Integer val = commonSequenceMapper.currval(tenantCode, seqName);
    if (val == null) {
      createSeq(tenantCode, seqName);
    }
    val = commonSequenceMapper.currval(tenantCode, seqName);
    return val;
  }

  /**
   * 获取下一个序列号[租户隔离]
   * 
   * @param seqName
   * @return
   */
  private Integer nextvalExe(String seqName, int incrementBy) {
    String tenantCode = TenantUtils.getTenantCode();
    CommonSequenceNextvalDto dto = new CommonSequenceNextvalDto();
    dto.setSeqName(seqName);
    dto.setTenantCode(tenantCode);
    dto.setIncrementBy(incrementBy);
    Integer affectRow;
    try {
      affectRow = commonSequenceMapper.nextval(dto);
    } catch (MyBatisSystemException mse) {
      String message = mse.getMessage();
      if (message.contains("SelectKey returned no data.")) {
        affectRow = 0;// 没有数据
      } else
        throw mse;
    }
    if (affectRow == 0) { // 查无序列
      // 创建序列,在获取下一个值
      createSeq(tenantCode, seqName);
      // 再次获取
      commonSequenceMapper.nextval(dto);
    }
    return dto.getSeqValue();
  }

  private final Lock lock = new ReentrantLock();

  /**
   * 创建序列
   * 
   * @param tenantCode
   * @param seqName
   * @return
   */
  private boolean createSeq(String tenantCode, String seqName) {
    CommonSequenceNextvalDto dto = new CommonSequenceNextvalDto();
    dto.setSeqName(seqName);
    dto.setTenantCode(tenantCode);
    boolean createOk = false;
    lock.lock();
    try {
      do {
        final Integer currval = commonSequenceMapper.currval(tenantCode, seqName);
        if (currval != null) {
          createOk = true;
          break;
        }
        // 创建序列
        CommonSequenceEntity entity = new CommonSequenceEntity();
        entity.setTenantCode(tenantCode);
        entity.setSeqName(seqName);
        int insert = commonSequenceMapper.insert(entity);
        if (insert != 0) {
          createOk = true;
          break;
        }
      } while (false);
    } catch (DuplicateKeyException e) { // 数据已经存在，可以继续了
      createOk = true;
    } finally {
      lock.unlock();
    }
    return createOk;
  }

}
