package com.biz.crm.common.sequese.local.generator;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.stereotype.Service;
import com.biz.crm.common.sequese.local.config.CommonSequeseConfig;
import com.biz.crm.common.sequese.sdk.generator.constant.SequeseConstant;
import com.biz.crm.common.sequese.sdk.generator.service.CrmSequeseFactoryService;
import com.biz.crm.common.sequese.sdk.generator.service.base.CrmBizSequenceServiceByNoParam;
import com.biz.crm.common.sequese.sdk.generator.service.base.CrmBizSequenceServiceByStrategy;
import cn.hutool.core.text.CharSequenceUtil;
import lombok.Getter;
import lombok.extern.slf4j.Slf4j;

/**
 * 序列号工程，统一使用该工厂服务进行[策略模式]
 * 
 * @author xzkne
 *
 */
@Slf4j
@Service
public class CrmSequeseFactoryServiceImpl implements CrmSequeseFactoryService {
  private static final Lock SEQ_SERV_MAP_LOCK = new ReentrantLock(true);
  private static Map<String, CrmBizSequenceServiceByStrategy> crmBizSequenceServiceMap;
  @Getter
  private List<CrmBizSequenceServiceByStrategy> crmBizSequenceServiceList;

  @Autowired
  private RedisTemplate<String, String> redisTemplate;
  @Autowired
  private CrmBizSequenceServiceByNoParam crmBizSequenceServiceByNoParam;

  @Autowired(required = false)
  public void setCrmBizSequenceServiceList(List<CrmBizSequenceServiceByStrategy> list) {
    this.crmBizSequenceServiceList = list;
    crmBizSequenceServiceMap = new HashMap<>(list.size());
  }

  @Override
  public <T> CrmBizSequenceServiceByStrategy getCrmBizSequenceService(T dto) {
    if (dto == null) {
      return null;
    }
    String seqServMapKey = dto.getClass().getName();
    CrmBizSequenceServiceByStrategy seqServ = null;
    // 获取缓存
    Map<String, CrmBizSequenceServiceByStrategy> map = this.crmBizSequenceServiceMap;
    seqServ = map.get(seqServMapKey);
    if (seqServ != null) {
      return seqServ;
    }
    // 如果没有缓存
    SEQ_SERV_MAP_LOCK.lock();
    try {
      // 二次检查是否有并发程序先行进入设置了map
      seqServ = map.get(seqServMapKey);
      if (seqServ != null) {
        return seqServ;
      }
      // 从list对象做策略检查，并缓存
      for (CrmBizSequenceServiceByStrategy oneSeqServ : crmBizSequenceServiceList) {
        boolean match = oneSeqServ.match(dto);
        if (match == true) {
          seqServ = oneSeqServ;
          map.put(seqServMapKey, oneSeqServ);// 记录缓存，下次方便调用
          break;
        }
      }
    } finally {
      SEQ_SERV_MAP_LOCK.unlock();
    }
    return seqServ;
  }

  @Override
  public <T> String nextVal(T dto) throws ClassCastException {
    CrmBizSequenceServiceByStrategy seqServ = this.getCrmBizSequenceService(dto);
    if (seqServ == null) {
      return this.nextVal();
    }
    String nextVal = seqServ.nextVal(dto);
    autoSaveRedis(seqServ.getSeqInfoByBizCode(), nextVal);
    return nextVal;
  }

  @Override
  public <T> String[] nextVal(T dto, int seqNum) throws ClassCastException {
    CrmBizSequenceServiceByStrategy seqServ = this.getCrmBizSequenceService(dto);
    if (seqServ == null) {
      return this.nextValArray(seqNum);
    }
    String[] nextValArray = seqServ.nextVal(dto, seqNum);
    autoSaveRedis(seqServ.getSeqInfoByBizCode(), nextValArray[nextValArray.length - 1]);
    // 3. 返回给前端页面
    return nextValArray;
  }

  @Override
  public <T> String nextVal() throws ClassCastException {
    // 系统默认实现，不记录redis
    return crmBizSequenceServiceByNoParam.nextVal();
  }

  @Override
  public <T> String[] nextValArray(int seqNum) throws ClassCastException {
    // 系统默认实现，不记录redis
    return crmBizSequenceServiceByNoParam.nextValArray(seqNum);
  }

  @Override
  public <T> String currVal(T dto) {
    CrmBizSequenceServiceByStrategy seqServ = this.getCrmBizSequenceService(dto);
    if (seqServ == null) {
      return null;
    }
    String cacheKey = CharSequenceUtil.format("{}:{}", CommonSequeseConfig.subsystem, seqServ.getSeqInfoByBizCode());
    String key = CharSequenceUtil.format(SequeseConstant.SEQ_CACHE_VAL, cacheKey);
    String currVal = redisTemplate.opsForValue().get(key);
    return currVal;
  }

  @Override
  public <T> boolean redo(T dto, String... sequenceValue) {
    CrmBizSequenceServiceByStrategy seqServ = this.getCrmBizSequenceService(dto);
    if (seqServ == null) {
      return false;// 默认不支持redo
    }
    return seqServ.redo(sequenceValue);
  }

  /**
   * 根据缓存记录，自动保存到redis
   * 
   * @param seqServiceName
   * @param nextVal 该对象将以String对象的形式存储到redis中
   */
  private void autoSaveRedis(String seqInfoByBizCode, String nextVal) {
    // 保存单个
    String cacheKey = CharSequenceUtil.format("{}:{}", CommonSequeseConfig.subsystem, seqInfoByBizCode);
    // 判断刷新标识是否存在 存在就不刷新
//    if (existsKey(cacheKey)) {
      this.saveRedis(cacheKey, "" + nextVal);
//    }
  }

  /**
   * 2秒内不会重复查redis
   * 
   * @param cacheKey
   * @return
   */
  private boolean existsKey(String cacheKey) {
    Boolean exists = CommonSequeseConfig.lastLoadRedisCache.get(cacheKey);
    if (exists == null) {
      synchronized (CommonSequeseConfig.lastLoadRedisCache) {
        exists = CommonSequeseConfig.lastLoadRedisCache.get(cacheKey);
        if (exists == null) {
          String saveKey = CharSequenceUtil.format(SequeseConstant.SEQ_CACHE_SAVE, cacheKey);
          exists = exists(saveKey) || exists(CommonSequeseConfig.SAVE_ALL_KEY);
          CommonSequeseConfig.lastLoadRedisCache.put(cacheKey, exists);
          log.debug("获取一次Redis");
          System.out.println("获取一次Redis");
        }
      }
    }
    return exists;
  }

  /**
   * 判断key是否存在
   * 
   * @param key
   * @return 如果存在返回true
   */
  private boolean exists(String key) {
    Boolean hasKey = redisTemplate.hasKey(key);
    return hasKey;
  }

  /**
   * 保存到redis缓存中
   * 
   * @param cacheKey
   * @param nextVal
   * @return
   */
  private void saveRedis(String cacheKey, String nextVal) {
    Boolean isSave = CommonSequeseConfig.saveLashSeqCache.get(cacheKey);
    if (isSave == null) {
      synchronized (CommonSequeseConfig.saveLashSeqCache) {
        isSave = CommonSequeseConfig.saveLashSeqCache.get(cacheKey);
        if (isSave == null) {
          String key = CharSequenceUtil.format(SequeseConstant.SEQ_CACHE_VAL, cacheKey);
          redisTemplate.opsForValue().set(key, nextVal);
          CommonSequeseConfig.saveLashSeqCache.put(cacheKey, Boolean.TRUE);
          log.debug("保存Redis");
        }
      }
    }
  }
}
