package com.biz.crm.common.errorlog.sdk.utils;


import com.biz.crm.common.errorlog.sdk.dto.ErrorInfo;
import org.elasticsearch.index.query.BoolQueryBuilder;
import org.elasticsearch.index.query.QueryBuilders;
import org.elasticsearch.index.query.RangeQueryBuilder;
import org.elasticsearch.index.query.TermQueryBuilder;
import org.springframework.beans.BeansException;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.data.elasticsearch.core.ElasticsearchRestTemplate;
import org.springframework.data.elasticsearch.core.SearchHits;
import org.springframework.data.elasticsearch.core.mapping.IndexCoordinates;
import org.springframework.data.elasticsearch.core.query.NativeSearchQuery;
import org.springframework.stereotype.Component;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

/**
 * @author HanJiaJun
 * @describe: 用于错误日志的ES工具类
 * @createTime 2022年05月12日 11:06:00
 */


@Component
public class EsUtils implements ApplicationContextAware {

  /**
   * 手动从Factory中获取ES连接对象
   */
  private static ApplicationContext applicationContext = null;
  private static ElasticsearchRestTemplate template = null;
  // 默认索引名称
  public static String DEFAULT_INDEX_NAME = "log_error";

  @Override
  public void setApplicationContext(ApplicationContext applicationContext) throws BeansException {
    EsUtils.applicationContext = applicationContext;
  }

  /**
   * 获取连接
   *
   * @return
   */
  public static ElasticsearchRestTemplate getTemplate() {
    ElasticsearchRestTemplate bean = applicationContext.getBean(ElasticsearchRestTemplate.class);
    template = bean;
    return bean;
  }

  /**
   * 创建错误日志索引
   *
   * @param
   * @param index
   * @throws IOException
   */
  public static void createIndex(String index) throws IOException {
    if (template == null) {
      getTemplate();
    }
    if (!isIndexExist()) {
      template.indexOps(IndexCoordinates.of("log_error")).create();
    }
  }

  /**
   * 判断索引是否存在
   *
   * @param
   * @return
   */
  public static boolean isIndexExist() {
    if (template == null) {
      getTemplate();
    }
    IndexCoordinates indexCoordinates = IndexCoordinates.of(DEFAULT_INDEX_NAME);
    return template.indexOps(indexCoordinates).exists();
  }

  /**
   * 添加错误信息
   *
   * @param info
   * @throws IOException
   */
  public static void addErrorMsg(ErrorInfo info) throws IOException {
    if (template == null) {
      getTemplate();
    }
    isIndexExist();
    template.save(info, IndexCoordinates.of(DEFAULT_INDEX_NAME));
  }

  /**
   * 根据sources查询全部错误信息
   *
   * @return
   */
  public static List<ErrorInfo> search(String sources) {
    if (template == null) {
      getTemplate();
    }
    TermQueryBuilder queryBuilder = QueryBuilders.termQuery("type.keyword", sources);
    NativeSearchQuery nativeSearchQuery = new NativeSearchQuery(queryBuilder);
    nativeSearchQuery.setMaxResults(10000);
    SearchHits<ErrorInfo> search = template.search(nativeSearchQuery, ErrorInfo.class, IndexCoordinates.of(DEFAULT_INDEX_NAME));
    List<ErrorInfo> res = new ArrayList<>();
    search.forEach(e -> {
      res.add(e.getContent());
    });
    return res;
  }


  /**
   * 删除过期数据
   *
   * @param overtime 过期分钟数
   */
  public static void delete(int overtime) {
    RangeQueryBuilder date = QueryBuilders.rangeQuery("date").lte(System.currentTimeMillis() - 60000 * overtime);
    NativeSearchQuery query = new NativeSearchQuery(new BoolQueryBuilder().must(date));
    template.delete(query, ErrorInfo.class, IndexCoordinates.of(DEFAULT_INDEX_NAME));
  }
}
