package com.biz.crm.common.form.local.controller;

import java.util.List;
import java.util.Map;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.util.CollectionUtils;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.biz.crm.business.common.sdk.model.Result;
import com.biz.crm.common.form.sdk.DynamicFormFieldMappingService;
import com.biz.crm.common.form.sdk.vo.DynamicChildrenFormVo;
import com.biz.crm.common.form.sdk.vo.DynamicFieldVo;
import com.biz.crm.common.form.sdk.vo.DynamicFormVo;
import com.biz.crm.common.form.sdk.widget.WidgetKey;
import com.google.common.collect.Lists;

import io.swagger.annotations.ApiOperation;
import lombok.extern.slf4j.Slf4j;

@Slf4j
@RestController
@RequestMapping(value = {"/v1/dynamicform/dynamicForms"})
public class DynamicFormController {
  
  @Autowired
  private DynamicFormFieldMappingService dynamicFormFieldMappingService;
  
  @ApiOperation(value = "按照动态表单在系统中唯一的业务编码进行查询，注意，这种查询方式，将忽略对字段选择范围的过滤。意思就是说动态表单中所有的业务字段都将被返回")
  @GetMapping(value = "findByDynamicFormCode")
  public Result<?> findByDynamicFormCode(@RequestParam("dynamicFormCode") String dynamicFormCode) {
    try {
      DynamicFormVo dynamicForm = dynamicFormFieldMappingService.findByDynamicFormCode(dynamicFormCode);
      if(dynamicForm == null) {
        return Result.ok();
      }
      // 注意WidgetKey转json的问题，其中没有GET/SET方法，而是要自行构建JSON结构
      return Result.ok(this.buildJson(dynamicForm));
    } catch(RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }
  
  @ApiOperation(value = "按照动态表单在系统中唯一的业务编码和为这个表单设置的字段映射编号进行查询，这种查询结果将只会返回这个动态表单中和这个映射编号绑定的字段信息")
  @GetMapping(value = "findByDynamicFormCodeAndMappingCode")
  public Result<?> findByDynamicFormCodeAndMappingCode(@RequestParam("dynamicFormCode") String dynamicFormCode
                                                       , @RequestParam("mappingCode")  String mappingCode) {
    try {
      DynamicFormVo dynamicForm = dynamicFormFieldMappingService.findByDynamicFormCodeAndMappingCode(dynamicFormCode, mappingCode);
      if(dynamicForm == null) {
        return Result.ok();
      }
      // 注意WidgetKey转json的问题，其中没有GET/SET方法，而是要自行构建JSON结构
      return Result.ok(this.buildJson(dynamicForm));
    } catch(RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }
  
  // TODO 在多次尝试@JSONField serializeUsing等多种序列化设定不起作用后
  // 暂时使用该方法进行人工转换
  private JSONObject buildJson(DynamicFormVo dynamicForm) {
    JSONObject result = new JSONObject();
    result.put("dynamicFormCode", dynamicForm.getDynamicFormCode());
    
    // 处理本身的字段信息
    List<DynamicFieldVo> dynamicFields = dynamicForm.getDynamicFields();
    if(!CollectionUtils.isEmpty(dynamicFields)) {
      JSONArray dynamicFieldJsons = buildDynamicFieldJsons(dynamicFields);
      result.put("dynamicFields", dynamicFieldJsons);
    }
    
    // 处理 dynamicForm.getChildrenForms()，注意，这里需要递归处理
    List<DynamicChildrenFormVo> childrenForms = dynamicForm.getChildrenForms();
    if(!CollectionUtils.isEmpty(childrenForms)) {
      JSONArray dynamicChildrenFormJsons = this.buildDynamicChildrenFormJsons(childrenForms);
      result.put("childrenForms", dynamicChildrenFormJsons);
    }
    return result;
  }
  
  /**
   * 该私有方法可对某模型下的子级模型进行json输出处理
   * @param childrenForms
   * @return
   */
  private JSONArray buildDynamicChildrenFormJsons(List<DynamicChildrenFormVo> childrenForms) {
    JSONArray results = new JSONArray();
    if(CollectionUtils.isEmpty(childrenForms)) {
      return results;
    }
    
    for (DynamicChildrenFormVo dynamicChildrenForm : childrenForms) {
      JSONObject dynamicChildrenFormJson = new JSONObject();
      dynamicChildrenFormJson.put("parentFieldCode", dynamicChildrenForm.getParentFieldCode());
      
      // 处理本身的字段
      List<DynamicFieldVo> childDynamicFields = dynamicChildrenForm.getDynamicFields();
      if(!CollectionUtils.isEmpty(childDynamicFields)) {
        JSONArray dynamicFieldJsons = buildDynamicFieldJsons(childDynamicFields); 
        dynamicChildrenFormJson.put("dynamicFields", dynamicFieldJsons);
      }
      
      // 处理下级字段
      List<DynamicChildrenFormVo> dynamicChildrenForms = dynamicChildrenForm.getChildrenForms();
      if(!CollectionUtils.isEmpty(dynamicChildrenForms)) {
        JSONArray dynamicChildrenFormJsons = this.buildDynamicChildrenFormJsons(dynamicChildrenForms);
        dynamicChildrenFormJson.put("childrenForms", dynamicChildrenFormJsons);
      }
      results.add(dynamicChildrenFormJson);
    }
    return results;
  }
  
  private JSONArray buildDynamicFieldJsons(List<DynamicFieldVo> dynamicFields) {
    JSONArray dynamicFieldJsons = new JSONArray();
    if(CollectionUtils.isEmpty(dynamicFields)) {
      return dynamicFieldJsons;
    }
    
    for (DynamicFieldVo dynamicFieldVo : dynamicFields) {
      Class<?> fieldClass = dynamicFieldVo.getFieldClass();
      Boolean array = dynamicFieldVo.getArray();
      Boolean collection = dynamicFieldVo.getCollection();
      String fieldName = dynamicFieldVo.getFieldName();
      String fieldCode = dynamicFieldVo.getFieldCode();
      boolean required = dynamicFieldVo.isRequired();
      boolean modifiable = dynamicFieldVo.isModifiable();
      Integer sortIndex = dynamicFieldVo.getSortIndex();
      String groupIndex = dynamicFieldVo.getGroupIndex();
      // 处理controllKey
      WidgetKey controllKey = dynamicFieldVo.getControllKey();
      JSONObject widgetKeyJson = new JSONObject();
      String widgetCode = controllKey.widgetCode();
      String widgetName = controllKey.widgetName();
      Map<String , Object> widgetParam = controllKey.widgetParam();
      widgetKeyJson.put("widgetCode", widgetCode);
      widgetKeyJson.put("widgetName", widgetName);
      widgetKeyJson.put("widgetParam", widgetParam);
      
      // 开始生成dynamicFieldVo的json
      JSONObject dynamicFieldJson = new JSONObject();
      dynamicFieldJson.put("fieldClass", fieldClass);
      dynamicFieldJson.put("array", array);
      dynamicFieldJson.put("collection", collection);
      dynamicFieldJson.put("fieldName", fieldName);
      dynamicFieldJson.put("fieldCode", fieldCode);
      dynamicFieldJson.put("required", required);
      dynamicFieldJson.put("modifiable", modifiable);
      dynamicFieldJson.put("collection", collection);
      dynamicFieldJson.put("controllKey", widgetKeyJson);
      dynamicFieldJson.put("sortIndex", sortIndex);
      dynamicFieldJson.put("groupIndex", groupIndex);
      
      // 进行子表的选择
      dynamicFieldJsons.add(dynamicFieldJson);
    }
    return dynamicFieldJsons;
  }
  
  @ApiOperation(value = "该方法将建立/重新建立指定动态表单下，基于某个映射编号（可以是任意业务意义的字符串），所绑定的动态表单字段信息。注意，这种情况下，表示所有选定的字段都是非必填")
  @PostMapping(value = "rebinding")
  public Result<?> rebinding(@RequestParam("dynamicFormCode") String dynamicFormCode
                             , @RequestParam("mappingCode")  String mappingCode
                             , @RequestParam("fieldCode")  String[] fieldCodes) {
    try {
      dynamicFormFieldMappingService.rebinding(dynamicFormCode, mappingCode, Lists.newArrayList(fieldCodes));
      return Result.ok();
    } catch(RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }
  
  @ApiOperation(value = "该方法将建立/重新建立指定动态表单下，基于某个映射编号（可以是任意业务意义的字符串），所绑定的动态表单字段信息。")
  @PostMapping(value = "rebindingSetNullables")
  public Result<?> rebinding(@RequestParam("dynamicFormCode") String dynamicFormCode
                             , @RequestParam("mappingCode")  String mappingCode
                             , @RequestParam("fieldCode")  String[] fieldCodes
                             , @RequestParam(name = "nullable" , required = false)  Boolean[] nullables
                             , @RequestParam(name = "groupIndex" , required = false)  String[] groupIndexs) {
    try {
      if(nullables == null) {
        nullables = new Boolean[] {};
      }
      if(groupIndexs == null) {
        groupIndexs = new String[] {};
      }
      dynamicFormFieldMappingService.rebinding(dynamicFormCode, mappingCode, Lists.newArrayList(fieldCodes) , Lists.newArrayList(nullables));
      return Result.ok();
    } catch(RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }
}
