package com.biz.crm.common.form.local.utils;

import java.lang.reflect.Field;
import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.net.URLClassLoader;
import java.util.Collection;
import java.util.List;
import java.util.Map;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Triple;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.context.ApplicationContext;
import org.springframework.util.CollectionUtils;

import com.biz.crm.common.form.sdk.field.annotation.DynamicField;
import com.biz.crm.common.form.sdk.field.annotation.Validate;
import com.biz.crm.common.form.sdk.model.DynamicForm;
import com.biz.crm.common.form.sdk.model.OperationStrategy;
import com.biz.crm.common.form.sdk.vo.DynamicChildrenFormVo;
import com.biz.crm.common.form.sdk.vo.DynamicFieldVo;
import com.biz.crm.common.form.sdk.vo.DynamicFormVo;
import com.biz.crm.common.form.sdk.widget.WidgetKey;
import com.google.common.collect.Lists;

/**
 * 动态表单构造工具
 * @author yinwenjie
 */
public final class DynamicFormUtils {
  /**
   * 工具类不允许实例化
   */
  private DynamicFormUtils() {
    
  }
  /**
   * 日志
   */
  private static final Logger LOGGER = LoggerFactory.getLogger(DynamicFormUtils.class);
  
  /**
   * 这个分析工具，可以分析传入的诸如VO、DTO等模型，得到这些模型中满足动态表单要求的结构，并返回描述信息；</br>
   * 注意，符合动态表单结构的主模型结构中的属性，一定是一个Map类型的属性。
   * @param formObject 任何的诸如VO、DTO性质的模型，都可以进行分析
   * @return 注意，返回的DynamicFormVo没有parentFieldCode信息和dynamicFormCode信息，需要调用者自行组装
   */
  public static DynamicFormVo analysis(Class<? extends DynamicForm> dynamicFormClass , ApplicationContext applicationContext) {
    if(dynamicFormClass == null) {
      return null;
    }
    List<DynamicChildrenFormVo> childrenForms = Lists.newArrayList();
    List<DynamicFieldVo> dynamicFields = Lists.newArrayList();
    // 开始进行表单及上级父表单结构的构造
    buildSuperForm(dynamicFormClass, dynamicFields, childrenForms, applicationContext);
    DynamicFormVo dynamicFormVo = new DynamicFormVo();
    dynamicFormVo.setDynamicFields(dynamicFields);
    dynamicFormVo.setChildrenForms(childrenForms);
    return dynamicFormVo;
  }
  
  /**
   * 试图递归构造上级表单，直到不再有上级表单结构，在构造本级表单
   */
  private static void buildSuperForm(Class<?> dynamicFormClass , List<DynamicFieldVo> dynamicFields , List<DynamicChildrenFormVo> childrenForms , ApplicationContext applicationContext) {
    Class<?> superClass = dynamicFormClass.getSuperclass();
    if(superClass != null) {
      buildSuperForm(superClass , dynamicFields , childrenForms , applicationContext);
    }
    
    /*
     * 步骤为：
     * 1、分析dynamicForm中，设定了DynamicField注解的字段信息；
     * 注意，如果出现array、collection同为true的情况，则报错
     * 2、接着，如果某个字段是Collection性质的字段，则还要试图验证其子级
     * */
    // 1、=======
    Field[] declaredFields = dynamicFormClass.getDeclaredFields();
    List<DynamicFieldVo> currentDynamicFields = Lists.newArrayList();
    if(declaredFields != null && declaredFields.length > 0) {
      currentDynamicFields = buildDynamicFields(declaredFields, applicationContext);
    }
    dynamicFields.addAll(currentDynamicFields);
    
    // 2、=======
    List<DynamicChildrenFormVo> currentChildrenForms = Lists.newArrayList();
    for (DynamicFieldVo dynamicFieldVo : currentDynamicFields) {
      if(!isChildrenForm(dynamicFieldVo)) {
        continue;
      }
      DynamicChildrenFormVo buildChildrenForm = buildChildrenForm(dynamicFieldVo, applicationContext);
      currentChildrenForms.add(buildChildrenForm);
    }
    childrenForms.addAll(currentChildrenForms);
  }
  
  /**
   * 该私有方法用于关联子级动态表单模型
   */
  private static DynamicChildrenFormVo buildChildrenForm(DynamicFieldVo dynamicFieldVo , ApplicationContext applicationContext) {
    /*
     * 构造子级表单是一个递归的过程，处理过程为：
     * 1、首先为上级字段dynamicFieldVo，生成对应的DynamicChildrenFormVo
     * 2、然后再为这个DynamicChildrenFormVo，组装对应的dynamicFields；
     * 3、递归做可能存在的再下级子模型
     * 4、最后进行返回
     * */
    // 1、======
    DynamicChildrenFormVo dynamicChildrenForm = new DynamicChildrenFormVo();
    dynamicChildrenForm.setParentFieldCode(dynamicFieldVo.getFieldCode());
    
    // 2、=====
    Class<?> dynamicFormClass = dynamicFieldVo.getFieldClass();
    Field[] declaredFields = dynamicFormClass.getDeclaredFields();
    List<DynamicFieldVo> dynamicFields = Lists.newArrayList();
    if(declaredFields != null && declaredFields.length > 0) {
      dynamicFields = buildDynamicFields(declaredFields, applicationContext);
    }
    dynamicChildrenForm.setDynamicFields(dynamicFields);
    
    // 3、=====
    List<DynamicChildrenFormVo> childrenForms = Lists.newArrayList();
    for (DynamicFieldVo dynamicFieldItem : dynamicFields) {
      if(!isChildrenForm(dynamicFieldItem)) {
        continue;
      }
      DynamicChildrenFormVo buildChildrenForm = buildChildrenForm(dynamicFieldItem, applicationContext);
      childrenForms.add(buildChildrenForm);
    }
    
    // 4、=====
    dynamicChildrenForm.setChildrenForms(childrenForms);
    return dynamicChildrenForm;
  }
  
  private static List<DynamicFieldVo> buildDynamicFields(Field[] declaredFields , ApplicationContext applicationContext) {
    List<DynamicFieldVo>  dynamicFields = Lists.newArrayList();
    for (Field declaredField : declaredFields) {
      DynamicField dynamicFieldAnnotation = declaredField.getAnnotation(DynamicField.class);
      if(dynamicFieldAnnotation == null) {
        continue;
      }
      
      // 来自于注解的字段信息
      String fieldName = dynamicFieldAnnotation.fieldName();
      boolean required = dynamicFieldAnnotation.required();
      boolean modifiable = dynamicFieldAnnotation.modifiable();
      Validate[] validates = dynamicFieldAnnotation.validates();
      // 寻找填写的key信息
      Class<? extends WidgetKey> controllKeyClass = dynamicFieldAnnotation.controllKey();
      WidgetKey widgetKey = null;
      if(controllKeyClass != null && applicationContext != null) {
        widgetKey = applicationContext.getBean(controllKeyClass);
      } else if(controllKeyClass != null && applicationContext == null) {
        try {
          widgetKey = controllKeyClass.newInstance();
        } catch (InstantiationException | IllegalAccessException e) {
          LOGGER.error(e.getMessage() , e);
          throw new IllegalArgumentException(e.getMessage());
        }
      }
      // 来自于字段本身定义的信息
      Triple<Class<?>, Boolean, Class<?> > fieldAnalysisResult = analysisFieldType(declaredField);
      Class<?> valueClass = fieldAnalysisResult.getLeft();
      Boolean isValueCollection = fieldAnalysisResult.getMiddle();
      Boolean array = valueClass.isArray();
      String fieldCode = declaredField.getName();
      
      DynamicFieldVo dynamicFieldVo = new DynamicFieldVo();
      dynamicFieldVo.setArray(array);
      dynamicFieldVo.setCollection(isValueCollection);
      dynamicFieldVo.setControllKey(widgetKey);
      dynamicFieldVo.setField(declaredField);
      dynamicFieldVo.setFieldClass(valueClass);
      dynamicFieldVo.setFieldCode(fieldCode);
      dynamicFieldVo.setFieldName(fieldName);
      dynamicFieldVo.setRequired(required);
      dynamicFieldVo.setModifiable(modifiable);
      dynamicFieldVo.setValidates(validates);
      dynamicFields.add(dynamicFieldVo);
    }
    return dynamicFields;
  }
  
  /**
   * 该私有方法用于判定某一个属性，是否是子级表单（动态表单支持无限多级的子级表单）
   * @param dynamicFieldVo 
   * @return
   */
  private static boolean isChildrenForm(DynamicFieldVo dynamicFieldVo) {
    /*
     * 子级模型，一定是在业务系统中开发人员自建的一个模型，其特点是其完整的路径名存在于:
     * a、首先它肯定不是一个原生类，isPrimitive为false
     * b、其次，它肯定不存在于一些关键包名下，如java.* , javax.* , org.springframework.** 等等
     * c、其classloader一定为“URLClassLoader”的子类
     * d、其下的字段，至少有一个字段存在DynamicField注解
     * 
     * 注意：是不是集合都行
     * */
    Class<?> fieldClass = dynamicFieldVo.getFieldClass();
    boolean isPrimitive = fieldClass.isPrimitive();
    String classFullName = fieldClass.getName();
    ClassLoader classLoader = fieldClass.getClassLoader();
    Field[] declaredChildFields = fieldClass.getDeclaredFields();
    boolean hasAnnotation = false;
    for (Field declaredChildFieldItem : declaredChildFields) {
      DynamicField dynamicFieldAnnotation = declaredChildFieldItem.getAnnotation(DynamicField.class);
      if(dynamicFieldAnnotation != null) {
        hasAnnotation = true;
        break;
      }
    }
    
    // a、====
    if(isPrimitive) {
      return false;
    }
    // b、====
    if(StringUtils.indexOfAny(classFullName, "java." , "javax." , "org.springframework.") != -1) {
      return false;
    }
    // c、=====
    if(classLoader == null || !URLClassLoader.class.isAssignableFrom(classLoader.getClass())) {
      return false;
    }
    
    // d、=====
    if(!hasAnnotation) {
      return false;
    }
    // 只有以上验证都通过了，才返回true
    return true;
  }
  
  /**
   * 该方法用于分析特定的Map性质的字段，其Key的（泛型）类型和Value的（泛型）类型
   * @param declaredField
   * @return 返回的Triple对象，左侧存储了Value的类型，中间存储的Value的类型是否是使用集合进行包裹；如果中间的布尔为true，则最右侧存储了集合的类型（注意，可能不是某种具体的类型）
   * @throws 如果当前Field类型不是Map性质的，则抛出异常。
   */
  public static Triple<Class<?> , Boolean , Class<?>> analysisMapFieldType(Field declaredField) {
    Class<?> mapDeclaredFieldType = declaredField.getType();
    String fieldName = declaredField.getName();
    if(!Map.class.isAssignableFrom(mapDeclaredFieldType)) {
      throw new IllegalArgumentException("给定的字段" + fieldName + "，并不是Map性质的字段，请检查!!");
    }

    // 确定泛型类型
    Type genericType = declaredField.getGenericType();
    if(!(genericType instanceof ParameterizedType)) {
      throw new IllegalArgumentException("给定的Map性质的字段" + fieldName + "，存在无法识别的泛型类型，请检查!!");
    }
    ParameterizedType parameterizedType = (ParameterizedType) genericType;
    Type[] actualTypeArguments = parameterizedType.getActualTypeArguments();
    // 首先对key部分的可能性进行处理：actualTypeArguments数组的第0号索引上的类型，就是Map类型中key部分的类型
    // key部分必须是一种字符串
    Type keyType = actualTypeArguments[0];
    if(!(keyType instanceof Class<?>)
        || !CharSequence.class.isAssignableFrom((Class<?>)keyType)) {
      throw new IllegalArgumentException("给定的Map性质的字段" + fieldName + "，其Key部分的类型并不是字符串性质，请检查!!");
    }
    
    // 接着处理value部分的可能性质：
    // Value部分要么是一种collection，要么不是
    Boolean isCollection = false;
    Class<?> valueClass = null;
    Class<?> collectionClass = null;
    try {
      Type valueType = actualTypeArguments[1]; 
      if(valueType instanceof ParameterizedType) { 
        ParameterizedType valueParameterizedType = (ParameterizedType)valueType; 
        valueType = valueParameterizedType.getActualTypeArguments()[0];
        valueClass = (Class<?>)valueType;
        Type rawType = valueParameterizedType.getRawType(); 
        isCollection = Collection.class.isAssignableFrom((Class<?>)rawType);
        if(isCollection) {
          collectionClass = (Class<?>)rawType;
        }
      } else {
        valueClass = (Class<?>)valueType;
      }
    } catch(RuntimeException e) {
      throw new IllegalArgumentException("给定的Map性质的字段" + fieldName + "，其value部分不符合设定要求(可能为双集合/双泛型嵌套)，请检查!!");
    }
    
    // 构造返回结果
    Triple<Class<?> , Boolean , Class<?>> triple = Triple.of(valueClass, isCollection , collectionClass);
    return triple;
  
  }
  
  /**
   * 该私有方法验证一个普通字段的各种关键类信息
   * @param declaredField
   * @return
   */
  private static Triple<Class<?> , Boolean , Class<?>> analysisFieldType(Field declaredField) {
    Type genericType = declaredField.getGenericType();
    String fieldName = declaredField.getName();
    
    // 接着处理value部分的可能性质：
    // Value部分要么是一种collection，要么不是
    Boolean isCollection = false;
    Class<?> valueClass = null;
    Class<?> collectionClass = null;
    // 如果条件成立说明其是一个泛型
    if(genericType instanceof ParameterizedType) {
      ParameterizedType parameterizedType = (ParameterizedType) genericType;
      Type[] actualTypeArguments = parameterizedType.getActualTypeArguments();
      Type realType = actualTypeArguments[0];
      if(!(realType instanceof Class<?>)) {
        throw new IllegalArgumentException("字段属性：" + fieldName + "，其类型并不是字符串性质，请检查!!");
      }
      valueClass = (Class<?>)realType;
      
      // 现在确定是否是集合
      Type rawType = parameterizedType.getRawType(); 
      isCollection = Collection.class.isAssignableFrom((Class<?>)rawType);
      if(isCollection) {
        collectionClass = (Class<?>)rawType;
      }
    } else {
      valueClass = (Class<?>)genericType;
    }
    
    // 进行返回
    Triple<Class<?> , Boolean , Class<?>> triple = Triple.of(valueClass, isCollection , collectionClass);
    return triple;
  }
  
  /**
   * 该工具方法在给定的一组operationStraties集合中，按照dynamicFormCode进行某一个匹配的OperationStrategy的搜索
   * @param dynamicFormCode
   * @param operationStraties
   * @return
   */
  public static OperationStrategy<? extends DynamicForm> findByDynamicFormCode(String dynamicFormCode , List<OperationStrategy<? extends DynamicForm>> operationStraties) {
    if(CollectionUtils.isEmpty(operationStraties)) {
      return null;
    }
    
    for (OperationStrategy<? extends DynamicForm> operationStrategy : operationStraties) {
      if(StringUtils.equals(operationStrategy.dynamicFormCode(), dynamicFormCode)) {
        return operationStrategy;
      }
    }
    return null;
  }
}
