package com.biz.crm.common.ie.sdk.excel.util;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.text.CharSequenceUtil;
import cn.hutool.core.util.ReflectUtil;
import com.biz.crm.common.ie.sdk.excel.annotations.CrmExcelColumn;
import com.biz.crm.common.ie.sdk.excel.annotations.CrmExcelImport;
import com.biz.crm.common.ie.sdk.excel.vo.CrmExcelVo;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import java.lang.reflect.Field;
import java.util.Arrays;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Objects;
import java.util.Optional;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import lombok.Data;
import org.apache.commons.lang3.Validate;

/**
 * excel 处理工具类
 *
 * @author sunx
 * @date 2022/5/14
 */
public class BzExcelUtil {

  private BzExcelUtil() {}

  /**
   * 数据写入对应javabean
   *
   * @param vo 导入导出vo实体
   * @param values 设置的值信息
   */
  public static void setCrmExcelVoValue(CrmExcelVo vo, List<Object> values) {
    if (CollUtil.isEmpty(values)) {
      return;
    }
    CrmExcelImport crmExcelImport = vo.getClass().getAnnotation(CrmExcelImport.class);
    final int fieldNum = vo.getClass().getDeclaredFields().length;
    Validate.isTrue(crmExcelImport.startColumn() <= fieldNum, "开始解析列索引必须小于等于处理bean的字段数");
    int startColumn = crmExcelImport.startColumn();
    int endColumn = crmExcelImport.endColumn();
    if (endColumn == -1) {
      endColumn = fieldNum - 1;
    } else {
      endColumn = Integer.max(startColumn, endColumn);
    }
    List<CrmExcelColumnField> fields = findCrmExcelColumnField(vo.getClass());
    if (CollUtil.isEmpty(fields)) {
      return;
    }

    Map<Integer, Object> map = Maps.newHashMap();
    int size = values.size();
    size = Integer.min(size, fieldNum);
    for (int i = 0; i < size; i++) {
      map.put(i, values.get(i));
    }

    for (int i = 0; i < fields.size(); i++) {
      CrmExcelColumnField item = fields.get(i);
      Object value = map.get(i);
      if (i < startColumn || i > endColumn || value == null) {
        continue;
      }
      Field field = item.getField();
      try {
        ReflectUtil.setFieldValue(vo, field, value);
      } catch (Exception e) {
        Validate.isTrue(false, CharSequenceUtil.format("对象字段{}赋值错误", field.getName()));
      }
    }
  }

  /**
   * 数据写入对应javabean
   *
   * @param vo 导入导出vo实体
   * @param data vo对应的key的值信息
   */
  public static void setCrmExcelVoValue(CrmExcelVo vo, Map<Integer, Object> data) {
    if (data.isEmpty()) {
      return;
    }
    List<Object> values = Lists.newLinkedList();
    for (Entry<Integer, Object> item : data.entrySet()) {
      values.add(item.getValue());
    }
    setCrmExcelVoValue(vo, values);
  }

  /**
   * 根据导出实体获取导出excel title 信息
   *
   * @param crmExcelVoClass 导入导出vo实体class
   * @return
   */
  public static List<List<String>> findCrmExcelExportTitle(Class<?> crmExcelVoClass) {
    List<List<String>> list = Lists.newLinkedList();
    final List<CrmExcelColumnField> fields = findCrmExcelColumnField(crmExcelVoClass);
    if (CollUtil.isEmpty(fields)) {
      return list;
    }
    for (CrmExcelColumnField field : fields) {
      list.add(Arrays.asList(field.getTitle()));
    }
    return list;
  }

  /**
   * 获取excel对应的title集合
   *
   * @param crmExcelVoClass 导入导出vo实体class
   * @return
   */
  public static List<CrmExcelColumnField> findCrmExcelColumnField(Class<?> crmExcelVoClass) {
    final Field[] fields = ReflectUtil.getFields(crmExcelVoClass);
    List<CrmExcelColumnField> list = Lists.newArrayList();
    Stream.of(fields)
        .filter(a -> Objects.nonNull(a.getAnnotation(CrmExcelColumn.class)))
        .forEach(
            a -> {
              CrmExcelColumnField cur = new CrmExcelColumnField();
              cur.setField(a);
              cur.setColumn(a.getAnnotation(CrmExcelColumn.class));
              cur.setOrder(cur.getColumn().order());
              final String[] title = cur.getColumn().value();
              cur.setTitle(title);
              if (cur.getTitle() == null || cur.getTitle().length == 0) {
                cur.setTitle(new String[] {cur.getField().getName()});
              }
              list.add(cur);
            });
    return list.stream()
        .sorted(Comparator.comparing(CrmExcelColumnField::getOrder))
        .collect(Collectors.toList());
  }

  /**
   * 根据任务total获取消息发送阈值
   *
   * @param total 消息记录数量
   * @return
   */
  public static Integer findMsgCount(Integer total) {
    total = Integer.max(Optional.ofNullable(total).orElse(0), 0);
    if (total < 10) {
      return 10;
    } else if (total < 500) {
      return 25;
    } else if (total < 2000) {
      return 100;
    } else if (total < 10000) {
      return 500;
    } else {
      return 1000;
    }
  }

  @Data
  public static class CrmExcelColumnField {
    /** title */
    private String[] title;
    /** 字段 */
    private Field field;
    /** 排序，升序 */
    private Integer order;
    /** 注解信息 */
    private CrmExcelColumn column;
  }
}
