package com.biz.crm.common.ie.sdk.excel.process;

import com.alibaba.fastjson.JSONArray;
import com.biz.crm.common.ie.sdk.constant.ImportExportConstant;
import com.biz.crm.common.ie.sdk.excel.vo.CrmExcelVo;
import com.biz.crm.common.ie.sdk.vo.EsParagraphFieldRangeVo;
import com.biz.crm.common.ie.sdk.vo.ExportTaskProcessVo;
import com.bizunited.nebula.common.register.ElasticsearchQueryRegister;
import com.bizunited.nebula.common.service.es.ElasticsearchQueryService;
import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import org.apache.commons.lang3.Validate;
import org.apache.lucene.search.TotalHits;
import org.elasticsearch.action.search.SearchRequest;
import org.elasticsearch.action.search.SearchResponse;
import org.elasticsearch.index.query.BoolQueryBuilder;
import org.elasticsearch.index.query.QueryBuilder;
import org.elasticsearch.index.query.QueryBuilders;
import org.elasticsearch.search.builder.SearchSourceBuilder;
import org.elasticsearch.search.sort.SortOrder;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.util.CollectionUtils;

/**
 * @author ning.zhang
 * @description ES导出抽象基座,用户如果使用ES大数据分段导出需要继承此抽象类
 * @date 2024/07/23
 */
public abstract class AbstractEsParagraphExportProcess<T extends CrmExcelVo> implements ExportProcess<T> {

  @Autowired(required = false)
  private ElasticsearchQueryService elasticsearchQueryService;

  /**
   * 本次导出的ES查询注册器实现类
   *
   * @return ES查询注册器实现类
   */
  public abstract Class<? extends ElasticsearchQueryRegister> getElasticsearchQueryRegister();

  @Override
  public Integer getTotal(Map<String, Object> params) {
    ElasticsearchQueryRegister queryRegister = elasticsearchQueryService.findRegisterByClass(getElasticsearchQueryRegister());
    Validate.notNull(queryRegister, "未找到对应的ES查询注册器实现");
    SearchSourceBuilder sourceBuilder = queryRegister.buildSearch(params);
    sourceBuilder.from(0);
    sourceBuilder.size(1);
    sourceBuilder.trackTotalHits(true);
    SearchRequest searchRequest = new SearchRequest(queryRegister.getIndexName()).source(sourceBuilder);
    SearchResponse searchResponse = elasticsearchQueryService.queryWithRequest(searchRequest);
    if (Objects.isNull(searchResponse) || Objects.isNull(searchResponse.getHits()) || Objects.isNull(searchResponse.getHits().getTotalHits())) {
      return 0;
    }
    TotalHits totalHits = searchResponse.getHits().getTotalHits();
    return (int) totalHits.value;
  }

  @Override
  public JSONArray getData(ExportTaskProcessVo vo, Map<String, Object> params) {
    List<EsParagraphFieldRangeVo> paragraphFieldRanges = vo.getQueryParagraphFieldRanges();
    Validate.isTrue(!CollectionUtils.isEmpty(paragraphFieldRanges), "ES分段数据异常");
    ElasticsearchQueryRegister queryRegister = elasticsearchQueryService.findRegisterByClass(getElasticsearchQueryRegister());
    Validate.notNull(queryRegister, "未找到对应的ES查询注册器实现");
    SearchSourceBuilder sourceBuilder = queryRegister.buildSearch(params);
    sourceBuilder.sorts().clear();
    BoolQueryBuilder builder = QueryBuilders.boolQuery();
    paragraphFieldRanges.forEach(fieldRangeVo -> {
      builder.filter(QueryBuilders.rangeQuery(fieldRangeVo.getFieldName()).lte(fieldRangeVo.getEnd()).gte(fieldRangeVo.getStart()));
      sourceBuilder.sort(fieldRangeVo.getFieldName(), SortOrder.fromString(fieldRangeVo.getSort()));
    });
    if (Objects.nonNull(sourceBuilder.query())) {
      builder.must(sourceBuilder.query());
    }
    sourceBuilder.query(builder);
    sourceBuilder.from(0);
    sourceBuilder.size(this.getQueryPageSize());
    SearchRequest searchRequest = new SearchRequest(queryRegister.getIndexName()).source(sourceBuilder);
    SearchResponse searchResponse = elasticsearchQueryService.queryWithRequest(searchRequest);
    List<Map> mapList = elasticsearchQueryService.convertResponse(searchResponse, Map.class);
    return toJSONArray(mapList);
  }

  /**
   * 设置ES默认的最大查询大小10000
   *
   * @return 查询大小
   */
  @Override
  public Integer getQueryPageSize() {
    return ImportExportConstant.IE_ES_EXPORT_QUERY_PAGE_SIZE;
  }

  /**
   * 获取对象转换实体
   *
   * @return
   */
  @Override
  public Class<T> findCrmExcelVoClass() {
    return (Class<T>) ((ParameterizedType) getClass().getGenericSuperclass()).getActualTypeArguments()[0];
  }
}
