package com.biz.crm.common.pay.support.cpcn.base.cpcn.tx;

import com.biz.crm.common.pay.support.cpcn.base.common.security.Signer;
import com.biz.crm.common.pay.support.cpcn.base.common.security.Verifier;
import com.biz.crm.common.pay.support.cpcn.base.common.utils.PayUtils;
import com.biz.crm.common.pay.support.cpcn.base.common.utils.XmlUtils;
import com.biz.crm.common.pay.support.cpcn.base.cpcn.config.CpcnConfig;
import lombok.Getter;
import lombok.Setter;
import lombok.ToString;
import lombok.extern.slf4j.Slf4j;
import org.springframework.util.Base64Utils;
import org.springframework.util.CollectionUtils;
import org.w3c.dom.Document;

import javax.validation.ConstraintViolation;
import javax.validation.Validator;
import javax.validation.constraints.NotBlank;
import java.nio.charset.StandardCharsets;
import java.util.Set;

/**
 * @author Keller
 */
@Getter
@Setter
@ToString
@Slf4j
public abstract class TxBaseRequest {
  /**
   * 交易编码
   * 如：4600(影印件采集）
   */
  @NotBlank(message = "交易编码不能诶空")
  protected String txCode;
  /**
   * 请求明文
   */
  protected String requestPlainText;
  /**
   * 请求消息
   */
  protected String requestMessage;
  /**
   * 请求签名
   */
  protected String requestSignature;
  /**
   * 签名算法
   */
  protected String signAlgorithm;
  /**
   * 签名序列号
   */
  protected String signSN;
  /**
   * 加密序列号
   */
  protected String encryptSN;
  /**
   *
   */
  protected String dgtlEnvlp;
  /**
   * 机构配置信息
   */
  protected CpcnConfig cpcnConfig;
  /**
   * 属性验证方法
   */
  protected Validator validator;

  /**
   * 执行请求接口
   */
  public abstract void process() throws Exception;

  /**
   * 执行属性校验
   */
  public void validate() {
    Set<ConstraintViolation<TxBaseRequest>> violationSet = validator.validate(this);
    if(!CollectionUtils.isEmpty(violationSet)){
      throw new IllegalArgumentException(violationSet.iterator().next().getMessage());
    }
  }

  /**
   * 生成请求信息与签名
   *
   * @param document
   */
  protected void postProcess(Document document) throws Exception {
    this.requestPlainText = XmlUtils.createPrettyFormat(document).trim();
    byte[] data = this.requestPlainText.getBytes(StandardCharsets.UTF_8);
    this.requestMessage = new String(Base64Utils.encode(data));
    Signer signer = this.cpcnConfig.getSigner();
    byte[] signature = signer.sign(data);
    this.requestSignature = PayUtils.bytes2hex(signature);

    if (this.cpcnConfig.getIsDgEnv()) {
      // 组装数字信封
      this.signAlgorithm = this.cpcnConfig.getAlgorithm();
      String symmetricCipher = signer.getSymmetricCipher();
      this.signSN = signer.getSN();

      Verifier verifier = this.cpcnConfig.getVerifier();
      this.encryptSN = verifier.getSN();
      this.dgtlEnvlp = "02|" + symmetricCipher;
      try {
        this.requestMessage = verifier.symmetricEncrypt(this.requestPlainText, symmetricCipher);
        this.dgtlEnvlp = verifier.asymmetricEncrypt(this.dgtlEnvlp.getBytes(StandardCharsets.UTF_8));
      } catch (Exception e) {
        log.error("组装数字信封失败", e);
      }
      log.info("数字信封组装成功,非对称加密公钥序列号：" + this.encryptSN);
    }
  }
}
