package com.biz.crm.common.pay.support.cpcn.base.common.security;

import cfca.sadk.util.CertUtil;
import cfca.sadk.x509.certificate.X509Cert;
import com.biz.crm.common.pay.support.cpcn.base.common.utils.DgEnvUtils;
import com.biz.crm.common.pay.support.cpcn.base.common.utils.PayUtils;

import java.nio.charset.StandardCharsets;
import java.security.PrivateKey;
import java.security.Signature;

/**
 * Rsa加密方法实现
 *
 * @author Keller
 */
public class PfxSigner implements Signer {
  /**
   * 加密密钥
   */
  private PrivateKey privateKey;
  /**
   * 签名算法（默认）
   */
  private String algorithm = "SHA1withRSA";
  /**
   * 公钥证书
   */
  private X509Cert x509Cert;

  public PfxSigner(String pfxFile, String password) throws Exception {
    this.privateKey = PayUtils.getPrivateKeyFromPFX(pfxFile, password);
  }

  public PfxSigner(String pfxFile, String password, String algorithm) throws Exception {
    this.x509Cert = CertUtil.getCertFromPFX(pfxFile, password);
    this.privateKey = PayUtils.getPrivateKeyFromPFX(pfxFile, password);
    this.algorithm = algorithm;
  }

  @Override
  public String sign(String message) throws Exception {
    Signature signature = Signature.getInstance(this.algorithm);
    signature.initSign(this.privateKey);
    signature.update(message.getBytes(StandardCharsets.UTF_8));
    return PayUtils.bytes2hex(signature.sign());
  }

  @Override
  public byte[] sign(byte[] data, String algorithm) throws Exception {
    Signature signature = Signature.getInstance(algorithm);
    signature.initSign(this.privateKey);
    signature.update(data);
    return signature.sign();
  }

  @Override
  public byte[] sign(byte[] data) throws Exception {
    Signature signature = Signature.getInstance(this.algorithm);
    signature.initSign(this.privateKey);
    signature.update(data);
    return signature.sign();
  }

  @Override
  public String decrypt(String message, String digitalEnvelope) throws Exception {
    String key = PayUtils.decryptRSA(digitalEnvelope, this.privateKey);
    String[] strs = key.split("\\|");
    return DgEnvUtils.generatePlainText(strs, message);
  }

  @Override
  public String decrypt(String message) throws Exception {
    return PayUtils.decryptRSA(message, this.privateKey);
  }

  @Override
  public String getSN() {
    return String.valueOf(this.x509Cert.getStringSerialNumber());
  }

  @Override
  public String getSymmetricCipher() {
    return PayUtils.randomHexString(16);
  }

  @Override
  public String getAlgorithm() {
    return this.algorithm;
  }

  public void setAlgorithm(String algorithm) {
    this.algorithm = algorithm;
  }
}
