package com.biz.crm.common.weixinsign.local.controller;

import com.alibaba.fastjson.JSONObject;
import com.biz.crm.business.common.sdk.model.Result;
import com.biz.crm.common.weixinsign.local.config.WXAccountProperties;
import com.biz.crm.common.weixinsign.sdk.common.enums.WXMsgTypeEnum;
import com.biz.crm.common.weixinsign.sdk.common.utils.AesException;
import com.biz.crm.common.weixinsign.sdk.common.utils.XMLConverUtil;
import com.biz.crm.common.weixinsign.sdk.common.utils.msg.EventReceiveXML;
import com.biz.crm.common.weixinsign.sdk.common.utils.msg.TextMsgReplyXML;
import com.biz.crm.common.weixinsign.sdk.dto.ComponentVerifyTicketDto;
import com.biz.crm.common.weixinsign.sdk.dto.MsgEventDto;
import com.biz.crm.common.weixinsign.sdk.service.WXOpenPlatformVoService;
import com.biz.crm.common.weixinsign.sdk.vo.WXJsConfigVo;
import com.biz.crm.common.weixinsign.sdk.vo.WxWebViewAccessTokenRespVo;
import com.fasterxml.jackson.core.JsonEncoding;
import com.fasterxml.jackson.core.JsonGenerator;
import com.fasterxml.jackson.databind.ObjectMapper;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.bind.annotation.RestController;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.BufferedReader;
import java.io.IOException;
import java.io.PrintWriter;
import java.io.UnsupportedEncodingException;
import java.net.URLDecoder;
import java.util.List;
import java.util.Objects;

/**
 * 微信开放平台
 *
 * @author 作者 hc
 */
@Api(value = "微信开放平台 controller")
@RestController
@RequestMapping("/v1/wx/open")
@Slf4j
public class WXChatOpenPlatformController extends WxBaseController {

  @Autowired
  private WXOpenPlatformVoService wxOpenPlatformService;
  @Autowired
  private WXAccountProperties wxAccountProperties;

  /**
   * 获取全局jssdk配置
   *
   * @param url
   * @param appid
   * @return
   */
  @ApiOperation(value = "获取全局jssdk配置")
  @GetMapping("/{appid}/findWXJsConfig")
  public Result<WXJsConfigVo> findWXJsConfig(@ApiParam("签名url") String url, @PathVariable(value = "appid") String appid, @RequestParam(required = false) Boolean isUrlEncoder) {
    try {
      if (Objects.nonNull(isUrlEncoder) && isUrlEncoder && StringUtils.isNotBlank(url)) {
        try {
          url = URLDecoder.decode(url, "utf-8");
        } catch (UnsupportedEncodingException e) {
          throw new IllegalArgumentException("url不能为空！");
        }
      }
      WXJsConfigVo value = this.wxOpenPlatformService.findWXJsConfig(url, appid);
      return Result.ok(value);
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }


  @ApiOperation(value = "获取微信网页授权accessToken")
  @GetMapping("/{appid}/findWebViewAccessToken")
  public Result<WxWebViewAccessTokenRespVo> findWebViewAccessToken(@RequestParam(required = true) String code,
          @PathVariable(value = "appid") String appid) {
    try {
      WxWebViewAccessTokenRespVo accessTokenRespVo = wxOpenPlatformService.findWebViewAccessToken(code, appid);
      return Result.ok(accessTokenRespVo);
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  /**
   * <pre>
   *  授权事件接收URL的处理
   *   1、微信服务器发送给服务自身的事件推送（如取消授权通知，Ticket推送等）
   *   此时，消息XML体中没有ToUserName字段，而是AppId字段，即公众号服务的AppId。
   *   这种系统事件推送通知（现在包括推送component_verify_ticket协议和推送取消
   *   授权通知），服务开发者收到后也需进行解密，接收到后只需直接返回字符串“success”。
   * 授权事件接受url 每隔10分钟 获取微信服务器推送ticket 接收后需要解密 接收到后 必须返回字符串success
   * </pre>
   *
   * @param request
   * @param response
   * @throws Exception
   */
  @RequestMapping("/sysMsg")
  public void sysMsg(@RequestParam("signature") String signature, @RequestParam("timestamp") String timestamp, @RequestParam("nonce") String nonce, @RequestParam("msg_signature") String msgSignature,
          HttpServletRequest request, HttpServletResponse response) throws Exception {
    StringBuilder sb = new StringBuilder();
    BufferedReader in = request.getReader();
    String line;
    while ((line = in.readLine()) != null) {
      sb.append(line);
    }
    String xml = sb.toString();
    ComponentVerifyTicketDto dto = new ComponentVerifyTicketDto();
    dto.setMsgSignature(msgSignature);
    dto.setNonce(nonce);
    dto.setTimestamp(timestamp);
    dto.setSignature(signature);
    dto.setXmlEncrypt(xml);
    this.wxOpenPlatformService.componentVerifyTicket(dto);
    PrintWriter pw = response.getWriter();
    pw.write("success");
    pw.flush();
  }

  /**
   * 消息与事件接收URL(第三方平台处理微信发送的消息)
   *
   * @param request
   * @param response
   * @param appId
   * @throws IOException
   * @throws AesException
   */
  @ResponseBody
  @RequestMapping(value = "{appid}/event", method = RequestMethod.POST)
  public void event(HttpServletRequest request, HttpServletResponse response,
          @PathVariable(value = "appid") String appId)// springMVC 获取地址里面的参数信息
          throws IOException {
    String nonce = request.getParameter("nonce");
    String timestamp = request.getParameter("timestamp");
    String msgSignature = request.getParameter("msg_signature");
    String signature = request.getParameter("signature");
    StringBuilder sb = new StringBuilder();
    BufferedReader in = request.getReader();
    String line;
    while ((line = in.readLine()) != null) {
      sb.append(line);
    }
    in.close();
    String xml = sb.toString();
    MsgEventDto dto = new MsgEventDto();
    dto.setMsgSignature(msgSignature);
    dto.setNonce(nonce);
    dto.setTimestamp(timestamp);
    dto.setSignature(signature);
    dto.setXmlEncrypt(xml);
    EventReceiveXML eventMsg = this.wxOpenPlatformService.analyzeMsgEvent(dto);
    if (Objects.isNull(eventMsg)) {
      return;
    }
    String msgType = eventMsg.getMsgType();
    String toUserName = eventMsg.getToUserName();
    String fromUserName = eventMsg.getFromUserName();
    String event = eventMsg.getEvent();
    String content = eventMsg.getContent();
    // 微信自动化测试的专用测试公众账号
    List<String> testAppid = wxAccountProperties.getOpenPlatformAccount().getTestAppidUsername();
    if (testAppid.contains(appId)) {
      this.testAppidEvent(request, response, eventMsg, appId);
    } else {
      //TODO 这里处理非微信自动化测试消息
    }
  }

  /**
   * 第三方平台发布消息验证
   */
  private void testAppidEvent(HttpServletRequest request, HttpServletResponse response, EventReceiveXML eventMsg, String appid) {
    log.info("微信开放平台自动化测试验证消息开始。。。。");
    String msgType = eventMsg.getMsgType();
    String toUserName = eventMsg.getToUserName();
    String fromUserName = eventMsg.getFromUserName();
    String event = eventMsg.getEvent();
    String content = eventMsg.getContent();
    // 返回类型值，做一下区分
    if (Objects.equals(WXMsgTypeEnum.EVENT.getCode(), msgType)) {
      // 返回时， 将发送人和接收人 调换一下即可
      replyEventMessage(request, response, event, fromUserName, toUserName);
    }
    if (Objects.equals(WXMsgTypeEnum.TEXT.getCode(), msgType)) {
      // 标示文本消息，
      // 返回时， 将发送人和接收人 调换一下即可
      // 用文本消息去拼接字符串。微信规定
      processTextMessage(request, response, content, fromUserName, toUserName, appid);
    }
  }

  /**
   * 方法描述: 类型为event的时候，拼接
   *
   * @param request
   * @param response
   * @param event
   * @param toUserName   发送接收人
   * @param fromUserName 发送人
   */
  private void replyEventMessage(HttpServletRequest request, HttpServletResponse response,
          String event, String toUserName, String fromUserName) {
    String content = event + "from_callback";
    replyTextMessage(request, response, content, toUserName, fromUserName);
  }


  /**
   * 微信模推送给第三方平台方：文本消息，其中Content字段的内容固定为：TESTCOMPONENT_MSG_TYPE_TEXT， 第三方平台方立马回应文本消息并最终触达粉丝：Content必须固定为：TESTCOMPONENT_MSG_TYPE_TEXT_callback
   *
   * @param content      文本
   * @param toUserName   发送接收人
   * @param fromUserName 发送人
   */
  private void processTextMessage(HttpServletRequest request, HttpServletResponse response,
          String content, String toUserName, String fromUserName, String appid) {
    if ("TESTCOMPONENT_MSG_TYPE_TEXT".equals(content)) {
      String returnContent = content + "_callback";
      replyTextMessage(request, response, returnContent, toUserName, fromUserName);
    } else if (StringUtils.startsWithIgnoreCase(content, "QUERY_AUTH_CODE")) {
      // 需在5秒内返回空串表明暂时不回复，然后再立即使用客服消息接口发送消息回复粉丝
      try {
        response.getWriter().print("");
      } catch (IOException e) {
        e.printStackTrace();
      }
      log.info("content:" + content + " content[1]:" + content.split(":")[1] + " fromUserName:"
              + fromUserName + " toUserName:" + toUserName);
      // 接下来客服API再回复一次消息
      // 此时 content字符的内容为是 QUERY_AUTH_CODE:adsg5qe4q35
      replyApiTextMessage(content.split(":")[1], toUserName, appid);
    }
  }


  /**
   * 方法描述: 直接返回给微信开放平台
   *
   * @param request
   * @param response
   * @param content      文本
   * @param toUserName   发送接收人
   * @param fromUserName 发送人
   */
  private void replyTextMessage(HttpServletRequest request, HttpServletResponse response,
          String content, String toUserName, String fromUserName) {
    Long createTime = System.currentTimeMillis() / 1000;
    TextMsgReplyXML xmlObj = new TextMsgReplyXML(toUserName, fromUserName, createTime, WXMsgTypeEnum.TEXT.getCode(), content);
    String replyMsg = XMLConverUtil.convertToXML(xmlObj);
    // 千万别加密
    log.info("确定发送的XML为：" + replyMsg);
    returnJSON(replyMsg, response);
  }

  /**
   * 方法描述: 调用客服回复消息给粉丝
   *
   * @param auth_code
   * @param fromUserName
   * @return void
   * @throws DocumentException
   * @throws IOException
   */
  private void replyApiTextMessage(String auth_code, String fromUserName, String appid) {
    // 得到微信授权成功的消息后，应该立刻进行处理！！相关信息只会在首次授权的时候推送过来
    // 得到微信授权成功的消息后，应该立刻进行处理！！相关信息只会在首次授权的时候推送过来
    String authorizerAccessToken = wxOpenPlatformService.getApiQueryAuth(auth_code);
    JSONObject message = new JSONObject();
    message.put("touser", fromUserName);
    message.put("msgtype", "text");
    JSONObject text = new JSONObject();
    text.put("content", auth_code + "_from_api");
    message.put("text", text);
    String result = this.wxOpenPlatformService.customSendMessage(message, authorizerAccessToken);
    log.info("客服发送接口返回值:" + result);
  }


  /**
   * 方法描述: 返回数据到请求方
   *
   * @param data     数据
   * @param response
   */
  private void returnJSON(Object data, HttpServletResponse response) {
    try {
      ObjectMapper objectMapper = new ObjectMapper();
      response.setContentType("application/json");
      JsonGenerator generator =
              objectMapper.getJsonFactory().createJsonGenerator(response.getOutputStream(), JsonEncoding.UTF8);
      objectMapper.writeValue(generator, data);
    } catch (IOException e) {
      e.printStackTrace();
    }
  }
}
