package com.biz.crm.cpcnpay.common.utils;

import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.bouncycastle.util.encoders.Base64;

import javax.crypto.Cipher;
import javax.crypto.spec.IvParameterSpec;
import javax.crypto.spec.SecretKeySpec;
import javax.net.ssl.KeyManager;
import javax.net.ssl.KeyManagerFactory;
import javax.net.ssl.TrustManager;
import javax.net.ssl.TrustManagerFactory;
import java.io.ByteArrayInputStream;
import java.io.FileInputStream;
import java.io.UnsupportedEncodingException;
import java.nio.charset.StandardCharsets;
import java.security.*;
import java.security.cert.CertificateException;
import java.security.cert.CertificateFactory;
import java.security.cert.X509Certificate;
import java.security.spec.PKCS8EncodedKeySpec;
import java.util.Enumeration;
import java.util.Random;
import java.util.concurrent.ThreadLocalRandom;

/**
 * 支付工具类
 * 包含了在支付中的加解密方法 对RSA密钥的生成以及验证的方法等
 *
 * @author Keller
 */
@Slf4j

public final class PayUtils {
  /**
   * MD5数据摘要算法，输出128位
   */
  public static final String DIGEST_ALGORITHM_MD5 = "MD5";
  /**
   * SHA-1数据摘要算法，输出160位
   */
  public static final String DIGEST_ALGORITHM_SHA1 = "SHA-1";
  /**
   * RSA 数据加密算法 非对称RSA
   */
  public static final String DATA_SECURITY_RSA = "RSA";
  /**
   * 签名算法 MD5withRSA
   */
  public static final String SIGNATURE_ALGORITHM_MD5_WITH_RSA = "MD5withRSA";
  /**
   * 签名算法 SHA1withRSA
   */
  public static final String SIGNATURE_ALGORITHM_SHA1_WITH_RSA = "SHA1withRSA";
  /**
   * 签名算法 SHA256withRSA
   */
  public static final String SIGNATURE_ALGORITHM_SHA256_WITH_RSA = "SHA256withRSA";
  /**
   * 证书存储格式 PKCS12
   */
  public static final String KEY_STORE_TYPE_PKCS12 = "PKCS12";
  /**
   * 公钥证书格式 X.509
   */
  public static final String KEY_TYPE_X509 = "X.509";
  /**
   * 证书内容头部信息
   */
  public static final String BEGIN_CERTIFICATE = "-----BEGIN CERTIFICATE-----\n";
  /**
   * 证书内容尾部信息
   */
  public static final String END_CERTIFICATE = "\n-----END CERTIFICATE-----\n";

  static byte[] KEY_DATA = new byte[] {
          -28, -72, -83, -23, -121, -111, -26, -108, -81, -28,
          -69, -104, -26, -100, -119, -23, -103, -112, -27, -123,
          -84, -27, -113, -72 };

  /**
   * 偏移量 必须为8位
   */
  static byte[] IV_DATA = new byte[] { 67, 112, 99, 110, 49, 64, 51, 52 };

  /**
   * 可以使用的签名算法
   */
  public static String[] SIGNATURE_ALGORITHMS_IN_USE = new String[] { SIGNATURE_ALGORITHM_SHA1_WITH_RSA, SIGNATURE_ALGORITHM_SHA256_WITH_RSA };

  private PayUtils() {
    throw new UnsupportedOperationException("该工具类不支持实例化");
  }

    /**
     * 通过私钥密钥文件获取私钥对象
     * 注意：私钥密钥文件采用PKCS12格式进行存储，加载时候使用PKCS12格式
     *
     * @param pfxfilename
     * @param password
     * @return
     * @throws Exception
     */
  public static PrivateKey getPrivateKeyFromPFX(String pfxfilename, String password) throws Exception {
    KeyStore keyStore = KeyStore.getInstance(KEY_STORE_TYPE_PKCS12);
    FileInputStream fis = new FileInputStream(pfxfilename);
    keyStore.load(fis, password.toCharArray());
    fis.close();
    Enumeration<String> aliases = keyStore.aliases();
    String alias = aliases.nextElement();
    return (PrivateKey)keyStore.getKey(alias, password.toCharArray());
  }

  /**
   * 通过私钥密钥文件获取私钥对象
   *
   * 注意：私钥密钥文件采用PKCS12格式进行存储，加载时候使用PKCS12格式，需要提供对应的密钥文件的密码
   * @param pfxFileName
   * @param pfxPassword
   * @return
   * @throws Exception
   */
  public static PrivateKey getPrivateKeyFromPFXForRSA(String pfxFileName, String pfxPassword) throws Exception {
    FileInputStream file_inputstream = new FileInputStream(pfxFileName);
    KeyStore keyStore = KeyStore.getInstance(KEY_STORE_TYPE_PKCS12);
    keyStore.load(file_inputstream, pfxPassword.toCharArray());
    String alias = null;
    Enumeration<String> aliases = keyStore.aliases();
    if (aliases.hasMoreElements()) {
      alias = ((String) aliases.nextElement()).toString();
    }
    Key key = keyStore.getKey(alias, pfxPassword.toCharArray());
    PKCS8EncodedKeySpec keySpec = new PKCS8EncodedKeySpec(key.getEncoded());
    KeyFactory keyFactory = KeyFactory.getInstance(DATA_SECURITY_RSA);
    return keyFactory.generatePrivate(keySpec);
  }

  /**
   * 获取消息内容的摘要
   *
   * @param message   需要生成摘要的信息内容
   * @param algorithm 算法
   * @return
   * @throws Exception
   */
  public static byte[] digest(String message, String algorithm) throws Exception {
    MessageDigest md = MessageDigest.getInstance(algorithm);
    md.update(message.getBytes(StandardCharsets.UTF_8));
    return md.digest();
  }

  /**
   * 获取消息内容的摘要（默认使用MD5算法）
   *
   * @param message
   * @return
   * @throws Exception
   */
  public static byte[] digest(String message) throws Exception {
    return digest(message, DIGEST_ALGORITHM_MD5);
  }

  /**
   * 获取公钥证书
   *
   * @param base64string
   * @return
   * @throws CertificateException
   * @throws UnsupportedEncodingException
   */
  public static X509Certificate generateCertificate(String base64string) throws CertificateException, UnsupportedEncodingException {
    if (!base64string.startsWith(BEGIN_CERTIFICATE)) {
      base64string = StringUtils.join(BEGIN_CERTIFICATE ,base64string , END_CERTIFICATE);
    }
    ByteArrayInputStream bais = new ByteArrayInputStream(base64string.getBytes(StandardCharsets.UTF_8));
    CertificateFactory cf = CertificateFactory.getInstance(KEY_TYPE_X509);
    return (X509Certificate)cf.generateCertificate(bais);
  }

  /**
   * 获取公钥证书
   *
   * @param base64string
   * @param charset
   * @return
   * @throws CertificateException
   * @throws UnsupportedEncodingException
   */
  public static X509Certificate generateCertificate(String base64string, String charset) throws CertificateException, UnsupportedEncodingException {
    if (!base64string.startsWith(BEGIN_CERTIFICATE)) {
      base64string = StringUtils.join(BEGIN_CERTIFICATE ,base64string , END_CERTIFICATE);
    }
    ByteArrayInputStream bais = new ByteArrayInputStream(base64string.getBytes(charset));
    CertificateFactory cf = CertificateFactory.getInstance(KEY_TYPE_X509);
    return (X509Certificate)cf.generateCertificate(bais);
  }

  /**
   * 获取公钥证书
   *
   * @param fis
   * @return
   * @throws CertificateException
   */
  public static X509Certificate generateCertificate(FileInputStream fis) throws CertificateException {
    CertificateFactory cf = CertificateFactory.getInstance(KEY_TYPE_X509);
    return (X509Certificate)cf.generateCertificate(fis);
  }

  public static KeyManager[] getKeyManagers(String keyStore, String keyStoreType, String keyStorePassword) throws Exception {
    String algorithm = KeyManagerFactory.getDefaultAlgorithm();
    KeyManagerFactory kmf = KeyManagerFactory.getInstance(algorithm);
    KeyStore ks = KeyStore.getInstance(keyStoreType);
    FileInputStream fis = new FileInputStream(keyStore);
    ks.load(fis, keyStorePassword.toCharArray());
    fis.close();
    kmf.init(ks, keyStorePassword.toCharArray());
    KeyManager[] kms = kmf.getKeyManagers();
    return kms;
  }

  public static TrustManager[] getTrustManagers(String trustStore, String trustStoreType, String trustStorePassword) throws Exception {
    String algorithm = TrustManagerFactory.getDefaultAlgorithm();
    TrustManagerFactory trustManagerFactory = TrustManagerFactory.getInstance(algorithm);
    KeyStore keyStore = KeyStore.getInstance(trustStoreType);
    FileInputStream fileInputStream = new FileInputStream(trustStore);
    keyStore.load(fileInputStream, trustStorePassword.toCharArray());
    fileInputStream.close();
    trustManagerFactory.init(keyStore);
    TrustManager[] tms = trustManagerFactory.getTrustManagers();
    return tms;
  }

  /**
   * 通过私钥解密rsa加密内容
   *
   * @param encryptedDataBase64
   * @param privateKey
   * @return
   */
  public static byte[] decryptRSAToByte(String encryptedDataBase64, PrivateKey privateKey)  {
    Validate.notBlank(encryptedDataBase64,"解密文本内容为空，请检查!");
    Validate.notNull(privateKey,"私钥为空，请检查！");
    byte[] plainBinary = null;
    try {
      // RSA算法，ECS模式，PKCS1格式填充
      Cipher rsaCipher = Cipher.getInstance("RSA/ECB/PKCS1Padding");
      rsaCipher.init(2, privateKey);
      byte[] encodedBytes = Base64.decode(encryptedDataBase64);
      plainBinary = rsaCipher.doFinal(encodedBytes);
    } catch (Exception e) {
      log.error("解密错误，请检查解密内容或者密钥是否正确!",e);
      throw new RuntimeException("解密错误，请检查解密内容或者密钥是否正确!",e);
    }
    return plainBinary;
  }

  /**
   * 字符串转换加密算法名（忽略大小写）
   *
   * @param algorithm
   * @return
   */
  public static String convertSignatureAlgorithm(String algorithm) {
    String result = null;
    algorithm = StringUtils.trim(algorithm);
    for (String algorithmInUse : SIGNATURE_ALGORITHMS_IN_USE) {
      if (algorithmInUse.equalsIgnoreCase(algorithm)) {
        result = algorithmInUse;
        break;
      }
    }
    return result;
  }

  /**
   * 三重DES解密
   */
  public static byte[] DES3_CBC_Decrypt(byte[] ivData, byte[] keyData, byte[] cipherText) {
    SecretKeySpec key3Des = new SecretKeySpec(keyData, "DESede");
    IvParameterSpec IvSpec = new IvParameterSpec(ivData);
    byte[] plainText = null;
    try {
      Cipher cipher = Cipher.getInstance("DESede/CBC/PKCS5Padding");
      cipher.init(2, key3Des, IvSpec);
      plainText = cipher.doFinal(cipherText);
    } catch (Exception e) {
      log.error("DecryptCipher Exception", e);
    }
    return plainText;
  }

  /**
   * 三重DES加密
   * @param ivData
   * @param keyData
   * @param plainText
   * @return
   * @throws Exception
   */
  public static byte[] DES3_CBC_Encrypt(byte[] ivData, byte[] keyData, byte[] plainText) throws Exception {
    SecretKeySpec key3Des = new SecretKeySpec(keyData, "DESede");
    IvParameterSpec IvSpec = new IvParameterSpec(ivData);
    byte[] cipherText = null;
    Cipher cipher = Cipher.getInstance("DESede/CBC/PKCS5Padding");
    cipher.init(1, key3Des, IvSpec);
    cipherText = cipher.doFinal(plainText);
    return cipherText;
  }

  /**
   * 三重DES解密
   *
   * @param cipherText
   * @return
   * @throws Exception
   */
  public static String DES3_Decrypt(String cipherText) throws Exception {
    String plainText = "";
    if (!StringUtils.isEmpty(cipherText)) {
      SecretKeySpec key3Des = new SecretKeySpec(KEY_DATA, "DESede");
      IvParameterSpec IvSpec = new IvParameterSpec(IV_DATA);
      Cipher cipher = Cipher.getInstance("DESede/CBC/PKCS5Padding");
      cipher.init(2, key3Des, IvSpec);
      plainText = new String(cipher.doFinal(hex2bytes(cipherText)));
    }
    return plainText;
  }

  /**
   * 三重DES加密
   *
   * @param plainText
   * @return
   * @throws Exception
   */
  public static String DES3_Encrypt(String plainText) throws Exception {
    String cipherText = "";
    if (!StringUtils.isEmpty(plainText)) {
      cipherText = bytes2hex(DES3_CBC_Encrypt(IV_DATA, KEY_DATA, plainText.getBytes(StandardCharsets.UTF_8)));
    }
    return cipherText;
  }

  /**
   * 三重DES解密
   *
   * @param cipherText
   * @return
   * @throws Exception
   */
  public static String DES3_CBC_Decrypt(String cipherText) throws Exception {
    String plainText = "";
    if (!StringUtils.isEmpty(cipherText)) {
      SecretKeySpec key3Des = new SecretKeySpec(KEY_DATA, "DESede");
      IvParameterSpec IvSpec = new IvParameterSpec(IV_DATA);
      Cipher cipher = Cipher.getInstance("DESede/CBC/PKCS5Padding");
      cipher.init(2, key3Des, IvSpec);
      plainText = new String(cipher.doFinal(hex2bytes(cipherText)));
    }
    return plainText;
  }

  /**
   * 三重DES加密
   *
   * @param plainText
   * @return
   * @throws Exception
   */
  public static String DES3_CBC_Encrypt(String plainText) throws Exception {
    String cipherText = "";
    if (!StringUtils.isEmpty(plainText)) {
      cipherText = bytes2hex(DES3_CBC_Encrypt(IV_DATA, KEY_DATA, plainText.getBytes(StandardCharsets.UTF_8)));
    }
    return cipherText;
  }

  /**
   * Rsa公钥加密
   * @param plainData
   * @param publicKey
   * @return
   * @throws Exception
   */
  public static String encryptByRSA(byte[] plainData, PublicKey publicKey) throws Exception {
    Cipher cipher = Cipher.getInstance("RSA/ECB/PKCS1Padding");
    cipher.init(1, publicKey);
    byte[] output = cipher.doFinal(plainData);
    return new String(Base64.encode(output));
  }

  /**
   * Rsa私钥解密
   * @param plainData
   * @param privateKey
   * @return
   * @throws Exception
   */
  public static String decryptRSA(String plainData, PrivateKey privateKey) throws Exception {
    Cipher cipher = Cipher.getInstance("RSA/ECB/PKCS1Padding");
    cipher.init(2, privateKey);
    byte[] output = cipher.doFinal(Base64.decode(plainData));
    return new String(output);
  }

  /**
   * 获取Rsa解密key
   * @param signData
   * @param privateKey
   * @return
   * @throws Exception
   */
  public static String getDecryptKeyByteByRSA(String signData, PrivateKey privateKey) throws Exception {
    Cipher cipher = Cipher.getInstance("RSA/ECB/PKCS1Padding");
    cipher.init(2, privateKey);
    byte[] output = cipher.doFinal(Base64.decode(signData));
    byte[] keyType = new byte[3];
    System.arraycopy(output, 0, keyType, 0, 3);
    byte[] key = new byte[output.length - 3];
    System.arraycopy(output, 3, key, 0, output.length - 3);
    String keyString = new String(keyType) + PayUtils.bytes2hex(key);
    return keyString;
  }

  /**
   * hex格式与byte转换
   *
   * @param hexString
   * @return
   */
  public static byte[] hex2bytes(String hexString) {
    hexString = hexString.toUpperCase();
    char[] chars = hexString.toCharArray();
    byte[] bytes = new byte[chars.length / 2];
    int index = 0;
    for (int i = 0; i < chars.length; i += 2) {
      byte newByte = 0;
      newByte = (byte)(newByte | char2byte(chars[i]));
      newByte = (byte)(newByte << 4);
      newByte = (byte)(newByte | char2byte(chars[i + 1]));
      bytes[index] = newByte;
      index++;
    }
    return bytes;
  }

  /**
   * byte与hex格式转换
   *
   * @param bytes
   * @return
   */
  public static String bytes2hex(byte[] bytes) {
    String result = "";
    StringBuilder b;
    if (null == bytes) {
      return null;
    }
    for (int i = 0; i < bytes.length; i++) {
      b = new StringBuilder(Integer.toHexString(bytes[i] & 0xFF));
      if (b.length() == 1){
        b.insert(0,"0");
      }
      result = result + b.toString();
    }
    return result.toUpperCase();
  }

  /**
   * 生成随机16进制字符串
   *
   * @param len
   * @return
   */
  public static String randomHexString(int len) {
    StringBuffer result = new StringBuffer();
    Random random = ThreadLocalRandom.current();
    for (int i = 0; i < len; i++) {
      result.append(Integer.toHexString(random.nextInt(16)));
    }
    return result.toString().toUpperCase();
  }

  /**
   * char转换成对应的byte
   * @param ch
   * @return
   */
  public static byte char2byte(char ch) {
    switch (ch) {
      case '0':
        return 0;
      case '1':
        return 1;
      case '2':
        return 2;
      case '3':
        return 3;
      case '4':
        return 4;
      case '5':
        return 5;
      case '6':
        return 6;
      case '7':
        return 7;
      case '8':
        return 8;
      case '9':
        return 9;
      case 'A':
        return 10;
      case 'B':
        return 11;
      case 'C':
        return 12;
      case 'D':
        return 13;
      case 'E':
        return 14;
      case 'F':
        return 15;
      default:
        throw new IllegalArgumentException(String.format("错误的参数[%c]",ch));
    }
  }
}
