package com.biz.crm.admin.web.service.internal;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.admin.web.dto.CostDetailReportPaginationDto;
import com.biz.crm.admin.web.repository.CostDetailReportVoRepository;
import com.biz.crm.admin.web.service.CostDetailReportVoService;
import com.biz.crm.admin.web.vo.CostDetailReportVo;
import com.biz.crm.cps.business.activity.scan.local.entity.ScanActivity;
import com.biz.crm.cps.business.activity.scan.local.service.ScanActivityService;
import com.biz.crm.cps.business.agreement.local.entity.ProfitAgreementTemplate;
import com.biz.crm.cps.business.agreement.local.service.ProfitAgreementTemplateService;
import com.biz.crm.cps.business.reward.sdk.common.enums.RewardRelateTypeEnum;
import com.google.common.collect.Maps;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * 费用明细流水报表接口定义实现
 * @Author dy
 * @DATE 2022/4/22
 **/
@Service("CostDetailReportVoServiceImpl")
public class CostDetailReportVoServiceImpl implements CostDetailReportVoService {

  @Autowired
  private CostDetailReportVoRepository costDetailReportVoRepository;
  @Autowired
  private ScanActivityService scanActivityService;
  @Autowired
  private ProfitAgreementTemplateService profitAgreementTemplateService;


  @Override
  public Page<CostDetailReportVo> findDealerByConditions(Pageable pageable, CostDetailReportPaginationDto dto) {

    dto = Optional.ofNullable(dto).orElse(new CostDetailReportPaginationDto());
    pageable = Optional.ofNullable(pageable).orElse(PageRequest.of(0, 50));
    Page<CostDetailReportVo> page = this.costDetailReportVoRepository.findDealerByConditions(pageable, dto);
    if (Objects.nonNull(page) && !CollectionUtils.isEmpty(page.getRecords())) {
      Map<String, String> mapAct = this.findScanActivityMap(page.getRecords());
      Map<String, String> mapTemplate = this.findAgreementMap(page.getRecords());
      for (CostDetailReportVo item : page.getRecords()) {
        if (StringUtils.isNoneBlank(item.getRelateType(), item.getRelateCode())
            && RewardRelateTypeEnum.SCAN_ACTIVITY.getDictCode().equals(item.getRelateType())) {
          item.setRelateActName(mapAct.get(item.getRelateCode()));
        }
        if (StringUtils.isNotBlank(item.getTemplateCode())) {
          item.setTemplateName(mapTemplate.get(item.getTemplateCode()));
        }
      }
    }
    return page;
  }

  /**
   * 返回扫码活动编码对应的扫码活动名称map
   *
   * @param list
   * @return k-actCode,v-扫码活动名称
   */
  private Map<String, String> findScanActivityMap(List<CostDetailReportVo> list) {
    if (CollectionUtils.isEmpty(list)) {
      return Maps.newHashMap();
    }
    Set<String> actCodeSet =
        list.stream()
            .filter(
                a ->
                    StringUtils.isNoneBlank(a.getRelateType(), a.getRelateCode())
                        && RewardRelateTypeEnum.SCAN_ACTIVITY
                        .getDictCode()
                        .equals(a.getRelateType()))
            .map(CostDetailReportVo::getRelateCode)
            .collect(Collectors.toSet());
    if (CollectionUtils.isEmpty(actCodeSet)) {
      return Maps.newHashMap();
    }
    List<ScanActivity> activityList = this.scanActivityService.findDetailByCodes(actCodeSet);
    if (CollectionUtils.isEmpty(activityList)) {
      return Maps.newHashMap();
    }

    return activityList.stream()
        .filter(a -> StringUtils.isNoneBlank(a.getCode(), a.getName()))
        .collect(Collectors.toMap(ScanActivity::getCode, ScanActivity::getName, (a, b) -> a));
  }

  /**
   * 返回协议模板名称
   *
   * @param list
   * @return k-协议模板编码,v-协议名称
   */
  private Map<String, String> findAgreementMap(List<CostDetailReportVo> list) {
    if (CollectionUtils.isEmpty(list)) {
      return Maps.newHashMap();
    }
    Set<String> templateCodeSet =
        list.stream()
            .filter(a -> StringUtils.isNotBlank(a.getTemplateCode()))
            .map(CostDetailReportVo::getTemplateCode)
            .collect(Collectors.toSet());
    if (CollectionUtils.isEmpty(templateCodeSet)) {
      return Maps.newHashMap();
    }
    List<ProfitAgreementTemplate> templateList =
        this.profitAgreementTemplateService.findByTemplateCodes(templateCodeSet);
    if (CollectionUtils.isEmpty(templateList)) {
      return Maps.newHashMap();
    }

    return templateList.stream()
        .filter(a -> StringUtils.isNoneBlank(a.getTemplateCode(), a.getTemplateName()))
        .collect(
            Collectors.toMap(
                ProfitAgreementTemplate::getTemplateCode,
                ProfitAgreementTemplate::getTemplateName,
                (a, b) -> a));
  }
}
