package com.biz.crm.admin.web.service.internal;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.admin.web.dto.DisplayCalculateTaskReportDto;
import com.biz.crm.admin.web.dto.DisplayTaskReportDto;
import com.biz.crm.admin.web.dto.TaskScanCodeItemStatisticsDto;
import com.biz.crm.admin.web.repository.DisplayPolicyReportRepository;
import com.biz.crm.admin.web.service.DisplayPolicyReportService;
import com.biz.crm.admin.web.service.RewardRelTriggerObjectVoService;
import com.biz.crm.admin.web.service.RewardTypeStatisticsVoService;
import com.biz.crm.admin.web.vo.DisplayCalculateTaskReportVo;
import com.biz.crm.admin.web.vo.DisplayStatisticsReportVo;
import com.biz.crm.admin.web.vo.DisplayTaskReportVo;
import com.biz.crm.admin.web.vo.ShareProfitStatisticsVo;
import com.biz.crm.admin.web.vo.TaskScanCodeItemStatisticsVo;
import com.biz.crm.admin.web.vo.TaskScanCodeStatisticsVo;
import com.biz.crm.cps.bisiness.policy.display.sdk.common.constant.DisplayConstant;
import com.biz.crm.cps.bisiness.policy.display.sdk.common.enums.DisplayTaskStatusEnum;
import com.biz.crm.cps.bisiness.policy.display.sdk.common.enums.DisplayTaskStatusNewEnum;
import com.biz.crm.cps.business.participator.sdk.service.ParticipatorTagVoService;
import com.biz.crm.cps.business.participator.sdk.vo.ParticipatorTagVo;
import com.biz.crm.cps.business.policy.display.ladder.local.entity.DisplayCalculateTask;
import com.biz.crm.cps.business.policy.display.ladder.local.model.DisplayCalculateTaskRelateTaskModelVo;
import com.biz.crm.cps.business.policy.display.ladder.local.model.DisplayPolicyConfigModelVo;
import com.biz.crm.cps.business.policy.display.ladder.local.service.DisplayCalculateTaskRelateTaskService;
import com.biz.crm.cps.business.policy.display.ladder.local.service.DisplayCalculateTaskService;
import com.biz.crm.cps.business.policy.display.ladder.local.service.DisplayPolicyService;
import com.biz.crm.cps.business.policy.display.ladder.local.service.observer.DisplayPolicyMountRegisterImpl;
import com.biz.crm.cps.business.reward.sdk.vo.RewardRelTriggerObjectVo;
import com.biz.crm.cps.business.reward.sdk.vo.RewardTypeStatisticsVo;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * 陈列政策报表service实现类
 *
 * @author HuaHongQiang
 */
@Service
public class DisplayPolicyReportServiceImpl implements DisplayPolicyReportService {

  @Autowired private DisplayPolicyReportRepository displayPolicyReportRepository;

  @Autowired private RewardTypeStatisticsVoService rewardTypeStatisticsVoService;

  @Autowired private RewardRelTriggerObjectVoService rewardRelTriggerObjectVoService;

  @Autowired private DisplayPolicyMountRegisterImpl displayPolicyMountRegister;

  @Autowired private ParticipatorTagVoService participatorTagVoService;

  @Autowired private DisplayPolicyService displayPolicyService;

  @Autowired private DisplayCalculateTaskService displayCalculateTaskService;

  @Autowired private DisplayCalculateTaskRelateTaskService displayCalculateTaskRelateTaskService;

  @Override
  public DisplayStatisticsReportVo findDisplayStatisticsReportByTenant() {
    DisplayStatisticsReportVo displayStatisticsReportVo = new DisplayStatisticsReportVo();
    // 获取本月开始时间
    Calendar cal = Calendar.getInstance();
    cal.set(
        cal.get(Calendar.YEAR), cal.get(Calendar.MONDAY), cal.get(Calendar.DAY_OF_MONTH), 0, 0, 0);
    cal.set(Calendar.DAY_OF_MONTH, cal.getActualMinimum(Calendar.DAY_OF_MONTH));
    Date monthStartTime = cal.getTime();
    // 获取本月结束时间(下个月开始时间)
    cal.set(
        cal.get(Calendar.YEAR), cal.get(Calendar.MONDAY), cal.get(Calendar.DAY_OF_MONTH), 0, 0, 0);
    cal.set(Calendar.DAY_OF_MONTH, cal.getActualMaximum(Calendar.DAY_OF_MONTH));
    cal.set(Calendar.HOUR_OF_DAY, 24);
    Date monthEndTime = cal.getTime();
    // 获取当月任务相关终端数
    displayStatisticsReportVo.setExecutionTerminalNumbers(
        this.displayPolicyReportRepository.findDisplayStatisticsReportByTenant(
            TenantUtils.getTenantCode(), monthStartTime, monthEndTime, null));
    // 获取已执行终端数
    displayStatisticsReportVo.setFinishTerminalNumbers(
        this.displayPolicyReportRepository.findDisplayStatisticsReportByTenant(
            TenantUtils.getTenantCode(),
            monthStartTime,
            monthEndTime,
            DisplayTaskStatusEnum.STATUS_PASS.getDictCode()));
    // 获取分利信息
    displayStatisticsReportVo.setRewardTypeStatisticsVos(
        rewardTypeStatisticsVoService.findByTenantCodeAndTriggerAction(
            TenantUtils.getTenantCode(), this.displayPolicyMountRegister.getName()));
    return displayStatisticsReportVo;
  }

  @Override
  public Page<DisplayTaskReportVo> findByConditions(Pageable pageable, DisplayTaskReportDto dto) {
    pageable = Optional.ofNullable(pageable).orElse(PageRequest.of(0, 50));
    dto = Optional.ofNullable(dto).orElse(new DisplayTaskReportDto());
    // k-终端编码，v-逗号隔开的标签信息
    Map<String, String> mapTag = Maps.newHashMap();
    Page<DisplayTaskReportVo> page =
        this.displayPolicyReportRepository.findByConditions(pageable, dto);
    if (Objects.isNull(page) || CollectionUtils.isEmpty(page.getRecords())) {
      return page;
    }

    // 陈列商品
    Set<String> policyIds =
        page.getRecords().stream()
            .filter(a -> StringUtils.isNotBlank(a.getDisplayPolicyId()))
            .map(DisplayTaskReportVo::getDisplayPolicyId)
            .collect(Collectors.toSet());
    // k-政策id，v-陈列商品
    Map<String, DisplayPolicyConfigModelVo> mapRange = this.buildRangeMap(policyIds);
    // 获取终端对应的标签数据
    Set<String> terminalCodes =
        page.getRecords().stream()
            .filter(a -> StringUtils.isNotBlank(a.getTerminalCode()))
            .map(DisplayTaskReportVo::getTerminalCode)
            .collect(Collectors.toSet());
    if (!CollectionUtils.isEmpty(terminalCodes)) {
      mapTag = this.findTerminalTagMap(terminalCodes);
    }
    for (DisplayTaskReportVo record : page.getRecords()) {
      record.setTerminalTag(mapTag.getOrDefault(record.getTerminalCode(), StringUtils.EMPTY));
      record.setRange(mapRange.getOrDefault(record.getDisplayPolicyId(), new DisplayPolicyConfigModelVo()).getSpecialName());
    }
    return page;
  }

  /**
   * 构建陈列商品范围信息映射(key:政策id,value:陈列商品)
   *
   * @param policyIds 陈列政策id集合
   * @return 陈列商品范围信息映射
   */
  private Map<String, DisplayPolicyConfigModelVo> buildRangeMap(Set<String> policyIds) {
    Map<String, DisplayPolicyConfigModelVo> mapRange = Maps.newHashMap();
    if (CollectionUtils.isEmpty(policyIds)) {
      return mapRange;
    }
    List<DisplayPolicyConfigModelVo> configModelVoList =
        displayPolicyService.findDisplayPolicyConfigModelVoByPolicyIds(policyIds);
    if (CollectionUtils.isEmpty(configModelVoList)) {
      return mapRange;
    }
    mapRange = configModelVoList.stream()
        .filter(
            a -> StringUtils.isNotBlank(a.getPolicyId()) && StringUtils.isNotBlank(a.getSpecialName()))
        .collect(Collectors.toMap(DisplayPolicyConfigModelVo::getPolicyId,t -> t,(a,b) -> {
          a.setSpecialCode(String.format("%s;%s",a.getSpecialCode(),b.getSpecialCode()));
          a.setSpecialName(String.format("%s;%s",a.getSpecialName(),b.getSpecialName()));
          return a;
        }));
    return mapRange;
  }

  @Override
  public Page<DisplayCalculateTaskReportVo> findByDisplayCalculateTaskReportDto(
      Pageable pageable, DisplayCalculateTaskReportDto dto) {
    pageable = Optional.ofNullable(pageable).orElse(PageRequest.of(0, 50));
    dto = Optional.ofNullable(dto).orElse(new DisplayCalculateTaskReportDto());
    // k-终端编码，v-逗号隔开的标签信息
    Map<String, String> mapTag = Maps.newHashMap();
    Page<DisplayCalculateTaskReportVo> page =
        this.displayPolicyReportRepository.findByDisplayCalculateTaskReportDto(pageable, dto);
    if (Objects.isNull(page) || CollectionUtils.isEmpty(page.getRecords())) {
      return page;
    }
    List<String> businessCodes = Lists.newArrayList();
    Set<String> terminalCodes = Sets.newHashSet();
    List<String> allBusinessCodes = Lists.newArrayList();
    Set<String> policyIds = Sets.newHashSet();
    page.getRecords().forEach(reportVo -> {
      if (DisplayTaskStatusNewEnum.SUCCESS.getDictCode().equals(reportVo.getTaskStatus()) && StringUtils.isNotBlank(reportVo.getBusinessCode())) {
        businessCodes.add(reportVo.getBusinessCode());
      }
      if (StringUtils.isNotBlank(reportVo.getTerminalCode())) {
        terminalCodes.add(reportVo.getTerminalCode());
      }
      if (StringUtils.isNotBlank(reportVo.getBusinessCode())) {
        allBusinessCodes.add(reportVo.getBusinessCode());
      }
      if (StringUtils.isNotBlank(reportVo.getDisplayPolicyId())) {
        policyIds.add(reportVo.getDisplayPolicyId());
      }
    });
    // 获取任务对应的分利数据信息
    Map<String, List<RewardTypeStatisticsVo>> mapReward = Maps.newHashMap();
    if (!CollectionUtils.isEmpty(businessCodes)) {
      List<RewardRelTriggerObjectVo> rewardRelTriggerObjectVoList =
          rewardRelTriggerObjectVoService.findByTriggerActionAndTriggerObjects(
              DisplayConstant.TRIGGER_ACTION, businessCodes);
      if (!CollectionUtils.isEmpty(rewardRelTriggerObjectVoList)) {
        mapReward =
            rewardRelTriggerObjectVoList.stream()
                .filter(
                    a ->
                        StringUtils.isNotBlank(a.getTriggerObject())
                            && !CollectionUtils.isEmpty(a.getRewardTypeStatisticsVos()))
                .collect(
                    Collectors.toMap(
                        RewardRelTriggerObjectVo::getTriggerObject,
                        RewardRelTriggerObjectVo::getRewardTypeStatisticsVos,
                        (a, b) -> a));
      }
    }
    // 获取终端对应的标签数据
    if (!CollectionUtils.isEmpty(terminalCodes)) {
      mapTag = this.findTerminalTagMap(terminalCodes);
    }
    // 陈列情况
    // k-陈列任务编码, v-陈列数量
    Map<String, Long> taskDisplayTimesMap = Maps.newHashMap();
    // k-陈列任务编码, v-已完成的陈列数量
    Map<String, Long> taskDisplayFinishTimesMap = Maps.newHashMap();
    if (!CollectionUtils.isEmpty(allBusinessCodes)) {
      List<DisplayCalculateTaskRelateTaskModelVo> relateTaskModelVoList =
          this.displayCalculateTaskRelateTaskService.findByTaskCodes(allBusinessCodes);
      if (!CollectionUtils.isEmpty(relateTaskModelVoList)) {
        taskDisplayTimesMap =
            relateTaskModelVoList.stream()
                .collect(
                    Collectors.groupingBy(
                        DisplayCalculateTaskRelateTaskModelVo::getCalculateTaskCode,
                        Collectors.counting()));
        taskDisplayFinishTimesMap =
            relateTaskModelVoList.stream()
                .filter(
                    a ->
                        StringUtils.isNotBlank(a.getDetailTaskStatus())
                            && DisplayTaskStatusNewEnum.SUCCESS
                                .getDictCode()
                                .equals(a.getDetailTaskStatus()))
                .collect(
                    Collectors.groupingBy(
                        DisplayCalculateTaskRelateTaskModelVo::getCalculateTaskCode,
                        Collectors.counting()));
      }
    }
    // k-政策id，v-陈列商品
    Map<String, DisplayPolicyConfigModelVo> mapRange = this.buildRangeMap(policyIds);
    for (DisplayCalculateTaskReportVo record : page.getRecords()) {
      DisplayPolicyConfigModelVo configModelVo = mapRange.getOrDefault(record.getDisplayPolicyId(), new DisplayPolicyConfigModelVo());
      record.setTagDesc(mapTag.getOrDefault(record.getTerminalCode(), StringUtils.EMPTY));
      record.setRewardTypeStatisticsVos(mapReward.get(record.getBusinessCode()));
      record.setDisplayTimes(
          new BigDecimal(taskDisplayTimesMap.getOrDefault(record.getBusinessCode(), 0L)));
      record.setDisplayFinishTimes(
          new BigDecimal(taskDisplayFinishTimesMap.getOrDefault(record.getBusinessCode(), 0L)));
      record.setDisplayFinishRate(
          this.rate(record.getDisplayFinishTimes(), record.getDisplayTimes()));
      record.setScanRate(this.rate(record.getSaleActual(), record.getSaleTarget()));
      record.setRange(configModelVo.getSpecialName());
      record.setDimensionName(configModelVo.getDimensionName());
    }
    //todo,暂时直接取值按照任务类型排序 (月。季度，半年，年)
    ArrayList<DisplayCalculateTaskReportVo> orderList = new ArrayList<>();
    Map<String, List<DisplayCalculateTaskReportVo>> collect = page.getRecords().stream().collect(Collectors.groupingBy(DisplayCalculateTaskReportVo::getTaskType));
    if (!CollectionUtils.isEmpty(collect.get(DisplayConstant.MONTH))) {
      orderList.addAll(collect.get(DisplayConstant.MONTH));
    }
    if (!CollectionUtils.isEmpty(collect.get(DisplayConstant.SEASON))) {
      orderList.addAll(collect.get(DisplayConstant.SEASON));
    }
    if (!CollectionUtils.isEmpty(collect.get(DisplayConstant.HALF_YEAR))) {
      orderList.addAll(collect.get(DisplayConstant.HALF_YEAR));
    }
    if (!CollectionUtils.isEmpty(collect.get(DisplayConstant.YEAR))) {
      orderList.addAll(collect.get(DisplayConstant.YEAR));
    }
    page.setRecords(orderList);
    return page;
  }

  @Override
  public Page<TaskScanCodeItemStatisticsVo> findTaskScanCodeItemStatisticsVo(
      Pageable pageable, TaskScanCodeItemStatisticsDto dto) {
    if (Objects.isNull(dto) || StringUtils.isBlank(dto.getBusinessCode())) {
      return null;
    }
    dto.setSaleActual(Optional.ofNullable(dto.getSaleActual()).orElse(BigDecimal.ZERO));
    pageable = Optional.ofNullable(pageable).orElse(PageRequest.of(0, 50));
    Page<TaskScanCodeItemStatisticsVo> page =
        this.displayPolicyReportRepository.findTaskScanCodeItemStatisticsVo(
            pageable, dto.getBusinessCode());
    if (Objects.nonNull(page) && !CollectionUtils.isEmpty(page.getRecords())) {
      for (TaskScanCodeItemStatisticsVo record : page.getRecords()) {
        if (BigDecimal.ZERO.compareTo(dto.getSaleActual()) < 0) {
          BigDecimal rate =
              Optional.ofNullable(record.getSaleActual())
                  .orElse(BigDecimal.ZERO)
                  .divide(dto.getSaleActual(), 4, RoundingMode.HALF_UP)
                  .multiply(new BigDecimal(100));
          record.setRate(rate);
        }
      }
    }
    return page;
  }

  @Override
  public Page<ShareProfitStatisticsVo> findShareProfitStatisticsVo(
      Pageable pageable, String businessCode) {
    if (StringUtils.isBlank(businessCode)) {
      return null;
    }
    pageable = Optional.ofNullable(pageable).orElse(PageRequest.of(0, 50));
    return this.displayPolicyReportRepository.findShareProfitStatisticsVo(pageable, businessCode);
  }

  @Override
  public TaskScanCodeStatisticsVo findTaskScanCodeStatisticsVo(String businessCode) {
    if (StringUtils.isBlank(businessCode)) {
      return null;
    }
    List<DisplayCalculateTask> list =
        this.displayCalculateTaskService.findByBusinessCodes(Lists.newArrayList(businessCode));
    if (CollectionUtils.isEmpty(list)) {
      return null;
    }
    DisplayCalculateTask task = list.get(0);
    TaskScanCodeStatisticsVo vo = new TaskScanCodeStatisticsVo();
    vo.setBindSaleFlag(Boolean.FALSE);
    if (DisplayConstant.ZERO.equals(task.getBindSaleStatus())) {
      vo.setBindSaleFlag(Boolean.TRUE);
    }
    if (!vo.getBindSaleFlag()) {
      return vo;
    }
    vo.setSaleTarget(Optional.ofNullable(task.getSaleTarget()).orElse(BigDecimal.ZERO));
    vo.setSaleActual(Optional.ofNullable(task.getSaleActual()).orElse(BigDecimal.ZERO));
    vo.setRate(BigDecimal.ZERO);
    if (BigDecimal.ZERO.compareTo(vo.getSaleTarget()) < 0
        && BigDecimal.ZERO.compareTo(vo.getSaleActual()) < 0) {
      BigDecimal rate =
          Optional.ofNullable(vo.getSaleActual())
              .orElse(BigDecimal.ZERO)
              .divide(vo.getSaleTarget(), 4, RoundingMode.HALF_UP)
              .multiply(new BigDecimal(100));
      vo.setRate(rate);
    }
    String policyId = task.getDisplayPolicyId();
    if (StringUtils.isBlank(policyId)) {
      return vo;
    }
    List<DisplayPolicyConfigModelVo> configModelVoList =
        displayPolicyService.findDisplayPolicyConfigModelVoByPolicyIds(Sets.newHashSet(policyId));
    if (CollectionUtils.isEmpty(configModelVoList)) {
      return vo;
    }
    vo.setDimensionName(configModelVoList.get(0).getDimensionName());
    vo.setRange(
        configModelVoList.stream()
            .filter(a -> StringUtils.isNotBlank(a.getSpecialName()))
            .map(DisplayPolicyConfigModelVo::getSpecialName)
            .collect(Collectors.joining(";")));
    return vo;
  }

  /**
   * 获取终端标签信息map k-terminalCode,v-tag,tag
   *
   * @param terminalCodes
   * @return
   */
  private Map<String, String> findTerminalTagMap(Set<String> terminalCodes) {
    Map<String, String> map = Maps.newHashMap();
    if (!CollectionUtils.isEmpty(terminalCodes)) {
      List<ParticipatorTagVo> tagVoList =
          participatorTagVoService.findByParticipatorCodes(Lists.newArrayList(terminalCodes));
      if (!CollectionUtils.isEmpty(tagVoList)) {
        map =
            tagVoList.stream()
                .collect(
                    Collectors.groupingBy(
                        ParticipatorTagVo::getParticipatorCode,
                        Collectors.mapping(
                            ParticipatorTagVo::getTagDescription, Collectors.joining(","))));
      }
    }
    return map;
  }

  /**
   * 获取比例
   *
   * @param v1
   * @param v2
   * @return
   */
  private BigDecimal rate(BigDecimal v1, BigDecimal v2) {
    v1 = Optional.ofNullable(v1).orElse(BigDecimal.ZERO);
    v2 = Optional.ofNullable(v2).orElse(BigDecimal.ZERO);
    if (v1.compareTo(BigDecimal.ZERO) == 0 || v2.compareTo(BigDecimal.ZERO) == 0) {
      return BigDecimal.ZERO;
    }
    return v1.divide(v2, 4, RoundingMode.HALF_UP).multiply(new BigDecimal(100));
  }
}
