package com.biz.crm.admin.web.service.internal;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.admin.web.dto.IntegralReportPaginationDto;
import com.biz.crm.admin.web.repository.IntegralReportRepository;
import com.biz.crm.admin.web.service.IntegralReportService;
import com.biz.crm.admin.web.vo.IntegralReportVo;
import com.biz.crm.cps.business.activity.scan.sdk.service.ScanActivityVoService;
import com.biz.crm.cps.business.activity.scan.sdk.vo.ScanActivityVo;
import com.biz.crm.cps.business.agreement.local.entity.ProfitAgreementTemplate;
import com.biz.crm.cps.business.agreement.local.service.ProfitAgreementTemplateService;
import com.biz.crm.cps.business.reward.sdk.common.enums.RewardRelateTypeEnum;
import com.google.common.collect.Maps;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

/**
 * 积分报表 vo 接口实现
 *
 * @author sunx
 * @date 2022/3/24
 */
@Service
@Slf4j
public class IntegralReportServiceImpl implements IntegralReportService {

  @Autowired private IntegralReportRepository integralReportRepository;

  @Autowired private ScanActivityVoService scanActivityVoService;

  @Autowired private ProfitAgreementTemplateService profitAgreementTemplateService;

  @Override
  public Page<IntegralReportVo> findByConditions(
      Pageable pageable, IntegralReportPaginationDto dto) {
    dto = Optional.ofNullable(dto).orElse(new IntegralReportPaginationDto());
    pageable = Optional.ofNullable(pageable).orElse(PageRequest.of(0, 50));
    Page<IntegralReportVo> page = this.integralReportRepository.findByConditions(pageable, dto);
    if (Objects.nonNull(page) && !CollectionUtils.isEmpty(page.getRecords())) {
      Map<String, String> mapAct = this.findScanActivityMap(page.getRecords());
      Map<String, String> mapTemplate = this.findAgreementMap(page.getRecords());
      for (IntegralReportVo item : page.getRecords()) {
        if (StringUtils.isNoneBlank(item.getRelateType(), item.getRelateCode())
            && RewardRelateTypeEnum.SCAN_ACTIVITY.getDictCode().equals(item.getRelateType())) {
          item.setRelateActName(mapAct.get(item.getRelateCode()));
        }
        if (StringUtils.isNotBlank(item.getTemplateCode())) {
          item.setTemplateName(mapTemplate.get(item.getTemplateCode()));
        }
      }
    }
    return page;
  }

  /**
   * 获取扫码活动名称map
   *
   * @param list
   * @return k-code,v-name
   */
  private Map<String, String> findScanActivityMap(List<IntegralReportVo> list) {
    Map<String, String> map = Maps.newHashMap();
    Set<String> actCodeSet =
        list.stream()
            .filter(
                a ->
                    StringUtils.isNoneBlank(a.getRelateCode(), a.getRelateType())
                        && RewardRelateTypeEnum.SCAN_ACTIVITY
                            .getDictCode()
                            .equals(a.getRelateType()))
            .map(IntegralReportVo::getRelateCode)
            .collect(Collectors.toSet());
    if (CollectionUtils.isEmpty(actCodeSet)) {
      return map;
    }
    List<ScanActivityVo> activityVoList = this.scanActivityVoService.findMainByActCodes(actCodeSet);
    if (CollectionUtils.isEmpty(activityVoList)) {
      return map;
    }
    return activityVoList.stream()
        .filter(a -> StringUtils.isNoneBlank(a.getCode(), a.getName()))
        .collect(Collectors.toMap(ScanActivityVo::getCode, ScanActivityVo::getName, (a, b) -> a));
  }

  /**
   * 获取协议名称map
   *
   * @param list
   * @return k-code,v-name
   */
  private Map<String, String> findAgreementMap(List<IntegralReportVo> list) {
    Map<String, String> map = Maps.newHashMap();
    Set<String> templateCodeSet =
        list.stream()
            .filter(a -> StringUtils.isNotBlank(a.getTemplateCode()))
            .map(IntegralReportVo::getTemplateCode)
            .collect(Collectors.toSet());
    if (CollectionUtils.isEmpty(templateCodeSet)) {
      return map;
    }
    List<ProfitAgreementTemplate> profitAgreementTemplateList =
        this.profitAgreementTemplateService.findByTemplateCodes(templateCodeSet);
    if (CollectionUtils.isEmpty(profitAgreementTemplateList)) {
      return map;
    }
    return profitAgreementTemplateList.stream()
        .filter(a -> StringUtils.isNoneBlank(a.getTemplateCode(), a.getTemplateName()))
        .collect(
            Collectors.toMap(
                ProfitAgreementTemplate::getTemplateCode,
                ProfitAgreementTemplate::getTemplateName,
                (a, b) -> a));
  }
}
