package com.biz.crm.admin.web.service.internal;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.admin.web.repository.QuantifyStatisticsVoRepository;
import com.biz.crm.admin.web.service.QuantifyStatisticsVoTaskService;
import com.biz.crm.admin.web.vo.QuantifyStatisticsVo;
import com.biz.crm.cps.business.policy.quantify.fiscal.local.entity.QuantifyConfiguration;
import com.biz.crm.cps.business.policy.quantify.fiscal.local.entity.QuantifyPolicy;
import com.biz.crm.cps.business.policy.quantify.fiscal.local.entity.QuantifyRange;
import com.biz.crm.cps.business.policy.quantify.fiscal.local.entity.QuantifyTask;
import com.biz.crm.cps.business.policy.quantify.fiscal.local.repository.QuantifyTaskRepository;
import com.biz.crm.cps.business.policy.quantify.fiscal.local.service.QuantifyPolicyService;
import com.biz.crm.cps.business.product.sdk.common.constant.MaterialDimensionConstant;
import com.biz.crm.cps.business.product.sdk.service.MaterialVoService;
import com.biz.crm.cps.external.barcode.sdk.dto.ScanCodeRecordPageDto;
import com.google.common.collect.Lists;
import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.Objects;
import java.util.Optional;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;

/** 包量任务统计接口实现 @Author: zengxingwang @Date: 2021/11/23 16:43 */
@Service
public class QuantifyStatisticsVoTaskServiceImpl implements QuantifyStatisticsVoTaskService {

  @Autowired private QuantifyTaskRepository quantifyTaskRepository;
  @Autowired private QuantifyPolicyService quantifyPolicyService;
  @Autowired private MaterialVoService materialVoService;
  @Autowired private QuantifyStatisticsVoRepository quantifyStatisticsVoRepository;

  /**
   * 查询销量明细
   *
   * @param businessCode
   * @return
   */
  @Override
  public Page<QuantifyStatisticsVo> findByBusinessCode(String businessCode, Pageable pageable) {
    ObjectUtils.defaultIfNull(pageable, PageRequest.of(0, 50));
    QuantifyTask task = quantifyTaskRepository.findByBusinessCode(businessCode);
    QuantifyPolicy policy = quantifyPolicyService.findByTemplateCode(task.getTemplateId());
    if (policy == null) {
      return new Page<>();
    }
    List<QuantifyStatisticsVo> list = new ArrayList<>();
    for (QuantifyConfiguration configuration : policy.getQuantifyConfigurations()) {
      ScanCodeRecordPageDto scanCodeRecordPageDto = new ScanCodeRecordPageDto();
      scanCodeRecordPageDto.setCreateTimeStart(task.getTaskStartTime());
      if (task.getProfitTime() == null) {
        scanCodeRecordPageDto.setCreateTimeEnd(task.getTaskEndTime());
      } else {
        scanCodeRecordPageDto.setCreateTimeEnd(task.getProfitTime());
      }
      if (MaterialDimensionConstant.DIMENSION_ALL.equals(policy.getDimensionFlag())) {
        // 如果商品标签是全部，则直接调用扫码模块获取该时间段内扫码数量
        scanCodeRecordPageDto.setProductCodes(Lists.newArrayList());
      } else {
        // 非全部商品，则通过产品维度获取物料编码集合，然后调用扫码模块获取扫码数量
        List<String> dimensionCodes =
            configuration.getQuantifyRanges().stream()
                .map(QuantifyRange::getSpecialCode)
                .collect(Collectors.toList());
        List<String> materialCodes =
            materialVoService.findMaterialCodeByDimensionCodesAndDimensionType(
                dimensionCodes, policy.getDimensionFlag());
        if (!CollectionUtils.isEmpty(materialCodes)) {
          scanCodeRecordPageDto.setProductCodes(materialCodes);
        }
      }
      List<QuantifyStatisticsVo> vos =
          quantifyStatisticsVoRepository.findByScanCodeRecordPageDto(scanCodeRecordPageDto);
      list.addAll(vos);
    }
    Page<QuantifyStatisticsVo> voPage = new Page<>();
    voPage.setRecords(list);
    voPage.setSize(pageable.getPageSize());
    voPage.setTotal(list.size());
    return voPage;
  }

  @Override
  public Page<QuantifyStatisticsVo> findByBusinessCodeForForMaterial(
      String businessCode, Pageable pageable) {
    pageable = Optional.ofNullable(pageable).orElse(PageRequest.of(0, 50));
    if(StringUtils.isBlank(businessCode)){
      return null;
    }
    QuantifyTask task = quantifyTaskRepository.findByBusinessCode(businessCode);
    if (Objects.isNull(task)) {
      return null;
    }
    BigDecimal saleActual = Optional.ofNullable(task.getSaleActual()).orElse(BigDecimal.ZERO);
    QuantifyPolicy policy = quantifyPolicyService.findByTemplateCode(task.getTemplateId());
    if (Objects.isNull(policy)) {
      return null;
    }
    String dimensionName = policy.getDimensionName();
    if (CollectionUtils.isEmpty(policy.getQuantifyConfigurations())) {
      return null;
    }
    QuantifyConfiguration configuration =
        policy.getQuantifyConfigurations().stream().findFirst().get();
    String rangesDesc = "";
    if (!CollectionUtils.isEmpty(configuration.getQuantifyRanges())) {
      rangesDesc =
          configuration.getQuantifyRanges().stream()
              .map(QuantifyRange::getSpecialName)
              .collect(Collectors.joining(","));
    }

    if (!MaterialDimensionConstant.DIMENSION_MATERIAL.equals(policy.getDimensionFlag())) {
      return null;
    }
    ScanCodeRecordPageDto scanCodeRecordPageDto = new ScanCodeRecordPageDto();
    scanCodeRecordPageDto.setBusinessCode(businessCode);
    Page<QuantifyStatisticsVo> pageResult =
        this.quantifyStatisticsVoRepository.findByScanCodeRecordPageDtoForMaterial(
            scanCodeRecordPageDto, pageable);
    if (Objects.nonNull(pageResult) && !CollectionUtils.isEmpty(pageResult.getRecords())) {
      for (QuantifyStatisticsVo item : pageResult.getRecords()) {
        item.setDimensionRange(rangesDesc);
        item.setDimensionName(dimensionName);
        item.setActualProportion(BigDecimal.ZERO);
        if (BigDecimal.ZERO.compareTo(saleActual) < 0
            && BigDecimal.ZERO.compareTo(item.getAmount()) != 0) {
          BigDecimal rate =
              item.getAmount()
                  .divide(saleActual, 4, RoundingMode.HALF_UP)
                  .multiply(new BigDecimal(100));
          item.setActualProportion(rate);
        }
      }
    }
    return pageResult;
  }

  /**
   * 查询返利明细
   *
   * @param businessCode
   * @return
   */
  @Override
  public Page<QuantifyStatisticsVo> findByBusinessCodeAndTriggerAction(
      String businessCode, Pageable pageable) {
    if (StringUtils.isBlank(businessCode)) {
      return new Page<>();
    }
    ObjectUtils.defaultIfNull(pageable, PageRequest.of(0, 50));
    return quantifyStatisticsVoRepository.findByBusinessCodeAndTriggerAction(
        businessCode, pageable);
  }
}
