package com.biz.crm.admin.web.service.internal;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.admin.web.dto.RedPacketReportPaginationDto;
import com.biz.crm.admin.web.repository.RedPacketReportRepository;
import com.biz.crm.admin.web.service.RedPacketReportService;
import com.biz.crm.admin.web.vo.RedPacketReportVo;
import com.biz.crm.cps.business.activity.scan.sdk.service.ScanActivityVoService;
import com.biz.crm.cps.business.activity.scan.sdk.vo.ScanActivityVo;
import com.biz.crm.cps.business.agreement.local.entity.ProfitAgreementTemplate;
import com.biz.crm.cps.business.agreement.local.service.ProfitAgreementTemplateService;
import com.biz.crm.cps.business.participator.sdk.common.enums.ParticipatorTypeEnum;
import com.biz.crm.cps.business.reward.sdk.common.enums.RewardRelateTypeEnum;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * 红包报表接口实现
 *
 * @author sunx
 * @date 2022/3/24
 */
@Service
@Slf4j
public class RedPacketReportServiceImpl implements RedPacketReportService {

  @Autowired private RedPacketReportRepository redPacketReportRepository;

  @Autowired private ScanActivityVoService scanActivityVoService;

  @Autowired private ProfitAgreementTemplateService profitAgreementTemplateService;

  @Override
  public Page<RedPacketReportVo> findByConditions(
      Pageable pageable, RedPacketReportPaginationDto dto) {
    dto = Optional.ofNullable(dto).orElse(new RedPacketReportPaginationDto());
    //  dto.setParticipatorType(ParticipatorTypeEnum.USER_GUIDE.getDictCode());
    pageable = Optional.ofNullable(pageable).orElse(PageRequest.of(0, 50));
    Page<RedPacketReportVo> page = this.redPacketReportRepository.findByConditions(pageable, dto);
    if (Objects.nonNull(page) && !CollectionUtils.isEmpty(page.getRecords())) {
      Map<String, String> mapAct = this.findScanActivityMap(page.getRecords());
      Map<String, String> mapTemplate = this.findAgreementMap(page.getRecords());
      for (RedPacketReportVo item : page.getRecords()) {
        if (StringUtils.isNoneBlank(item.getRelateType(), item.getRelateCode())
            && RewardRelateTypeEnum.SCAN_ACTIVITY.getDictCode().equals(item.getRelateType())) {
          item.setRelateActName(mapAct.get(item.getRelateCode()));
        }
        if (StringUtils.isNotBlank(item.getTemplateCode())) {
          item.setTemplateName(mapTemplate.get(item.getTemplateCode()));
        }
      }
    }
    return page;
  }

  @Override
  public Page<RedPacketReportVo> findTerminalByConditions(Pageable pageable, RedPacketReportPaginationDto dto) {
      dto = Optional.ofNullable(dto).orElse(new RedPacketReportPaginationDto());
      dto.setParticipatorType(ParticipatorTypeEnum.TERMINAL.getDictCode());
      pageable = Optional.ofNullable(pageable).orElse(PageRequest.of(0, 50));
      Page<RedPacketReportVo> page = this.redPacketReportRepository.findTerminalByConditions(pageable, dto);
      if (Objects.nonNull(page) && !CollectionUtils.isEmpty(page.getRecords())) {
        Map<String, String> mapAct = this.findScanActivityMap(page.getRecords());
        Map<String, String> mapTemplate = this.findAgreementMap(page.getRecords());
        for (RedPacketReportVo item : page.getRecords()) {
          if (StringUtils.isNoneBlank(item.getRelateType(), item.getRelateCode())
              && RewardRelateTypeEnum.SCAN_ACTIVITY.getDictCode().equals(item.getRelateType())) {
            item.setRelateActName(mapAct.get(item.getRelateCode()));
          }
          if (StringUtils.isNotBlank(item.getTemplateCode())) {
            item.setTemplateName(mapTemplate.get(item.getTemplateCode()));
          }
        }
      }
      return page;
  }

  @Override
  public Page<RedPacketReportVo> findDealerByConditions(Pageable pageable, RedPacketReportPaginationDto dto) {
    dto = Optional.ofNullable(dto).orElse(new RedPacketReportPaginationDto());
    dto.setParticipatorType(ParticipatorTypeEnum.DEALER.getDictCode());
    pageable = Optional.ofNullable(pageable).orElse(PageRequest.of(0, 50));
    Page<RedPacketReportVo> page = this.redPacketReportRepository.findDealerByConditions(pageable, dto);
    if (Objects.nonNull(page) && !CollectionUtils.isEmpty(page.getRecords())) {
      Map<String, String> mapAct = this.findScanActivityMap(page.getRecords());
      Map<String, String> mapTemplate = this.findAgreementMap(page.getRecords());
      for (RedPacketReportVo item : page.getRecords()) {
        if (StringUtils.isNoneBlank(item.getRelateType(), item.getRelateCode())
            && RewardRelateTypeEnum.SCAN_ACTIVITY.getDictCode().equals(item.getRelateType())) {
          item.setRelateActName(mapAct.get(item.getRelateCode()));
        }
        if (StringUtils.isNotBlank(item.getTemplateCode())) {
          item.setTemplateName(mapTemplate.get(item.getTemplateCode()));
        }
      }
    }
    return page;
  }

  /**
   * 获取扫码活动名称map
   *
   * @param list
   * @return k-code,v-name
   */
  private Map<String, String> findScanActivityMap(List<RedPacketReportVo> list) {
    Map<String, String> map = Maps.newHashMap();
    Set<String> actCodeSet =
        list.stream()
            .filter(
                a ->
                    StringUtils.isNoneBlank(a.getRelateCode(), a.getRelateType())
                        && RewardRelateTypeEnum.SCAN_ACTIVITY
                            .getDictCode()
                            .equals(a.getRelateType()))
            .map(RedPacketReportVo::getRelateCode)
            .collect(Collectors.toSet());
    if (CollectionUtils.isEmpty(actCodeSet)) {
      return map;
    }
    List<ScanActivityVo> activityVoList = this.scanActivityVoService.findMainByActCodes(actCodeSet);
    if (CollectionUtils.isEmpty(activityVoList)) {
      return map;
    }
    return activityVoList.stream()
        .filter(a -> StringUtils.isNoneBlank(a.getCode(), a.getName()))
        .collect(Collectors.toMap(ScanActivityVo::getCode, ScanActivityVo::getName, (a, b) -> a));
  }

  /**
   * 获取分利协议名称map
   *
   * @param list
   * @return k-code,v-name
   */
  private Map<String, String> findAgreementMap(List<RedPacketReportVo> list) {
    Map<String, String> map = Maps.newHashMap();
    Set<String> templateCodeSet =
        list.stream()
            .filter(a -> StringUtils.isNotBlank(a.getTemplateCode()))
            .map(RedPacketReportVo::getTemplateCode)
            .collect(Collectors.toSet());
    if (CollectionUtils.isEmpty(templateCodeSet)) {
      return map;
    }
    List<ProfitAgreementTemplate> profitAgreementTemplateList =
        this.profitAgreementTemplateService.findByTemplateCodes(templateCodeSet);
    if (CollectionUtils.isEmpty(profitAgreementTemplateList)) {
      return map;
    }
    return profitAgreementTemplateList.stream()
        .filter(a -> StringUtils.isNoneBlank(a.getTemplateCode(), a.getTemplateName()))
        .collect(
            Collectors.toMap(
                ProfitAgreementTemplate::getTemplateCode,
                ProfitAgreementTemplate::getTemplateName,
                (a, b) -> a));
  }
}
