package com.biz.crm.admin.web.service.internal;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.admin.web.dto.ScanCodeRecordReportPaginationDto;
import com.biz.crm.admin.web.repository.ScanCodeRecordReportAllVoRepository;
import com.biz.crm.admin.web.service.ScanCodeRecordReportAllVoService;
import com.biz.crm.admin.web.vo.ScanCodeRecordReportAllVo;
import com.biz.crm.cps.business.participator.sdk.common.enums.ParticipatorTypeEnum;
import com.biz.crm.cps.business.participator.sdk.dto.DealerConditionDto;
import com.biz.crm.cps.business.participator.sdk.dto.TerminalConditionDto;
import com.biz.crm.cps.business.participator.sdk.service.DealerVoService;
import com.biz.crm.cps.business.participator.sdk.service.TerminalVoService;
import com.biz.crm.cps.business.participator.sdk.vo.DealerVo;
import com.biz.crm.cps.business.participator.sdk.vo.TerminalVo;
import com.biz.crm.mdm.business.user.sdk.dto.UserFeignDto;
import com.biz.crm.mdm.business.user.sdk.service.UserFeignVoService;
import com.biz.crm.mdm.business.user.sdk.vo.UserVo;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * 扫码记录报表接口实现
 *
 * @author sunx
 * @date 2022/3/29
 */
@Service
@Slf4j
public class ScanCodeRecordReportAllVoServiceImpl implements ScanCodeRecordReportAllVoService {

  @Autowired private ScanCodeRecordReportAllVoRepository scanCodeRecordReportAllVoRepository;

  @Autowired private UserFeignVoService userFeignVoService;
  @Autowired
  private TerminalVoService terminalVoService;
  @Autowired
  private DealerVoService dealerVoService;

  @Override
  public Page<ScanCodeRecordReportAllVo> findUserGuideByConditions(
      Pageable pageable, ScanCodeRecordReportPaginationDto dto) {
    pageable = Optional.ofNullable(pageable).orElse(PageRequest.of(0, 50));
    dto = Optional.ofNullable(dto).orElse(new ScanCodeRecordReportPaginationDto());
    dto.setParticipatorType(ParticipatorTypeEnum.USER_GUIDE.getKey());

    this.buildParticipatorQueryInfo(dto);
    Page<ScanCodeRecordReportAllVo> page = this.scanCodeRecordReportAllVoRepository.findAllByConditions(pageable, dto);

    List<ScanCodeRecordReportAllVo> records = page.getRecords();
    Map<String, UserVo> userVoMap = this.findUserVoMapByScanCodeRecords(records);
    for (ScanCodeRecordReportAllVo record : records) {
      record.setRemark(Optional.ofNullable(record.getRemark()).orElse(StringUtils.EMPTY));
      UserVo userVo = userVoMap.get(record.getScanParticipatorCode());
      if (Objects.nonNull(userVo)) {
        record.setScanParticipator(userVo.getFullName());
        record.setPhone(userVo.getUserPhone());
        record.setOrgName(userVo.getOrgName());
      }
    }

    return page;
  }

  /**
   * 查询终端扫码记录
   * @param pageable
   * @param dto
   * @return
   */
  @Override
  public Page<ScanCodeRecordReportAllVo> findTerminalByConditions(Pageable pageable, ScanCodeRecordReportPaginationDto dto) {
    pageable = Optional.ofNullable(pageable).orElse(PageRequest.of(0, 50));
    dto = Optional.ofNullable(dto).orElse(new ScanCodeRecordReportPaginationDto());
    dto.setParticipatorType(ParticipatorTypeEnum.TERMINAL.getKey());
    Set<String> terminalCodes = this.findTerminalCodeByScanCodeRecordReportPaginationDto(dto);
    if(terminalCodes == null){
      return new Page<>();
    }
    dto.setScanParticipatorCodeSet(terminalCodes);
    Page<ScanCodeRecordReportAllVo> page = this.scanCodeRecordReportAllVoRepository.findAllByConditions(pageable, dto);
    List<ScanCodeRecordReportAllVo> records = page.getRecords();
    Map<String, TerminalVo> terminalVoMap = this.findTerminalVoMapByScanCodeRecords(records);
    for (ScanCodeRecordReportAllVo record : records) {
      TerminalVo terminalVo = terminalVoMap.get(record.getScanParticipatorCode());
      if(Objects.nonNull(terminalVo)){
        record.setTerminalType(terminalVo.getTerminalType());
        record.setScanParticipator(terminalVo.getTerminalName());
        record.setOrgName(terminalVo.getOrganization());
        record.setCustomerOrgName(terminalVo.getCustomerOrganization());
        record.setChannelCode(terminalVo.getChannel());
        record.setParticipatorTagDescription(terminalVo.getTagDescription());
      }
    }
    return page;
  }

  /**
   * 查询经销商扫码记录
   * @param pageable
   * @param dto
   * @return
   */
  @Override
  public Page<ScanCodeRecordReportAllVo> findDealerByConditions(Pageable pageable, ScanCodeRecordReportPaginationDto dto) {
    pageable = Optional.ofNullable(pageable).orElse(PageRequest.of(0, 50));
    dto = Optional.ofNullable(dto).orElse(new ScanCodeRecordReportPaginationDto());
    dto.setParticipatorType(ParticipatorTypeEnum.DEALER.getKey());

    Set<String> customerCodes = this.findCustomerCodesByScanCodeRecordReportPaginationDto(dto);
    if(customerCodes == null){
      return new Page<>();
    }
    dto.setScanParticipatorCodeSet(customerCodes);
    Page<ScanCodeRecordReportAllVo> page = this.scanCodeRecordReportAllVoRepository.findAllByConditions(pageable, dto);
    List<ScanCodeRecordReportAllVo> records = page.getRecords();
    Map<String, DealerVo> dealerVoMap = this.findDealerVoMapByScanCodeRecords(records);
    for (ScanCodeRecordReportAllVo record : records) {
      DealerVo dealerVo = dealerVoMap.get(record.getScanParticipatorCode());
      if(Objects.nonNull(dealerVo)){
        record.setDealerType(dealerVo.getCustomerType());
        record.setScanParticipator(dealerVo.getCustomerName());
        record.setOrgName(dealerVo.getOrgName());
        record.setCustomerOrgName(dealerVo.getCustomerOrgName());
        record.setChannelCode(dealerVo.getChannelCode());
      }
    }
    return page;
  }

  /**
   * 根据扫码记录查询参数查询终端编码,返回null表示有查询条件但没有符合条件的数据
   * @param dto
   * @return
   */
  private Set<String> findTerminalCodeByScanCodeRecordReportPaginationDto(ScanCodeRecordReportPaginationDto dto){
    if(StringUtils.isAllBlank(dto.getScanParticipator(),dto.getScanParticipatorCode(),
        dto.getChannelCode(),dto.getTerminalType(),
        dto.getTerminalTagDescription(),dto.getOrgName(),dto.getCustomerOrgName())){
      return Sets.newHashSet();
    }

    TerminalConditionDto terminalConditionDto = new TerminalConditionDto();
    terminalConditionDto.setTerminalCode(dto.getScanParticipatorCode());
    terminalConditionDto.setTerminalName(dto.getScanParticipator());
    terminalConditionDto.setChannel(dto.getChannelCode());
    terminalConditionDto.setTerminalType(dto.getTerminalType());
    terminalConditionDto.setTagDescription(dto.getTerminalTagDescription());
    terminalConditionDto.setOrganizationName(dto.getOrgName());
    terminalConditionDto.setCustomerOrganizationName(dto.getCustomerOrgName());

    Set<String> terminalCodes = terminalVoService.findTerminalCodeByConditions(terminalConditionDto);
    return CollectionUtils.isEmpty(terminalCodes) ? null : terminalCodes;
  }

  /**
   * 构建扫码参与者查询信息
   * @param dto
   */
  private void buildParticipatorQueryInfo(ScanCodeRecordReportPaginationDto dto){
    String participatorType = dto.getParticipatorType();
    Set<String> scanParticipatorCodeSet = Sets.newHashSet();
    if(ParticipatorTypeEnum.USER_GUIDE.getKey().equals(participatorType)){
      if (!StringUtils.isAllBlank(dto.getScanParticipator(), dto.getPhone(), dto.getOrgName())) {
        dto.setScanParticipatorCode(StringUtils.EMPTY);
        // 获取导购信息
        UserFeignDto userFeignDto = new UserFeignDto();
        userFeignDto.setUserName(dto.getScanParticipatorCode());
        userFeignDto.setFullName(dto.getScanParticipator());
        userFeignDto.setUserPhone(dto.getPhone());
        userFeignDto.setOrgName(dto.getOrgName());
        Set<String> userNameSet = this.userFeignVoService.findUserNamesByUserDto(userFeignDto);
        if (CollectionUtils.isEmpty(userNameSet)) {
          scanParticipatorCodeSet = Sets.newHashSet("-1");
        } else {
          scanParticipatorCodeSet = userNameSet;
        }
      }
    }
    dto.setScanParticipatorCodeSet(scanParticipatorCodeSet);
  }


  /**
   * 根据扫码记录查询参数查询经销商编码,返回null表示有查询条件但没有符合条件的数据
   * @param dto
   * @return
   */
  private Set<String> findCustomerCodesByScanCodeRecordReportPaginationDto(ScanCodeRecordReportPaginationDto dto){
    if(StringUtils.isAllBlank(dto.getScanParticipator(),
        dto.getScanParticipatorCode(),
        dto.getOrgName(),
        dto.getCustomerOrgName(),dto.getChannelCode())){
      return Sets.newHashSet();
    }
    DealerConditionDto dealerConditionDto = new DealerConditionDto();
    dealerConditionDto.setCustomerCode(dto.getScanParticipatorCode());
    dealerConditionDto.setCustomerName(dto.getScanParticipator());
    dealerConditionDto.setOrgName(dto.getOrgName());
    dealerConditionDto.setCustomerOrgName(dto.getCustomerOrgName());
    dealerConditionDto.setChannelCode(dto.getChannelCode());

    Set<String> customerCodes = dealerVoService.findCustomerCodeByConditions(dealerConditionDto);
    return CollectionUtils.isEmpty(customerCodes) ? null : customerCodes;
  }

  /**
   * 通过终端扫码记录查询终端信息
   * @param records
   */
  private Map<String,TerminalVo> findTerminalVoMapByScanCodeRecords(List<ScanCodeRecordReportAllVo> records){
    if(CollectionUtils.isEmpty(records)){
      return Maps.newHashMap();
    }
    List<String> terminalCodes = records.stream().filter(o -> ParticipatorTypeEnum.TERMINAL.getKey().equals(o.getParticipatorType())).map(ScanCodeRecordReportAllVo::getScanParticipatorCode).collect(Collectors.toList());
    Map<String,TerminalVo> terminalVoMap = Maps.newHashMap();
    List<TerminalVo> terminalVos = terminalVoService.findByTerminalCodes(terminalCodes);
    if(!CollectionUtils.isEmpty(terminalVos)){
      terminalVoMap = terminalVos.stream().collect(Collectors.toMap(TerminalVo::getTerminalCode, Function.identity(), (a, b) -> a));
    }
    return terminalVoMap;
  }

  /**
   * 通过经销商扫码记录查询经销商信息
   * @param records
   */
  private Map<String,DealerVo> findDealerVoMapByScanCodeRecords(List<ScanCodeRecordReportAllVo> records){
    Map<String,DealerVo> dealerVoMap = Maps.newHashMap();
    if(CollectionUtils.isEmpty(records)){
      return dealerVoMap;
    }
    List<String> customerCodes = records.stream().filter(o -> ParticipatorTypeEnum.DEALER.getKey().equals(o.getParticipatorType())).map(ScanCodeRecordReportAllVo::getScanParticipatorCode).collect(Collectors.toList());
    List<DealerVo> dealerVos = dealerVoService.findByCustomerCodes(customerCodes);

    if(!CollectionUtils.isEmpty(dealerVos)){
      dealerVoMap = dealerVos.stream().collect(Collectors.toMap(DealerVo::getCustomerCode, Function.identity(), (a, b) -> a));
    }
    return dealerVoMap;
  }

  /**
   * 通过导购扫码记录查询导购信息
   * @param records
   * @return
   */
  private Map<String,UserVo> findUserVoMapByScanCodeRecords(List<ScanCodeRecordReportAllVo> records){
    Map<String,UserVo> userVoMap = Maps.newHashMap();
    if(CollectionUtils.isEmpty(records)){
      return userVoMap;
    }
    List<String> userNames = records.stream().filter(o -> ParticipatorTypeEnum.USER_GUIDE.getKey().equals(o.getParticipatorType())).map(ScanCodeRecordReportAllVo::getScanParticipatorCode).collect(Collectors.toList());
    List<UserVo> userVos = userFeignVoService.findByUserNames(userNames);
    if(!CollectionUtils.isEmpty(userVos)){
      userVoMap = userVos.stream().collect(Collectors.toMap(UserVo::getUserName, Function.identity(), (a, b) -> a));
    }
    return userVoMap;
  }
}
