package com.biz.crm.admin.web.service.internal;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.admin.web.dto.ScanCodeRecordReportDto;
import com.biz.crm.admin.web.repository.ScanCodeRecordReportRepository;
import com.biz.crm.admin.web.service.ScanCodeRecordReportService;
import com.biz.crm.admin.web.vo.AssociateBarCodeReportVo;
import com.biz.crm.admin.web.vo.BarCodeTraceReportVo;
import com.biz.crm.admin.web.vo.ScanCodeRecordReportVo;
import com.biz.crm.admin.web.vo.ScanCodeRecordTraceVo;
import com.biz.crm.cps.business.participator.sdk.common.enums.ParticipatorTypeEnum;
import com.biz.crm.cps.business.participator.sdk.vo.ParticipatorTagVo;
import com.biz.crm.cps.business.policy.quantify.fiscal.local.entity.QuantifyPolicy;
import com.biz.crm.cps.business.policy.quantify.fiscal.local.entity.QuantifyRange;
import com.biz.crm.cps.business.policy.quantify.fiscal.local.entity.QuantifyTask;
import com.biz.crm.cps.business.policy.quantify.fiscal.local.service.QuantifyPolicyService;
import com.biz.crm.cps.business.policy.quantify.fiscal.local.service.QuantifyTaskService;
import com.biz.crm.cps.business.product.sdk.common.constant.MaterialDimensionConstant;
import com.biz.crm.cps.business.product.sdk.dto.MaterialStandardUnitDto;
import com.biz.crm.cps.business.product.sdk.service.MaterialStandardUnitVoService;
import com.biz.crm.cps.business.product.sdk.service.MaterialVoService;
import com.biz.crm.cps.business.product.sdk.vo.MaterialDimensionVo;
import com.biz.crm.cps.business.product.sdk.vo.MaterialStandardUnitVo;
import com.biz.crm.cps.external.barcode.sdk.common.enums.BarCodeTypeEnum;
import com.biz.crm.cps.external.barcode.sdk.common.enums.ScanCodeExceptionEnum;
import com.biz.crm.cps.external.barcode.sdk.common.enums.ScanTypeEnum;
import com.biz.crm.cps.external.barcode.sdk.service.BarCodeVoService;
import com.biz.crm.cps.external.barcode.sdk.vo.BarCodeVo;
import com.biz.crm.mdm.business.user.sdk.service.UserFeignVoService;
import com.biz.crm.mdm.business.user.sdk.vo.UserVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.google.common.collect.Lists;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import javax.annotation.Resource;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * 扫码记录报表service实现类
 *
 * @author songjingen
 */
@Service
public class ScanCodeRecordReportServiceImpl implements ScanCodeRecordReportService {

  @Resource
  private ScanCodeRecordReportRepository scanCodeRecordReportRepository;

  @Autowired
  private QuantifyTaskService quantifyTaskService;

  @Autowired
  private QuantifyPolicyService quantifyPolicyService;

  @Autowired
  private MaterialVoService materialVoService;

  @Autowired
  private MaterialStandardUnitVoService materialStandardUnitVoService;

  @Autowired
  private BarCodeVoService barCodeVoService;

  @Autowired
  private NebulaToolkitService nebulaToolkitService;
  @Autowired
  private UserFeignVoService userFeignVoService;

  /**
   * 通过查询条件查询当前参与者的扫码记录分页信息
   *
   * @param pageable
   * @param dto
   * @return
   */
  @Override
  public Page<ScanCodeRecordReportVo> findByConditions(Pageable pageable, ScanCodeRecordReportDto dto) {
    ObjectUtils.defaultIfNull(pageable, PageRequest.of(0, 50));
    if (dto == null) {
      dto = new ScanCodeRecordReportDto();
    }
    Page<ScanCodeRecordReportVo> page = scanCodeRecordReportRepository.findByConditions(pageable, dto);
    //转成前端所需要的字段
    List<ScanCodeRecordReportVo> records = page.getRecords();
    if (!CollectionUtils.isEmpty(records)) {
      buildScanCodeList(records);
      buildTemplateCodeList(records);
      //查询 协议模板
      page.getRecords().forEach(item -> {
        if (!CollectionUtils.isEmpty(item.getParticipatorTagVos())) {
          item.setTagName(item.getParticipatorTagVos()
                  .stream()
                  .map(ParticipatorTagVo::getTagDescription)
                  .collect(Collectors.joining("，")));
        }
      });
    }
    return page;
  }

  @Override
  public Page<ScanCodeRecordReportVo> findByQuantifyTaskBusinessCode(Pageable pageable, String quantifyTaskId) {
    /**
     * step1：校验pageable，quantifyTaskId
     * step2：查询包量任务，政策并校验
     * step3：根据包量政策获取产品维度，从物料处获取物料编码列表(如有配置为选择宣布产品，则无需查询物料编码)
     * step4：构造扫码查询DTO，查询扫码记录
     * step5：根据查询到的结果，取出物料编码获取物料名称，和标箱转换信息
     */

    ObjectUtils.defaultIfNull(pageable, PageRequest.of(0, 50));
    if (StringUtils.isBlank(quantifyTaskId)) {
      return null;
    }
    //查询包量任务
    QuantifyTask quantifyTask = quantifyTaskService.findById(quantifyTaskId);
    if (!ObjectUtils.allNotNull(quantifyTask)) {
      return null;
    }
    //查询包量政策
    QuantifyPolicy quantifyPolicy = quantifyPolicyService.findDetailByTemplateCode(quantifyTask.getTemplateId());
    if (!ObjectUtils.allNotNull(quantifyTask)) {
      return null;
    }
    List<String> dimensionCodes = Lists.newArrayList();
    if (!quantifyPolicy.getDimensionFlag().equals(MaterialDimensionConstant.DIMENSION_ALL)) {
      //如果政策下所有配置均没有配置为所有产品，则取商品维度去获取物料编码集合
      quantifyPolicy.getQuantifyConfigurations().forEach(configuration -> {
        List<String> dimensions = configuration.getQuantifyRanges()
                .stream().map(QuantifyRange::getSpecialCode).collect(Collectors.toList());
        List<String> materialCodes = materialVoService.findMaterialCodeByDimensionCodesAndDimensionType(dimensions, quantifyPolicy.getDimensionFlag());
        if (!CollectionUtils.isEmpty(materialCodes)) {
          dimensionCodes.addAll(materialCodes);
        }
      });
    }
    if (CollectionUtils.isEmpty(dimensionCodes)) {
      return null;
    }
    //构造扫码查询DTO
    ScanCodeRecordReportDto scanCodeRecordReportDto = new ScanCodeRecordReportDto();
    scanCodeRecordReportDto.setParticipatorType(ParticipatorTypeEnum.TERMINAL.getKey());
    scanCodeRecordReportDto.setScanParticipatorCode(quantifyTask.getTerminalCode());
    scanCodeRecordReportDto.setCreateTimeStart(quantifyTask.getTaskStartTime());
    scanCodeRecordReportDto.setCreateTimeEnd(quantifyTask.getTaskEndTime());
    scanCodeRecordReportDto.setProductCodes(dimensionCodes);
    //查询扫码记录
    Page<ScanCodeRecordReportVo> page = this.scanCodeRecordReportRepository.findByConditions(pageable, scanCodeRecordReportDto);
    if (CollectionUtils.isEmpty(page.getRecords())) {
      //未查询到扫码记录，直接返回
      return page;
    }
    buildScanCodeList(page.getRecords());
    //根据扫码记录构造标箱信息查询DTO
    List<MaterialStandardUnitDto> materialStandardUnitDtos = Lists.newArrayList();
    page.getRecords().forEach(scanCodeRecordVo -> {
      MaterialStandardUnitDto materialStandardUnitDto = new MaterialStandardUnitDto();
      materialStandardUnitDto.setMaterialCode(scanCodeRecordVo.getProductCode());
      materialStandardUnitDtos.add(materialStandardUnitDto);
    });
    //查询标箱信息
    List<MaterialStandardUnitVo> unitVos = materialStandardUnitVoService.findByMaterialCodeAndBarCodeType(materialStandardUnitDtos);
    if (CollectionUtils.isEmpty(unitVos)) {
      //未查询到标箱信息，直接返回
      return page;
    }
    //转换标箱信息列表为物料编码-标箱信息vo的map，方便进行取值匹配
    Map<String, MaterialStandardUnitVo> unitVoMap = unitVos.stream().collect(Collectors.toMap(MaterialStandardUnitVo::getMaterialCode, Function.identity()));
    page.getRecords().forEach(scanCodeRecordVo -> {
      MaterialStandardUnitVo unitVo = unitVoMap.get(scanCodeRecordVo.getProductCode());
      if (ObjectUtils.allNotNull(unitVo)) {
        //填充物料名称和标箱信息
        scanCodeRecordVo.setProductName(unitVo.getMaterialName());
        scanCodeRecordVo.setStandardUnit(unitVo.getStandardUnit());
        scanCodeRecordVo.setStandardUnitDesc(unitVo.getStandardUnitDesc());
      }
    });
    return page;
  }

  /**
   * 根据扫码编码查询
   *
   * @param recordCode
   * @return
   */
  @Override
  public ScanCodeRecordReportVo findByRecordCodeAndParticipatorCode(String recordCode, String participatorCode) {
    if (StringUtils.isBlank(recordCode)) {
      return null;
    }
    return scanCodeRecordReportRepository.findByRecordCodeAndParticipatorCode(recordCode, participatorCode);
  }

  /**
   * 码回溯报表
   * step1:获取码基本信息
   * step2:获取关联扫码记录
   * step3:获取关联码信息，以及关联码扫码记录进行组装
   *
   * @param barCode 码编码
   * @return 码溯源VO
   */
  @Override
  public BarCodeTraceReportVo findBarCodeTraceByBarCode(String barCode) {
    if (StringUtils.isBlank(barCode)) {
      return null;
    }
    //======= STEP1 =======
    BarCodeVo barCodeVo = this.barCodeVoService.findByBarCode(barCode);
    if (ObjectUtils.isEmpty(barCodeVo)) {
      return null;
    }
    BarCodeTraceReportVo barCodeTraceReportVo = this.nebulaToolkitService.copyObjectByWhiteList(barCodeVo, BarCodeTraceReportVo.class, HashSet.class, ArrayList.class);
    MaterialDimensionVo product = materialVoService.findByMaterialCode(barCodeVo.getProductCode());
    barCodeTraceReportVo.setProductName(product == null ? StringUtils.EMPTY : product.getMaterialName());
    //======= STEP2 =======
    ScanCodeRecordReportDto dto = new ScanCodeRecordReportDto();
    dto.setBarCode(barCode);
    List<ScanCodeRecordReportVo> recordReportVos = this.scanCodeRecordReportRepository.findByConditions(dto);
    List<ScanCodeRecordTraceVo> scanCodeRecordTraceVos = Lists.newArrayList();
    if (!CollectionUtils.isEmpty(recordReportVos)) {
      //根据创建时间进行排序
      recordReportVos.stream().sorted(Comparator.comparing(ScanCodeRecordReportVo::getCreateTime)).collect(Collectors.toList());
      for (int i = 0; i < recordReportVos.size(); i++) {
        ScanCodeRecordReportVo reportVo = recordReportVos.get(i);
        if(StringUtils.isNotBlank(reportVo.getScanCodeException()) && !ScanCodeExceptionEnum.NO.getCode().equals(reportVo.getScanCodeException())){
          continue;
        }
        ScanCodeRecordTraceVo traceVo = this.nebulaToolkitService.copyObjectByWhiteList(reportVo, ScanCodeRecordTraceVo.class, HashSet.class, ArrayList.class);
        //当扫码类型为入库时，流出方为当前扫码用户，流入方位空
        if (traceVo.getScanType().equals(ScanTypeEnum.ENTER_WAREHOUSE.getCode())) {
          this.setOutFlowParams(traceVo, reportVo);
        }
        //当扫码类型为出库时，流出方为上一个扫码用户，流入方为当前扫码用户
        else if (traceVo.getScanType().equals(ScanTypeEnum.OUT_WAREHOUSE.getCode())) {
          //设置扫码流入方信息
          this.setInFlowParams(traceVo, reportVo);
          //设置扫码流出方
          if (i > 0 && ObjectUtils.isNotEmpty(recordReportVos.get(i))) {
            ScanCodeRecordReportVo lastReportVo = recordReportVos.get(i - 1);
            this.setOutFlowParams(traceVo, lastReportVo);
          }
        }
        scanCodeRecordTraceVos.add(traceVo);
      }
    }
    barCodeTraceReportVo.setScanCodeRecordTraceVos(scanCodeRecordTraceVos);
    //======= STEP3 =======
    List<AssociateBarCodeReportVo> associateVos  = this.findAssociateByBarCode(barCode);
    List<String> barCodes = associateVos.stream().map(AssociateBarCodeReportVo::getBarCode).collect(Collectors.toList());
    if (!CollectionUtils.isEmpty(barCodes)) {
      ScanCodeRecordReportDto reportDto = new ScanCodeRecordReportDto();
      reportDto.setBarCodes(barCodes);
      //获取关联码的扫码列表
      List<ScanCodeRecordReportVo> scanCodeRecordReportVos = this.scanCodeRecordReportRepository.findByConditions(reportDto);
      if (!CollectionUtils.isEmpty(scanCodeRecordReportVos)) {
        //转换为map key --barCode value --扫码记录列表
        Map<String, List<ScanCodeRecordReportVo>> map = scanCodeRecordReportVos.stream()
            .filter(r -> StringUtils.isBlank(r.getScanCodeException()) || ScanCodeExceptionEnum.NO.getCode().equals(r.getScanCodeException()))
            .collect(Collectors.groupingBy(ScanCodeRecordReportVo::getBarCode));

        associateVos.forEach(associateBarCodeReportVo -> {
          //转换扫码记录编码和remake
          List<ScanCodeRecordReportVo> scanLists = map.get(associateBarCodeReportVo.getBarCode());
          if (!CollectionUtils.isEmpty(scanLists)) {
            StringBuilder recordBuilder = new StringBuilder();
            StringBuilder remarkBuilder = new StringBuilder();
            for (ScanCodeRecordReportVo reportVo : scanLists) {
              recordBuilder.append(",").append(reportVo.getRecordCode());
              if(StringUtils.isNotBlank(reportVo.getRemark())){
                remarkBuilder.append(",").append(reportVo.getRemark());
              }
            }
            associateBarCodeReportVo.setAssociateRecords(recordBuilder.substring(1).toString());
            associateBarCodeReportVo.setAssociateRemarks(remarkBuilder.length() == 0 ? StringUtils.EMPTY : remarkBuilder.substring(1).toString());
          }
        });
      }
    }
    barCodeTraceReportVo.setAssociateBarCodeReportVos(associateVos);
    return barCodeTraceReportVo;
  }

  /**
   * 组织扫码数据的扫码人信息
   *
   * @param records
   */
  private void buildScanCodeList(List<ScanCodeRecordReportVo> records) {
    records.stream().forEach(scanCodeRecordReportVo -> {
      if (ParticipatorTypeEnum.DEALER.getKey().equals(scanCodeRecordReportVo.getParticipatorType())) {
        scanCodeRecordReportVo.setChannelName(scanCodeRecordReportVo.getDealerChannelName());
        scanCodeRecordReportVo.setOrgName(scanCodeRecordReportVo.getDealerOrgName());
        scanCodeRecordReportVo.setCustomerCode(scanCodeRecordReportVo.getDealerCode());
        scanCodeRecordReportVo.setCustomerName(scanCodeRecordReportVo.getDealerName());
      }
      if (ParticipatorTypeEnum.TERMINAL.getKey().equals(scanCodeRecordReportVo.getParticipatorType())) {
        scanCodeRecordReportVo.setChannelName(scanCodeRecordReportVo.getTerminalChannelName());
        scanCodeRecordReportVo.setOrgName(scanCodeRecordReportVo.getTerminalOrgName());
        scanCodeRecordReportVo.setCustomerCode(scanCodeRecordReportVo.getTerminalCode());
        scanCodeRecordReportVo.setCustomerName(scanCodeRecordReportVo.getTerminalName());
      }
      if (ParticipatorTypeEnum.CONSUMER.getKey().equals(scanCodeRecordReportVo.getParticipatorType())) {
        scanCodeRecordReportVo.setCustomerCode(scanCodeRecordReportVo.getConsumerExternalId());
        scanCodeRecordReportVo.setCustomerName(scanCodeRecordReportVo.getConsumerNickname());
      }
    });
  }


  private void buildTemplateCodeList(List<ScanCodeRecordReportVo> records) {
    List<String> collect = records.stream().map(ScanCodeRecordReportVo::getRecordCode).collect(Collectors.toList());
    List<ScanCodeRecordReportVo> templateCodeByRecordCodes = this.scanCodeRecordReportRepository.findTemplateCodeByRecordCodes(collect);
    if (!CollectionUtils.isEmpty(collect)) {
      Map<String, String> map = templateCodeByRecordCodes.stream().distinct().collect(Collectors.toMap(ScanCodeRecordReportVo::getRecordCode, ScanCodeRecordReportVo::getTemplateCode));
      records.forEach(scanCodeRecordReportVo -> {
        scanCodeRecordReportVo.setTemplateCode(map.get(scanCodeRecordReportVo.getRecordCode()));
      });
    }
  }

  /**
   * 设置扫码流入方信息
   *
   * @param traceVo
   * @param reportVo
   */
  private void setInFlowParams(ScanCodeRecordTraceVo traceVo, ScanCodeRecordReportVo reportVo) {
    traceVo.setCodeInFlowType(reportVo.getParticipatorType());
    ParticipatorTypeEnum eType = ParticipatorTypeEnum.getByKey(traceVo.getParticipatorType());
    switch (eType) {
      case DEALER: {
        traceVo.setCodeInFlowCode(reportVo.getDealerCode());
        traceVo.setCodeInFlowName(reportVo.getDealerName());
        break;
      }
      case TERMINAL: {
        traceVo.setCodeInFlowCode(reportVo.getTerminalCode());
        traceVo.setCodeInFlowName(reportVo.getTerminalName());
        break;
      }
      case USER_GUIDE:{
        String scanParticipatorCode = traceVo.getScanParticipatorCode();
        List<UserVo> userVos = userFeignVoService.findByUserNames(Arrays.asList(scanParticipatorCode));
        if(CollectionUtils.isEmpty(userVos)){
          break;
        }
        UserVo userVo = userVos.get(0);
        traceVo.setCodeInFlowCode(userVo.getUserCode());
        traceVo.setCodeInFlowName(userVo.getFullName());
        break;
      }
      default: {
        traceVo.setCodeInFlowCode(reportVo.getCustomerCode());
        traceVo.setCodeInFlowName(reportVo.getCustomerName());
      }
    }
  }

  /**
   * 设置扫码流出方信息
   *
   * @param traceVo
   * @param reportVo
   */
  private void setOutFlowParams(ScanCodeRecordTraceVo traceVo, ScanCodeRecordReportVo reportVo) {
    traceVo.setCodeOutFlowType(reportVo.getParticipatorType());
    ParticipatorTypeEnum eType = ParticipatorTypeEnum.getByKey(traceVo.getParticipatorType());
    switch (eType) {
      case DEALER: {
        traceVo.setCodeOutFlowCode(traceVo.getDealerCode());
        traceVo.setCodeOutFlowName(reportVo.getDealerName());
        break;
      }
      case TERMINAL: {
        traceVo.setCodeOutFlowCode(traceVo.getTerminalCode());
        traceVo.setCodeOutFlowName(reportVo.getTerminalName());
        break;
      }
      case USER_GUIDE:{
        String scanParticipatorCode = traceVo.getScanParticipatorCode();
        List<UserVo> userVos = userFeignVoService.findByUserNames(Arrays.asList(scanParticipatorCode));
        if(CollectionUtils.isEmpty(userVos)){
          break;
        }
        UserVo userVo = userVos.get(0);
        traceVo.setCodeOutFlowCode(userVo.getUserCode());
        traceVo.setCodeOutFlowName(userVo.getFullName());
        break;
      }
      default: {
        traceVo.setCodeOutFlowCode(traceVo.getCustomerCode());
        traceVo.setCodeOutFlowName(reportVo.getCustomerName());
      }
    }
  }

  /**
   * 根据码编码查询所有的关联码列表
   *
   * @param barCode 码编码
   * @return 关联码列表
   */
  private List<AssociateBarCodeReportVo> findAssociateByBarCode(String barCode) {
    List<AssociateBarCodeReportVo> associateVos = Lists.newArrayList();
    BarCodeVo codeVo = this.barCodeVoService.findByBarCode(barCode);
    associateVos.add(this.nebulaToolkitService.copyObjectByWhiteList(codeVo, AssociateBarCodeReportVo.class, HashSet.class, ArrayList.class));
    List<BarCodeVo> childrenBarCodes = this.barCodeVoService.findByParentBarCode(barCode);
    if (!CollectionUtils.isEmpty(childrenBarCodes)) {
      associateVos.addAll(this.nebulaToolkitService.copyCollectionByWhiteList(childrenBarCodes, BarCodeVo.class, AssociateBarCodeReportVo.class, HashSet.class, ArrayList.class));
    }
    List<String> codes = Arrays.asList(codeVo.getFlatAncestors().split(","));
    //如果是瓶码，则取第二个父级码为箱码
    if (codeVo.getBarCodeType().equals(BarCodeTypeEnum.BOTTLE.getKey())) {
      AssociateBarCodeReportVo associateVo = new AssociateBarCodeReportVo();
      associateVo.setBarCode(codes.get(1));
      associateVo.setBarCodeType(BarCodeTypeEnum.BOX.getKey());
      associateVos.add(associateVo);
    }
    //如果是盖内码，则取第二个父级码为箱码，第三个父级码为瓶码
    else if (codeVo.getBarCodeType().equals(BarCodeTypeEnum.COVER_CODE.getKey())) {
      AssociateBarCodeReportVo associateVo = new AssociateBarCodeReportVo();
      associateVo.setBarCode(codes.get(1));
      associateVo.setBarCodeType(BarCodeTypeEnum.BOX.getKey());
      associateVos.add(associateVo);
      AssociateBarCodeReportVo bottleVo = new AssociateBarCodeReportVo();
      bottleVo.setBarCode(codes.get(2));
      bottleVo.setBarCodeType(BarCodeTypeEnum.BOTTLE.getKey());
      associateVos.add(bottleVo);
    }
    return associateVos;
  }

}
