package com.biz.crm.cps.mobile.terminal.interceptor;

import com.biz.crm.business.common.sdk.model.LoginUserDetailsForCPS;
import com.biz.crm.business.common.sdk.service.LoginUserService;
import com.bizunited.nebula.common.controller.model.ResponseCode;
import com.bizunited.nebula.common.controller.model.ResponseModel;
import com.bizunited.nebula.security.local.service.handle.HandleOutPut;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;
import org.springframework.web.filter.OncePerRequestFilter;

import javax.servlet.FilterChain;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;

/**
 * 从请求头能拿到web请求信息，需要通过此拦截器设置SecurityContextHolder上下文信息
 *
 * @author songjingen
 * @date 2022/4/1
 */
@Slf4j
@Component
public class CpsSecurityAuthorizationFilter extends OncePerRequestFilter implements HandleOutPut {

  @Autowired
  private LoginUserService loginUserService;

  @Value("${security.ignoreUrls:}")
  private String ignoreUrls;


  /**
   * 1、判断当前用户是否有上下文信息，并且当前请求是否在白名单中
   * 1.1、有上下文信息且不在白名单中-验证当前上下文信息和header的userType类型是否匹配，不匹配直接抛异常（当前用户已失效，请重新登录！）
   * 否-正常放行
   *
   * @param request
   * @param response
   * @param filterChain
   * @throws ServletException
   * @throws IOException
   */
  @Override
  protected void doFilterInternal(HttpServletRequest request, HttpServletResponse response, FilterChain filterChain) throws ServletException, IOException {
    // 1、======
    LoginUserDetailsForCPS loginDetails = this.loginUserService.getLoginDetails(LoginUserDetailsForCPS.class);
    // 1.1、======
    if (ObjectUtils.isNotEmpty(loginDetails) && this.isTrueIgnoreUrls(request.getServletPath())) {
      String userType = request.getHeader("userType");
      if (StringUtils.isBlank(userType)) {
        log.error("SecurityAuthorizationFilter Error：{}", ResponseCode.E401.getDesc());
        ResponseModel responseModel = new ResponseModel(System.currentTimeMillis(), null, ResponseCode.E401, null);
        this.writeResponse(response, responseModel);
        return;
      }
      if (!StringUtils.equals(userType, loginDetails.getUsertype())) {
        log.error("SecurityAuthorizationFilter Error：{}", ResponseCode.E601.getDesc());
        ResponseModel responseModel = new ResponseModel(System.currentTimeMillis(), null, ResponseCode.E601, null);
        this.writeResponse(response, responseModel);
        return;
      }
    }
    filterChain.doFilter(request, response);
  }

  /**
   * 验证请求路径是否在白名单中，存在返回false，反之则返回true
   *
   * @param servletPath
   * @return
   */
  protected boolean isTrueIgnoreUrls(String servletPath) {
    if (StringUtils.isBlank(this.ignoreUrls)) {
      return true;
    }
    String firstContextPath = servletPath.substring(0, servletPath.lastIndexOf("/"));
    String lastContextPath = servletPath.substring(servletPath.lastIndexOf("/") + 1);
    String[] split = this.ignoreUrls.split(",");
    for (int i = 0; i < split.length; i++) {
      String url = split[i];
      String firstUrl = url.substring(0, url.lastIndexOf("/"));
      String lastUrl = url.substring(url.lastIndexOf("/") + 1);
      if (firstContextPath.equals(firstUrl)) {
        if (lastUrl.equals("**")) {
          return false;
        }
        if (lastContextPath.equals(lastUrl)) {
          return false;
        }
      }
    }
    return true;
  }
}
