package com.biz.crm.cps.mobile.terminal.service.internal;

import com.biz.crm.cps.business.activity.scan.local.entity.ScanActivity;
import com.biz.crm.cps.business.activity.scan.local.service.ScanActivityService;
import com.biz.crm.cps.business.agreement.local.entity.ProfitAgreementTemplate;
import com.biz.crm.cps.business.agreement.local.service.ProfitAgreementTemplateService;
import com.biz.crm.cps.business.participator.sdk.common.enums.ParticipatorTypeEnum;
import com.biz.crm.cps.business.participator.sdk.service.DealerVoService;
import com.biz.crm.cps.business.participator.sdk.service.TerminalVoService;
import com.biz.crm.cps.business.participator.sdk.vo.DealerVo;
import com.biz.crm.cps.business.participator.sdk.vo.TerminalVo;
import com.biz.crm.cps.business.policy.scan.sdk.common.enums.RebateTypeEnum;
import com.biz.crm.cps.business.reward.sdk.common.enums.RewardRelateTypeEnum;
import com.biz.crm.cps.business.reward.sdk.common.enums.RewardTypeEnum;
import com.biz.crm.cps.mobile.terminal.repository.RewardReportVoRepository;
import com.biz.crm.cps.mobile.terminal.service.RewardReportVoService;
import com.biz.crm.cps.mobile.terminal.vo.RewardReportVo;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

/**
 * 奖励报表相关接口实现
 *
 * @author sunx
 * @date 2022/3/28
 */
@Service
@Slf4j
public class RewardReportVoServiceImpl implements RewardReportVoService {

  @Autowired private RewardReportVoRepository rewardReportVoRepository;

  @Autowired private ProfitAgreementTemplateService profitAgreementTemplateService;

  @Autowired private ScanActivityService scanActivityService;

  @Autowired private DealerVoService dealerVoService;

  @Autowired private TerminalVoService terminalVoService;

  @Override
  public List<RewardReportVo> findByRecordCode(String recordCode) {
    if (StringUtils.isBlank(recordCode)) {
      return Lists.newLinkedList();
    }
    List<RewardReportVo> list = this.rewardReportVoRepository.findByRecordCode(recordCode);
    if (CollectionUtils.isEmpty(list)) {
      return Lists.newLinkedList();
    }
    // k-participatorType,v-set<participatorCode>
    Map<String, Set<String>> map =
        list.stream()
            .filter(
                a ->
                    StringUtils.isNoneBlank(
                            a.getRewardType(), a.getParticipatorType(), a.getParticipatorCode())
                        && RewardTypeEnum.COST.getDictCode().equals(a.getRewardType()))
            .collect(
                Collectors.groupingBy(
                    RewardReportVo::getParticipatorType,
                    Collectors.mapping(RewardReportVo::getParticipatorCode, Collectors.toSet())));
    Set<String> customerCodeSet = map.get(ParticipatorTypeEnum.DEALER.getDictCode());
    Set<String> terminalCodeSet = map.get(ParticipatorTypeEnum.TERMINAL.getDictCode());
    // k- customerCode,v-customerName
    Map<String, String> mapCustomer = Maps.newHashMap();
    // k- terminalCode,v-terminalName
    Map<String, String> mapTerminal = Maps.newHashMap();
    if (!CollectionUtils.isEmpty(customerCodeSet)) {
      List<DealerVo> dealerVoList =
          this.dealerVoService.findByCustomerCodes(Lists.newArrayList(customerCodeSet));
      if (!CollectionUtils.isEmpty(dealerVoList)) {
        mapCustomer =
            dealerVoList.stream()
                .filter(a -> StringUtils.isNoneBlank(a.getCustomerCode(), a.getCustomerName()))
                .collect(
                    Collectors.toMap(
                        DealerVo::getCustomerCode, DealerVo::getCustomerName, (a, b) -> a));
      }
    }

    if (!CollectionUtils.isEmpty(terminalCodeSet)) {
      List<TerminalVo> terminalVoList =
          this.terminalVoService.findByTerminalCodes(Lists.newArrayList(terminalCodeSet));
      if (!CollectionUtils.isEmpty(terminalVoList)) {
        mapTerminal =
            terminalVoList.stream()
                .filter(a -> StringUtils.isNoneBlank(a.getTerminalCode(), a.getTerminalName()))
                .collect(
                    Collectors.toMap(
                        TerminalVo::getTerminalCode, TerminalVo::getTerminalName, (a, b) -> a));
      }
    }

    Map<String, String> templateMap = this.findAgreementMap(list);
    Map<String, String> actMap = this.findScanActivityMap(list);

    for (RewardReportVo item : list) {
      item.setType(RebateTypeEnum.findTypeByValue(item.getRebateType()));
      if (StringUtils.isBlank(item.getRelateType())) {
        item.setRelateType(RewardRelateTypeEnum.AGREEMENT.getDictCode());
      }
      item.setPolicyName(Optional.ofNullable(item.getPolicyName()).orElse(StringUtils.EMPTY));
      if (StringUtils.isNotBlank(item.getTemplateCode())) {
        item.setTemplateName(templateMap.getOrDefault(item.getTemplateCode(), StringUtils.EMPTY));
        item.setScanActivityName(item.getTemplateName());
      }
      if (StringUtils.isNoneBlank(item.getRelateCode())
          && RewardRelateTypeEnum.SCAN_ACTIVITY.getDictCode().equals(item.getRelateType())) {
        item.setTemplateName(StringUtils.EMPTY);
        item.setScanActivityName(actMap.getOrDefault(item.getRelateCode(), StringUtils.EMPTY));
      }
      if (!RewardTypeEnum.COST.getDictCode().equals(item.getRewardType())) {
        continue;
      }
      if (ParticipatorTypeEnum.DEALER.getDictCode().equals(item.getParticipatorType())) {
        item.setParticipatorName(mapCustomer.get(item.getParticipatorCode()));
      }
      if (ParticipatorTypeEnum.TERMINAL.getDictCode().equals(item.getParticipatorType())) {
        item.setParticipatorName(mapTerminal.get(item.getParticipatorCode()));
      }
    }
    return list;
  }

  /**
   * 返回扫码活动编码对应的扫码活动名称map
   *
   * @param list
   * @return k-actCode,v-扫码活动名称
   */
  private Map<String, String> findScanActivityMap(List<RewardReportVo> list) {
    if (CollectionUtils.isEmpty(list)) {
      return Maps.newHashMap();
    }
    Set<String> actCodeSet =
        list.stream()
            .filter(
                a ->
                    StringUtils.isNoneBlank(a.getRelateType(), a.getRelateCode())
                        && RewardRelateTypeEnum.SCAN_ACTIVITY
                            .getDictCode()
                            .equals(a.getRelateType()))
            .map(RewardReportVo::getRelateCode)
            .collect(Collectors.toSet());
    if (CollectionUtils.isEmpty(actCodeSet)) {
      return Maps.newHashMap();
    }
    List<ScanActivity> activityList = this.scanActivityService.findDetailByCodes(actCodeSet);
    if (CollectionUtils.isEmpty(activityList)) {
      return Maps.newHashMap();
    }

    return activityList.stream()
        .filter(a -> StringUtils.isNoneBlank(a.getCode(), a.getName()))
        .collect(Collectors.toMap(ScanActivity::getCode, ScanActivity::getName, (a, b) -> a));
  }

  /**
   * 返回协议模板名称
   *
   * @param list
   * @return k-协议模板编码,v-协议名称
   */
  private Map<String, String> findAgreementMap(List<RewardReportVo> list) {
    if (CollectionUtils.isEmpty(list)) {
      return Maps.newHashMap();
    }
    Set<String> templateCodeSet =
        list.stream()
            .filter(a -> StringUtils.isNotBlank(a.getTemplateCode()))
            .map(RewardReportVo::getTemplateCode)
            .collect(Collectors.toSet());
    if (CollectionUtils.isEmpty(templateCodeSet)) {
      return Maps.newHashMap();
    }
    List<ProfitAgreementTemplate> templateList =
        this.profitAgreementTemplateService.findByTemplateCodes(templateCodeSet);
    if (CollectionUtils.isEmpty(templateList)) {
      return Maps.newHashMap();
    }

    return templateList.stream()
        .filter(a -> StringUtils.isNoneBlank(a.getTemplateCode(), a.getTemplateName()))
        .collect(
            Collectors.toMap(
                ProfitAgreementTemplate::getTemplateCode,
                ProfitAgreementTemplate::getTemplateName,
                (a, b) -> a));
  }
}
