package com.biz.crm.cps.mobile.terminal.strategy.internal;

import com.biz.crm.business.common.local.config.DefaultLoginAuthenticationConfig;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.enums.LockStateEnum;
import com.biz.crm.business.common.sdk.model.LoginUserDetailsForCPS;
import com.biz.crm.cps.business.participator.sdk.common.enums.ParticipatorTypeEnum;
import com.biz.crm.cps.mobile.terminal.strategy.LoginUserAuthenticateStrategy;
import com.biz.crm.mdm.business.customer.user.sdk.constant.CustomerUserConstant;
import com.biz.crm.mdm.business.customer.user.sdk.service.CustomerUserVoService;
import com.biz.crm.mdm.business.customer.user.sdk.vo.CustomerUserRelaCustomerVo;
import com.biz.crm.mdm.business.customer.user.sdk.vo.CustomerUserVo;
import com.bizunited.nebula.common.util.Aes128Utils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;
import org.springframework.util.DigestUtils;

import java.nio.charset.StandardCharsets;
import java.util.List;

/**
 * @author songjingen
 */
@Slf4j
@Component
public class CustomerLoginUserAuthenticateStrategy implements LoginUserAuthenticateStrategy {

  @Autowired
  private CustomerUserVoService customerUserVoService;

  @Autowired
  private DefaultLoginAuthenticationConfig defaultLoginAuthenticationConfig;

  @Override
  public String getUserType() {
    return ParticipatorTypeEnum.DEALER.getKey();
  }

  @Override
  public void handleLoginUserAuthenticate(LoginUserDetailsForCPS loginUserDetailsForCPS) {
    /*
     * 处理过程如下：
     * 1、首先通过手机号和用户类型、查询当前用户的存在性
     * 2、然后判断当前用户的状态是否正常
     * 3、再判断当前用户的登录密码是否正确
     */
    // 1、
    CustomerUserVo customerUserVo = new CustomerUserVo();
    this.defaultLoginAuthenticationConfig.defaultAdminAuthentication();
    try {
      customerUserVo = this.customerUserVoService.findDetailsByUserPhone(loginUserDetailsForCPS.getPhone());
    } finally {
      SecurityContextHolder.clearContext();
    }
    Validate.notNull(customerUserVo, "用户不存在或已删除！");
    // 2、
    Validate.isTrue(EnableStatusEnum.ENABLE.getCode().equals(customerUserVo.getEnableStatus()), "当前用户已禁用！");
    Validate.isTrue(LockStateEnum.UNLOCK.getCode().equals(customerUserVo.getLockState()), "当前用户已锁定！");
    // 3、密码校验过程先按与前端约定的加密方式做解密，然后md5再做校验
    String password = Aes128Utils.decrypt(loginUserDetailsForCPS.getPassword(), CustomerUserConstant.ENCRYPT_KEY, Aes128Utils.
        EncodeType.CBC, Aes128Utils.Padding.PKCS_7_PADDING);
    log.error("password decrypt:{}", password);
    String passwordEncryption = DigestUtils.md5DigestAsHex(password.getBytes(StandardCharsets.UTF_8));
    Validate.isTrue(passwordEncryption.equals(customerUserVo.getUserPassword()), "登录密码错误！");
    this.buildLoginUserDetailsForCPS(loginUserDetailsForCPS, customerUserVo);
  }

  /**
   * 构建登录信息
   *
   * @param loginUserDetailsForCPS
   * @param customerUserVo
   */
  private void buildLoginUserDetailsForCPS(LoginUserDetailsForCPS loginUserDetailsForCPS, CustomerUserVo customerUserVo) {
    List<CustomerUserRelaCustomerVo> customerInfoList = customerUserVo.getCustomerInfoList();
    Validate.isTrue(!CollectionUtils.isEmpty(customerInfoList), "经销商用户登录时，未获取到关联的经销商信息！");
    //随机取一个
    CustomerUserRelaCustomerVo customerUserRelaCustomerVo = customerInfoList.get(0);
    loginUserDetailsForCPS.setConsumerCode(customerUserRelaCustomerVo.getCustomerCode());
    loginUserDetailsForCPS.setConsumerName(customerUserRelaCustomerVo.getCustomerName());
    loginUserDetailsForCPS.setAccount(customerUserVo.getUserCode());
    loginUserDetailsForCPS.setUsername(customerUserVo.getFullName());
  }
}
