package com.biz.crm.cps.mobile.terminal.strategy.internal;

import com.biz.crm.business.common.local.config.DefaultLoginAuthenticationConfig;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.enums.LockStateEnum;
import com.biz.crm.business.common.sdk.model.LoginUserDetailsForCPS;
import com.biz.crm.cps.business.participator.sdk.common.enums.ParticipatorTypeEnum;
import com.biz.crm.cps.mobile.terminal.strategy.LoginUserAuthenticateStrategy;
import com.biz.crm.mdm.business.terminal.user.sdk.constant.TerminalUserConstant;
import com.biz.crm.mdm.business.terminal.user.sdk.service.TerminalUserVoService;
import com.biz.crm.mdm.business.terminal.user.sdk.vo.TerminalUserRelaTerminalVo;
import com.biz.crm.mdm.business.terminal.user.sdk.vo.TerminalUserVo;
import com.bizunited.nebula.common.util.Aes128Utils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;
import org.springframework.util.DigestUtils;

import java.nio.charset.StandardCharsets;
import java.util.List;

/**
 * @author songjingen
 */
@Slf4j
@Component
public class TerminalLoginUserAuthenticateStrategy implements LoginUserAuthenticateStrategy {

  @Autowired
  private TerminalUserVoService terminalUserVoService;

  @Autowired
  private DefaultLoginAuthenticationConfig defaultLoginAuthenticationConfig;

  @Override
  public String getUserType() {
    return ParticipatorTypeEnum.TERMINAL.getKey();
  }

  @Override
  public void handleLoginUserAuthenticate(LoginUserDetailsForCPS loginUserDetailsForCPS) {
    /*
     * 处理过程如下：
     * 1、首先通过手机号和用户类型、查询当前用户的存在性
     * 2、然后判断当前用户的状态是否正常
     * 3、再判断当前用户的登录密码是否正确
     */
    // 1、
    TerminalUserVo terminalUserVo = new TerminalUserVo();
    this.defaultLoginAuthenticationConfig.defaultAdminAuthentication();
    try {
      terminalUserVo = this.terminalUserVoService.findByUserPhone(loginUserDetailsForCPS.getPhone());
    } finally {
      SecurityContextHolder.clearContext();
    }
    Validate.notNull(terminalUserVo, "用户不存在或已删除！");
    // 2、
    Validate.isTrue(EnableStatusEnum.ENABLE.getCode().equals(terminalUserVo.getEnableStatus()), "当前用户已禁用！");
    Validate.isTrue(LockStateEnum.UNLOCK.getCode().equals(terminalUserVo.getLockState()), "当前用户已锁定！");
    // 3、密码校验过程先按与前端约定的加密方式做解密，然后md5再做校验
    String password = Aes128Utils.decrypt(loginUserDetailsForCPS.getPassword(), TerminalUserConstant.ENCRYPT_KEY, Aes128Utils.
        EncodeType.CBC, Aes128Utils.Padding.PKCS_7_PADDING);
    log.error("password decrypt:{}", password);
    String passwordEncryption = DigestUtils.md5DigestAsHex(password.getBytes(StandardCharsets.UTF_8));
    Validate.isTrue(passwordEncryption.equals(terminalUserVo.getUserPassword()), "登录密码错误！");
    this.buildLoginUserDetailsForCPS(loginUserDetailsForCPS, terminalUserVo);
  }

  /**
   * 组装登录信息
   *
   * @param loginUserDetailsForCPS
   * @param terminalUserVo
   */
  private void buildLoginUserDetailsForCPS(LoginUserDetailsForCPS loginUserDetailsForCPS, TerminalUserVo terminalUserVo) {
    List<TerminalUserRelaTerminalVo> terminalList = terminalUserVo.getTerminalList();
    Validate.isTrue(!CollectionUtils.isEmpty(terminalList), "经销商用户登录时，未获取到关联的经销商信息！");
    //随机取一个
    TerminalUserRelaTerminalVo terminalUserRelaTerminalVo = terminalList.get(0);
    loginUserDetailsForCPS.setConsumerCode(terminalUserRelaTerminalVo.getTerminalCode());
    loginUserDetailsForCPS.setConsumerName(terminalUserRelaTerminalVo.getTerminalName());
    loginUserDetailsForCPS.setAccount(terminalUserVo.getUserCode());
    loginUserDetailsForCPS.setUsername(terminalUserVo.getFullName());
  }
}
