package com.biz.crm.cps.mobile.terminal.strategy.internal;

import com.biz.crm.business.common.local.config.DefaultLoginAuthenticationConfig;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.enums.LockStateEnum;
import com.biz.crm.business.common.sdk.model.LoginUserDetailsForCPS;
import com.biz.crm.cps.business.participator.sdk.common.enums.ParticipatorTypeEnum;
import com.biz.crm.cps.mobile.terminal.strategy.LoginUserAuthenticateStrategy;
import com.biz.crm.mdm.business.user.sdk.constant.UserConstant;
import com.biz.crm.mdm.business.user.sdk.service.UserVoService;
import com.biz.crm.mdm.business.user.sdk.vo.UserVo;
import com.bizunited.nebula.common.util.Aes128Utils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.stereotype.Component;
import org.springframework.util.DigestUtils;

import java.nio.charset.StandardCharsets;

/**
 * @author songjingen
 */
@Slf4j
@Component
public class UserGuideLoginUserAuthenticateStrategy implements LoginUserAuthenticateStrategy {

  @Autowired
  private UserVoService userVoService;

  @Autowired
  private DefaultLoginAuthenticationConfig defaultLoginAuthenticationConfig;

  @Override
  public String getUserType() {
    return ParticipatorTypeEnum.USER_GUIDE.getKey();
  }

  @Override
  public void handleLoginUserAuthenticate(LoginUserDetailsForCPS loginUserDetailsForCPS) {
    /*
     * 处理过程如下：
     * 1、首先通过手机号和用户类型、查询当前用户的存在性
     * 2、然后判断当前用户的状态是否正常
     * 3、再判断当前用户的登录密码是否正确
     */
    // 1、
    UserVo userVo = new UserVo();
    this.defaultLoginAuthenticationConfig.defaultAdminAuthentication();
    try {
      userVo = this.userVoService.findDetailsByPhone(loginUserDetailsForCPS.getPhone());
    } finally {
      SecurityContextHolder.clearContext();
    }
    Validate.notNull(userVo, "用户不存在或已删除！");
    // 2、
    Validate.isTrue(EnableStatusEnum.ENABLE.getCode().equals(userVo.getEnableStatus()), "当前用户已禁用！");
    Validate.isTrue(LockStateEnum.UNLOCK.getCode().equals(userVo.getLockState()), "当前用户已锁定！");
    Validate.isTrue(ParticipatorTypeEnum.USER_GUIDE.getKey().equals(userVo.getUserType()), "当前用户不是导购用户！");
    // 3、密码校验过程先按与前端约定的加密方式做解密，然后md5再做校验
    String password = Aes128Utils.decrypt(loginUserDetailsForCPS.getPassword(), UserConstant.ENCRYPT_KEY, Aes128Utils.
        EncodeType.CBC, Aes128Utils.Padding.PKCS_7_PADDING);
    log.error("password decrypt:{}", password);
    String passwordEncryption = DigestUtils.md5DigestAsHex(password.getBytes(StandardCharsets.UTF_8));
    Validate.isTrue(passwordEncryption.equals(userVo.getUserPassword()), "登录密码错误！");
    this.buildLoginUserDetailsForCPS(loginUserDetailsForCPS, userVo);
  }

  /**
   * 组装登录信息
   *
   * @param loginUserDetailsForCPS
   * @param userVo
   */
  private void buildLoginUserDetailsForCPS(LoginUserDetailsForCPS loginUserDetailsForCPS, UserVo userVo) {
    loginUserDetailsForCPS.setConsumerCode(userVo.getUserName());
    loginUserDetailsForCPS.setConsumerName(userVo.getFullName());
    loginUserDetailsForCPS.setUsername(userVo.getFullName());
    loginUserDetailsForCPS.setAccount(userVo.getUserCode());
  }
}
