package com.biz.crm.cps.business.activity.actual.local.service.internal;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.service.GenerateCodeService;
import com.biz.crm.business.common.sdk.service.LoginUserService;
import com.biz.crm.cps.business.activity.actual.local.entity.RewardActivityEntity;
import com.biz.crm.cps.business.activity.actual.local.entity.RewardActivityPrizeEntity;
import com.biz.crm.cps.business.activity.actual.local.entity.RewardActivityProductEntity;
import com.biz.crm.cps.business.activity.actual.local.repository.RewardActivityActualRepository;
import com.biz.crm.cps.business.activity.actual.local.service.RewardActivityActualService;
import com.biz.crm.cps.business.activity.actual.local.service.RewardActivityPrizeService;
import com.biz.crm.cps.business.activity.actual.local.service.RewardActivityProductService;
import com.biz.crm.cps.business.activity.actual.sdk.dto.RewardActivityDto;
import com.biz.crm.cps.business.common.sdk.enums.EnableStatusEnum;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.apache.commons.lang3.time.DateFormatUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Set;
import java.util.UUID;
import java.util.stream.Collectors;

/**
 * 活动奖励配置接口实现
 * @Author: zengxingwang
 * @Date: 2021/10/19 16:26
 */
@Service
public class RewardActivityActualServiceImpl implements RewardActivityActualService {

  @Autowired
  private RewardActivityActualRepository rewardActivityRepository;
  @Autowired
  private RewardActivityProductService rewardActivityProductService;
  @Autowired
  private RewardActivityPrizeService rewardActivityPrizeService;
  @Autowired
  private GenerateCodeService generateCodeService;
  @Autowired
  private LoginUserService loginUserService;
  /**
   * 创建活动奖励配置
   * @param rewardActivity
   * @return
   */
  @Override
  @Transactional
  public RewardActivityEntity create(RewardActivityEntity rewardActivity) {
    this.createValidate(rewardActivity);
    Set<RewardActivityProductEntity> productEntities = rewardActivity.getProductEntities();
    Set<RewardActivityPrizeEntity> prize = rewardActivity.getPrize();
    //生成编码
//    if (Objects.nonNull(this.generateCodeService)) {
//      rewardActivity.setActivityCode(this.generateCodeService.generateCode(RewardActivityCodeConstant.REWARD_ACTIVITY_CODE, 1).get(0));
//    }
    rewardActivity.setActivityCode(UUID.randomUUID().toString());
    rewardActivity.setTenantCode(TenantUtils.getTenantCode());
    rewardActivity.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
    rewardActivity.setDelFlag(EnableStatusEnum.ENABLE.getCode());
    String loginAccountName = loginUserService.getLoginAccountName();
    Date date = new Date();
    rewardActivity.setCreateAccount(loginAccountName);
    rewardActivity.setModifyAccount(loginAccountName);
    rewardActivity.setCreateTime(date);
    rewardActivity.setModifyTime(date);
    rewardActivityRepository.save(rewardActivity);
    //参与产品
    productEntities.stream().forEach(item -> item.setRewardActivityId(rewardActivity.getId()));
    rewardActivityProductService.createBatch(productEntities);
    //奖励信息
    prize.stream().forEach(item -> item.setRewardActivityId(rewardActivity.getId()));
    rewardActivityPrizeService.createBatch(prize);
    return rewardActivity;
  }

  /**
   * 条件查询活动奖励配置列表
   * @param dto
   * @param pageable
   * @return
   */
  @Override
  public Page<RewardActivityEntity> findByConditions(RewardActivityDto dto, Pageable pageable) {
    ObjectUtils.defaultIfNull(pageable, PageRequest.of(0, 50));
    dto.setTenantCode(TenantUtils.getTenantCode());
    try {
      //格式化时间
      SimpleDateFormat simpleDateFormat = new SimpleDateFormat("yyyy-MM-dd");
      String startCreateTime = dto.getStartCreateTime();
      if (StringUtils.isNotBlank(startCreateTime)) {
        dto.setStartCreateTime(DateFormatUtils.format(simpleDateFormat.parse(startCreateTime), "yyyy-MM-dd 00:00:00"));
      }
      String endCreateTime = dto.getEndCreateTime();
      if (StringUtils.isNotBlank(endCreateTime)) {
        dto.setEndCreateTime(DateFormatUtils.format(simpleDateFormat.parse(endCreateTime), "yyyy-MM-dd 23:59:59"));
      }
    } catch (ParseException e) {
      e.printStackTrace();
    }
    return rewardActivityRepository.findByConditions(dto, pageable);
  }

  /**
   * 编辑活动奖励配置
   * @param rewardActivity
   * @return
   */
  @Override
  @Transactional
  public RewardActivityEntity update(RewardActivityEntity rewardActivity) {
    this.updateValidate(rewardActivity);
    RewardActivityEntity old = rewardActivityRepository.findById(rewardActivity.getId());
    Validate.notNull(old, "未找到对应数据，请检查");
    old.setActivityName(rewardActivity.getActivityName());
    old.setStartTime(rewardActivity.getStartTime());
    old.setEndTime(rewardActivity.getEndTime());
    old.setActivityDescription(rewardActivity.getActivityDescription());
    old.setModifyTime(new Date());
    old.setModifyAccount(loginUserService.getLoginAccountName());
    rewardActivityRepository.updateById(old);
    //修改奖项信息
    Set<RewardActivityPrizeEntity> prize = rewardActivity.getPrize();
    rewardActivityPrizeService.updateBatch(prize);
    return null;
  }

  /**
   * 批量根据id禁用
   * @param ids
   */
  @Override
  @Transactional
  public void disable(List<String> ids) {
    Validate.notEmpty(ids, "禁用时，id不能为空");
    this.rewardActivityRepository.updateEnableStatusByIds(EnableStatusEnum.DISABLE, ids);
  }

  /**
   * 批量根据id启用
   * @param ids
   */
  @Override
  @Transactional
  public void enable(List<String> ids) {
    Validate.notEmpty(ids, "启用时，id不能为空");
    //验证是否可启用
    for(String id : ids){
      RewardActivityEntity rewardActivity = this.findDetailById(id);
      Set<RewardActivityProductEntity> productEntities = rewardActivity.getProductEntities();
      List<String> productCods = productEntities.stream().map(RewardActivityProductEntity::getMaterialCode).collect(Collectors.toList());
      long count = rewardActivityRepository.findByProductsAndDateAndId(productCods, rewardActivity.getStartTime(), rewardActivity.getEndTime(), rewardActivity.getId());
      Validate.isTrue(count == 0, rewardActivity.getActivityName() + "活动重复");
    }
    this.rewardActivityRepository.updateEnableStatusByIds(EnableStatusEnum.ENABLE, ids);
  }

  /**
   * 批量根据id查询详情（包括关联信息）
   * @param id
   * @return
   */
  @Override
  public RewardActivityEntity findDetailById(String id) {
    if(StringUtils.isBlank(id)){
      return null;
    }
    RewardActivityEntity activityEntity = rewardActivityRepository.findById(id);
    if(activityEntity == null){
      return null;
    }
    Set<RewardActivityProductEntity> productEntities = rewardActivityProductService.findByRewardActivityId(id);
    if(!CollectionUtils.isEmpty(productEntities)){
      activityEntity.setProductEntities(productEntities);
    }
    LinkedHashSet<RewardActivityPrizeEntity> prize = rewardActivityPrizeService.findDetailByRewardActivityId(id);
    if(!CollectionUtils.isEmpty(prize)){
      activityEntity.setPrize(prize);
    }
    return activityEntity;
  }

  /**
   * 获取当前时间的活动（包括关联信息）
   * @return
   */
  @Override
  public RewardActivityEntity findDetailByTime(String productCode) {
    RewardActivityEntity activityEntity = rewardActivityRepository.findByTime(productCode, new Date());
    if(activityEntity == null){
      return null;
    }
    Set<RewardActivityProductEntity> productEntities = rewardActivityProductService.findByRewardActivityId(activityEntity.getId());
    if(!CollectionUtils.isEmpty(productEntities)){
      activityEntity.setProductEntities(productEntities);
    }
    Set<RewardActivityPrizeEntity> prize = rewardActivityPrizeService.findDetailByRewardActivityId(activityEntity.getId());
    if(!CollectionUtils.isEmpty(prize)){
      activityEntity.setPrize(prize);
    }
    return activityEntity;
  }

  /**
   * 批量根据编码查询详情
   * @param activityCode
   * @return
   */
  @Override
  public RewardActivityEntity findByActivityCode(String activityCode) {
    if(StringUtils.isBlank(activityCode)){
      return null;
    }
    return rewardActivityRepository.findByActivityCode(activityCode);
  }

  /**
   * 数据验证
   * @param rewardActivity
   */
  private void updateValidate(RewardActivityEntity rewardActivity) {
    Validate.notNull(rewardActivity, "编辑奖励配置时，数据不能为空");
    Validate.notBlank(rewardActivity.getId(), "编辑奖励配置时，主键不能为空");
    Validate.notBlank(rewardActivity.getActivityName(), "编辑奖励配置时，活动名称不能为空");
    Validate.notNull(rewardActivity.getStartTime(), "编辑奖励配置时，活动开始时间不能为空");
    Validate.notNull(rewardActivity.getEndTime(), "编辑奖励配置时，活动结束时间不能为空");
    Validate.notBlank(rewardActivity.getProductType(), "编辑奖励配置时，产品维度不能为空");
    Set<RewardActivityProductEntity> productEntities = rewardActivity.getProductEntities();
    List<String> productCods = productEntities.stream().map(RewardActivityProductEntity::getMaterialCode).collect(Collectors.toList());
    long count = rewardActivityRepository.findByProductsAndDateAndId(productCods, rewardActivity.getStartTime(), rewardActivity.getEndTime(), rewardActivity.getId());
    Validate.isTrue(count == 0, "活动重复");
  }

  /**
   * 数据验证
   * @param rewardActivity
   */
  private void createValidate(RewardActivityEntity rewardActivity) {
    Validate.notNull(rewardActivity, "创建奖励配置时，数据不能为空");
    Validate.notEmpty(rewardActivity.getProductEntities(), "创建奖励配置时，参与产品数据不能为空");
    Validate.notBlank(rewardActivity.getActivityName(), "创建奖励配置时，活动名称不能为空");
    Validate.notNull(rewardActivity.getStartTime(), "创建奖励配置时，活动开始时间不能为空");
    Validate.notNull(rewardActivity.getEndTime(), "创建奖励配置时，活动结束时间不能为空");
    Validate.notBlank(rewardActivity.getProductType(), "创建奖励配置时，产品维度不能为空");
    Set<RewardActivityProductEntity> productEntities = rewardActivity.getProductEntities();
    List<String> productCods = productEntities.stream().map(RewardActivityProductEntity::getMaterialCode).collect(Collectors.toList());
    long count = rewardActivityRepository.findByProductsAndDate(productCods, rewardActivity.getStartTime(), rewardActivity.getEndTime());
    Validate.isTrue(count == 0, "活动重复");
  }
}
