package com.biz.crm.cps.business.activity.actual.local.service.internal;

import com.biz.crm.cps.business.activity.actual.local.entity.RewardActivityEntity;
import com.biz.crm.cps.business.activity.actual.local.entity.RewardActivityPrizeEntity;
import com.biz.crm.cps.business.activity.actual.local.entity.RewardActivityPrizeProductEntity;
import com.biz.crm.cps.business.activity.actual.local.repository.RewardActivityPrizeRepository;
import com.biz.crm.cps.business.activity.actual.local.service.RewardActivityActualService;
import com.biz.crm.cps.business.activity.actual.local.service.RewardActivityPrizeProductService;
import com.biz.crm.cps.business.activity.actual.local.service.RewardActivityPrizeService;
import com.google.common.collect.Sets;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.Set;

/**
 * 奖励等级信息接口实现
 * @Author: zengxingwang
 * @Date: 2021/10/21 9:21
 */
@Service
public class RewardActivityPrizeServiceImpl implements RewardActivityPrizeService {

  @Autowired
  private RewardActivityPrizeRepository rewardActivityPrizeRepository;
  @Autowired
  private RewardActivityPrizeProductService rewardActivityPrizeProductService;
  @Autowired
  private RewardActivityActualService rewardActivityActualService;

  /**
   * 批量设置奖项信息
   * @param prize
   */
  @Override
  @Transactional
  public void createBatch(Set<RewardActivityPrizeEntity> prize) {
    this.createValidate(prize);
    rewardActivityPrizeRepository.saveBatch(prize);
    //奖品数据
    Set<RewardActivityPrizeProductEntity> set = new HashSet<>();
    for(RewardActivityPrizeEntity item : prize){
      Set<RewardActivityPrizeProductEntity> prizeProductEntities = item.getPrizeProductEntities();
      prizeProductEntities.stream().forEach(prizeProduct -> prizeProduct.setRewardActivityPrizeId(item.getId()));
      set.addAll(prizeProductEntities);
    }
    rewardActivityPrizeProductService.createBatch(set);
  }

  /**
   * 修改奖项信息
   * @param prize
   */
  @Override
  @Transactional
  public void updateBatch(Set<RewardActivityPrizeEntity> prize) {
    this.updateValidate(prize);
    rewardActivityPrizeRepository.updateBatchById(prize);
  }

  /**
   * 根据配置活动查询参与奖项信息（包括奖品）
   * @param id
   * @return
   */
  @Override
  public LinkedHashSet<RewardActivityPrizeEntity> findDetailByRewardActivityId(String id) {
    if(StringUtils.isBlank(id)){
      return Sets.newLinkedHashSet();
    }
    LinkedHashSet<RewardActivityPrizeEntity> prizeEntities = rewardActivityPrizeRepository.findDetailByRewardActivityId(id);
    if(CollectionUtils.isEmpty(prizeEntities)){
      return Sets.newLinkedHashSet();
    }
    return prizeEntities;
  }

  /**
   * 根据配置活动查询参与奖项信息（包括奖品）
   * @param activityCode
   * @return
   */
  @Override
  public Set<RewardActivityPrizeEntity> findDetailByActivityCode(String activityCode) {
    if(StringUtils.isBlank(activityCode)){
      return Sets.newHashSet();
    }
    RewardActivityEntity activityEntity = rewardActivityActualService.findByActivityCode(activityCode);
    if(activityEntity == null){
      return Sets.newHashSet();
    }
    LinkedHashSet<RewardActivityPrizeEntity> prizeEntities = rewardActivityPrizeRepository.findDetailByRewardActivityId(activityEntity.getId());
    return prizeEntities;
  }

  /**
   * 根据活动编码+奖项查询
   * @param activityCode
   * @param amount
   * @return
   */
  @Override
  public RewardActivityPrizeEntity findByActivityCodeAndPrizeLevel(String activityCode, Integer amount) {
    if(StringUtils.isBlank(activityCode) || amount == null){
      return null;
    }
    return rewardActivityPrizeRepository.findByActivityCodeAndPrizeLevel(activityCode, amount);
  }

  /**
   * 修改库存
   * @param id
   */
  @Override
  @Transactional
  public void updateInventoryById(String id) {
    Validate.notBlank(id, "修改库存时，主键不能为空");
    rewardActivityPrizeRepository.updateInventoryById(id);
  }

  /**
   * 验证数据
   * @param prize
   */
  private void updateValidate(Set<RewardActivityPrizeEntity> prize) {
    Validate.notEmpty(prize, "奖励等级信息不能为空");
    Integer grids = 0;
    BigDecimal probability = BigDecimal.ZERO;
    for(RewardActivityPrizeEntity item : prize){
      Validate.notNull(item.getId(), "奖励等级主键不能为空");
      if (item.getPrizeLevel() != BigDecimal.ONE){
        Validate.notBlank(item.getUrl(), "图片不能为空");
      }
      Validate.notNull(item.getPrizeLevel(), "奖励等级不能为空");
      Validate.notNull(item.getProbability(), "中奖概率不能为空");
      Validate.notNull(item.getGrids(), "格子数不能为空");
      probability = probability.add(item.getProbability());
      grids = grids + item.getGrids();
    }
    Validate.isTrue(grids <= 12, "格子总数不能超过12");
    Validate.isTrue(probability.compareTo(new BigDecimal(100)) == 0, "概率不能超过100%");
  }

  /**
   * 验证数据
   * @param prize
   */
  private void createValidate(Set<RewardActivityPrizeEntity> prize) {
    Validate.notEmpty(prize, "奖励等级信息不能为空");
    Integer grids = 0;
    BigDecimal probability = BigDecimal.ZERO;
    for(RewardActivityPrizeEntity item : prize) {
      if (item.getPrizeLevel() != BigDecimal.ONE){
        Validate.notBlank(item.getUrl(), "图片不能为空");
      }
      Validate.notNull(item.getPrizeLevel(), "奖励等级不能为空");
      Validate.notNull(item.getProbability(), "中奖概率不能为空");
      Validate.notNull(item.getGrids(), "格子数不能为空");
      probability = probability.add(item.getProbability());
      grids = grids + item.getGrids();
    }
    Validate.isTrue(grids <= 12, "格子总数不能超过12");
    Validate.isTrue(probability.compareTo(new BigDecimal(100)) == 0, "概率不能超过100%");
  }
}
