package com.biz.crm.cps.business.activity.scan.local.service.internal;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.cps.business.activity.scan.local.entity.ScanActivity;
import com.biz.crm.cps.business.activity.scan.local.entity.ScanActivityUserScopeItem;
import com.biz.crm.cps.business.activity.scan.local.repository.ScanActivityListVoRepository;
import com.biz.crm.cps.business.activity.scan.local.repository.ScanActivityRepository;
import com.biz.crm.cps.business.activity.scan.local.repository.ScanActivityUserScopeItemRepository;
import com.biz.crm.cps.business.activity.scan.sdk.dto.ParticipateActivityDto;
import com.biz.crm.cps.business.activity.scan.sdk.dto.ScanActivityListPaginationDto;
import com.biz.crm.cps.business.activity.scan.sdk.enums.ScanActivityParticipateTypeEnum;
import com.biz.crm.cps.business.activity.scan.sdk.enums.ScanActivityScopeEnum;
import com.biz.crm.cps.business.activity.scan.sdk.enums.ScanParticipateStatusEnum;
import com.biz.crm.cps.business.activity.scan.sdk.service.ScanActivityListVoService;
import com.biz.crm.cps.business.activity.scan.sdk.vo.ScanActivityListVo;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

/**
 * 扫码活动前端分页列表接口实现
 *
 * @author sunx
 * @date 2022/3/28
 */
@Service
@Slf4j
public class ScanActivityListVoServiceImpl implements ScanActivityListVoService {

  @Autowired private ScanActivityListVoRepository scanActivityListVoRepository;

  @Autowired private ScanActivityUserScopeItemRepository scanActivityUserScopeItemRepository;

  @Autowired private ScanActivityRepository scanActivityRepository;

  @Override
  public Page<ScanActivityListVo> findByConditions(
      Pageable pageable, ScanActivityListPaginationDto paginationDto) {
    pageable = Optional.ofNullable(pageable).orElse(PageRequest.of(0, 50));
    paginationDto = Optional.ofNullable(paginationDto).orElse(new ScanActivityListPaginationDto());
    Validate.notBlank(paginationDto.getParticipateCode(), "参与者编码不能为空");
    Validate.notBlank(paginationDto.getParticipateObj(), "参与对象不能为空");
    Page<ScanActivityListVo> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
    Page<ScanActivityListVo> pageResult = null;

    if (StringUtils.isBlank(paginationDto.getParticipateStatus())) {
      pageResult = this.scanActivityListVoRepository.findListByConditions(page, paginationDto);
    }
    if (ScanParticipateStatusEnum.DEFAULT
        .getDictCode()
        .equals(paginationDto.getParticipateStatus())) {
      pageResult =
          this.scanActivityListVoRepository.findDefaultListByConditions(page, paginationDto);
    }
    if (ScanParticipateStatusEnum.PARTICIPATE
        .getDictCode()
        .equals(paginationDto.getParticipateStatus())) {
      pageResult =
          this.scanActivityListVoRepository.findParticipateListByConditions(page, paginationDto);
    }
    if (ScanParticipateStatusEnum.OVERDUE
        .getDictCode()
        .equals(paginationDto.getParticipateStatus())) {
      pageResult =
          this.scanActivityListVoRepository.findOverdueListByConditions(page, paginationDto);
    }

    if (Objects.isNull(pageResult) || CollectionUtils.isEmpty(pageResult.getRecords())) {
      return pageResult;
    }
    Set<String> actCodeSet =
        pageResult.getRecords().stream()
            .map(ScanActivityListVo::getCode)
            .collect(Collectors.toSet());
    // 获取参与情况
    List<ScanActivityUserScopeItem> list =
        this.scanActivityUserScopeItemRepository
            .findByParticipateObjAndUserCodeAndJoinFlagAndActCodeIn(
                paginationDto.getParticipateObj(),
                paginationDto.getParticipateCode(),
                Boolean.TRUE,
                actCodeSet);
    Map<String, ScanActivityUserScopeItem> map = Maps.newHashMap();
    if (!CollectionUtils.isEmpty(list)) {
      map =
          list.stream()
              .collect(
                  Collectors.toMap(
                      ScanActivityUserScopeItem::getActCode, Function.identity(), (a, b) -> b));
    }
    for (ScanActivityListVo item : pageResult.getRecords()) {
      ScanActivityUserScopeItem scopeItem = map.get(item.getCode());
      this.buildScanActivityListVo(scopeItem, item);
    }
    return pageResult;
  }

  @Override
  @Transactional
  public void participateActivity(ParticipateActivityDto dto) {
    Validate.notNull(dto, "参与请求参数不能为空");
    Validate.notBlank(dto.getCode(), "参与活动编码不能为空");
    Validate.notNull(dto.getParticipateCode(), "参与对象编码不能为空");
    Validate.notNull(dto.getParticipateObj(), "参与对象类型参数不能为空");
    List<ScanActivity> list =
        this.scanActivityRepository.findDetailsByCodes(Sets.newHashSet(dto.getCode()));
    Validate.isTrue(!CollectionUtils.isEmpty(list), "未获取到对应活动信息");
    ScanActivity scanActivity = list.get(0);
    Validate.isTrue(
        dto.getParticipateObj().equals(scanActivity.getParticipateObj()), "活动参与对象类型于请求参数不匹配");
    if (scanActivity.getParticipateScope().equals(ScanActivityScopeEnum.ALL.getDictCode())
        && ScanActivityParticipateTypeEnum.AUTOMATIC
            .getDictCode()
            .equals(scanActivity.getParticipateType())) {
      Validate.isTrue(false, "您已自动参与该活动，请勿执行该操作！");
    }

    Date now = new Date();
    if (ScanActivityParticipateTypeEnum.MANUAL
        .getDictCode()
        .equals(scanActivity.getParticipateType())) {
      Validate.isTrue(
          scanActivity.getParticipateStartTime().compareTo(now) <= 0, "尚未到参与开始时间，不能参与该活动!");
      Validate.isTrue(scanActivity.getParticipateEndTime().compareTo(now) > 0, "该活动已过期，不能参与该活动!");
    }

    List<ScanActivityUserScopeItem> scopeItemList =
        this.scanActivityUserScopeItemRepository
            .findByParticipateObjAndUserCodeAndJoinFlagAndActCodeIn(
                dto.getParticipateObj(),
                dto.getParticipateCode(),
                null,
                Sets.newHashSet(dto.getCode()));
    if (CollectionUtils.isEmpty(scopeItemList)) {
      Validate.isTrue(
          scanActivity.getParticipateScope().equals(ScanActivityScopeEnum.ALL.getDictCode()),
          "当前操作人不能参与该活动!");
      // 执行参与
      ScanActivityUserScopeItem scopeItem = new ScanActivityUserScopeItem();
      scopeItem.setCreateTime(now);
      scopeItem.setJoinFlag(Boolean.TRUE);
      scopeItem.setParticipateObj(dto.getParticipateObj());
      scopeItem.setUserCode(dto.getParticipateCode());
      scopeItem.setActCode(dto.getCode());
      this.scanActivityUserScopeItemRepository.save(scopeItem);
    } else {
      ScanActivityUserScopeItem scopeItem = scopeItemList.get(0);
      Validate.isTrue(!scopeItem.getJoinFlag(), "您已参与该活动，请勿重复执行该操作！");
      scopeItem.setCreateTime(now);
      scopeItem.setJoinFlag(Boolean.TRUE);
      this.scanActivityUserScopeItemRepository.saveOrUpdate(scopeItem);
    }
  }

  @Override
  public String findParticipateNumsByCode(String code) {
    if (StringUtils.isBlank(code)) {
      return StringUtils.EMPTY;
    }
    List<ScanActivity> list = this.scanActivityRepository.findDetailsByCodes(Sets.newHashSet(code));
    Validate.isTrue(!CollectionUtils.isEmpty(list), "未获取到对应活动信息");
    ScanActivity scanActivity = list.get(0);
    if (ScanActivityParticipateTypeEnum.AUTOMATIC
            .getDictCode()
            .equals(scanActivity.getParticipateType())
        && ScanActivityScopeEnum.ALL.getDictCode().equals(scanActivity.getParticipateScope())) {
      return "全部";
    }
    List<ScanActivityUserScopeItem> scopeItemList =
        this.scanActivityUserScopeItemRepository
            .findByParticipateObjAndUserCodeAndJoinFlagAndActCodeIn(
                scanActivity.getParticipateObj(), null, Boolean.TRUE, Sets.newHashSet(code));
    if (CollectionUtils.isEmpty(scopeItemList)) {
      return "0";
    }
    return String.valueOf(scopeItemList.size());
  }

  /**
   * 设置返回对象的参与时间信息，及活动参与状态信息
   *
   * @param scopeItem
   * @param item
   */
  private void buildScanActivityListVo(
      ScanActivityUserScopeItem scopeItem, ScanActivityListVo item) {
    if (Objects.isNull(item)) {
      return;
    }
    if (ScanActivityParticipateTypeEnum.AUTOMATIC.getDictCode().equals(item.getParticipateType())
        && ScanActivityScopeEnum.ALL.getDictCode().equals(item.getParticipateScope())) {
      item.setParticipateTime(item.getCreateTime());
      item.setParticipateStatus(ScanParticipateStatusEnum.PARTICIPATE.getDictCode());
      return;
    }
    if (Objects.nonNull(scopeItem)) {
      item.setParticipateTime(scopeItem.getCreateTime());
      item.setParticipateStatus(ScanParticipateStatusEnum.PARTICIPATE.getDictCode());
      return;
    }
    item.setParticipateStatus(ScanParticipateStatusEnum.DEFAULT.getDictCode());
    if (item.getParticipateEndTime().compareTo(new Date()) < 0) {
      item.setParticipateStatus(ScanParticipateStatusEnum.OVERDUE.getDictCode());
    }
  }
}
